/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test sca_vector

#include <systemc-ams>
#include "test_utilities.h"

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_util::sca_vector<double> v;

  // check empty

  std::cout << "Length: " << v.length()
            << " - resizable: " << std::boolalpha << v.is_auto_resizable()
            << std::endl;

  v.resize( 17 );

  std::cout << "Length: " << v.length()
            << " - resizable: " << std::boolalpha << v.is_auto_resizable()
            << std::endl;

  // check initialisation

  bool check = true;
  for (unsigned i = 0; i < v.length(); ++i)
    check = check && ( v(i) == 0. );

  std::cout << "Init: " << check << std::endl;

  // check access

  sca_util::sca_vector<double> const & v_const = v; // enforce constness

  v(1) = 42;
  std::cout << "v(1) == " << v(1) << " == " << v_const(1) << std::endl;

  // out-of-bounds access
  double const & foo = v_const(21);

  std::cout << "foo == " << foo  << " == v(21) == " << v_const(21) << std::endl;

  v(21) = 17;
  std::cout << "foo == " << foo << " == v(21) == " << v_const(21) << std::endl;

  // turn off auto resize
  v.unset_auto_resizable();

  std::cout << "Length: " << v.length()
            << " - resizable: " << std::boolalpha << v.is_auto_resizable()
            << std::endl;

#if TEST_CPLUSPLUS >= 201103L

  sca_util::sca_vector<int> vi = {1, 2, 3, 4, 5};

  if (vi.length() != 5)
  {
    SC_REPORT_ERROR("sca_vector", "C++11 Initializer list test failed - wrong size");
  }

  for (unsigned long i = 0; i < 5; i++)
  {
    if (vi(i) != int(i+1))
    {
      SC_REPORT_ERROR("sca_vector", "C++11 Initializer list test failed - wrong value");
    }
  }

  sca_util::sca_vector<double> vid = {1.0, 2.0, 3.0, 4.0, 5.0};

  if (vid.length() != 5)
  {
    SC_REPORT_ERROR("sca_vector", "C++11 Initializer list test failed - wrong size");
  }

  for (unsigned long i = 0; i < 5; i++)
  {
    if (int(vid(i)) != int(i+1))
    {
      SC_REPORT_ERROR("sca_vector", "C++11 Initializer list test failed - wrong value");
    }
  }

  sca_util::sca_vector<sca_util::sca_complex> vic = { {1.0,-2.0}, {2.0,-3.0}, {3.0,-4.0}, {4.0,-5.0}, {5.0,-6.0} };

  if (vic.length() != 5)
  {
    SC_REPORT_ERROR("sca_vector", "C++11 Initializer list test failed - wrong size");
  }

  for (unsigned long i = 0; i < 5; i++)
  {
    if ( (int(vic(i).real()) != int(i+1)) || (int(vic(i).imag()) != -int(i+2)) )
    {
      SC_REPORT_ERROR("sca_vector", "C++11 Initializer list test failed - wrong value");
    }
  }

#endif

#ifdef IEEE_16661_SYSTEMC_AMS

  std::vector<int> vs;
  vs.push_back(1);
  vs.push_back(2);
  vs.push_back(3);
  vs.push_back(4);
  vs.push_back(5);

  sca_util::sca_vector<int> vis = vs;

  if (vis.length() != 5)
  {
    SC_REPORT_ERROR("sca_vector", "cast from std::vector- wrong size");
  }

  for (unsigned long i = 0; i < 5; i++)
  {
    if (vis(i) != int(i+1))
    {
      SC_REPORT_ERROR("sca_vector", "cast from std::vector - wrong value");
    }
  }

  std::vector<int> v2;

  v2 = vis;

  if (v2.size() != 5)
  {
    SC_REPORT_ERROR("sca_vector", "std::vector cast from sca_vector - wrong size");
  }

  for (unsigned long i = 0; i < 5; i++)
  {
    if (v2[i] != int(i+1))
    {
      SC_REPORT_ERROR("sca_vector", "std::vector cast from sca_vector - wrong value");
    }
  }

  std::vector<int> v3(vis);

  if (v3.size() != 5)
  {
    SC_REPORT_ERROR("sca_vector", "std::vector cast from sca_vector - wrong size");
  }

  for (unsigned long i = 0; i < 5; i++)
  {
    if (v3[i] != int(i+1))
    {
      SC_REPORT_ERROR("sca_vector", "std::vector cast from sca_vector - wrong value");
    }
  }

#endif

  // out-of-bounds access
  std::cout << "test error reporting for out-of-bounds access..." << std::endl;
  try
  {
    v_const(41);
  }
  catch ( const sc_core::sc_report & r )
  {
    std::cout << "expected error occurred" << std::endl;
  }

  TEST_LABEL_END;

  return 0;
}
