/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test sca_tdf::sca_ltf_nd

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE( my_ltf )
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  my_ltf( sc_core::sc_module_name nm ) : in("in"), out("out")
  {
    ltf_nd = new sca_tdf::sca_ltf_nd("ltf_nd");
  }

  ~my_ltf()
  {
    delete ltf_nd;
  }
  
  void initialize()
  {
    num(0) = -1.0;  //               2
    num(2) = 1.0;   //              s  - 1
    den(0) = 4.0;   //  H(s)=  ----------------
    den(1) = 6.0;   //          3     2
    den(2) = 4.0;   //         s  + 4s  + 6s + 4
    den(3) = 1.0;
  }
 
  void processing()
  {
    double tmp = ltf_nd->calculate( num, den, in.read() );
    out.write(tmp);
  }

 private:
  sca_tdf::sca_ltf_nd* ltf_nd; // Laplace transfer function
  sca_util::sca_vector<double> num, den; // numerator and denominator coefficients
};

SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<double> out; // output port

  SCA_CTOR(my_step) : out("out")
  {}

  void set_attributes()
  {
    out.set_timestep(0.1, sc_core::SC_SEC);
    out.set_delay(1);
  }

  void processing()
  {
    out.write(1.0);
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;

  my_step step("step");
  my_ltf  ltf1("ltf1");

  step.out(sig1);
  ltf1.in(sig1);
  ltf1.out(sig2);

  // tracing
  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test03");
  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");

  sc_core::sc_start(7, sc_core::SC_SEC);
  
  sca_util::sca_close_tabular_trace_file(tf);
  
  test_util::check_results("test03", 2);

  TEST_LABEL_END;
    
  return 0;
}
