/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/
//  test04.cpp -- cluster tests

// In this example
// mod  : t_mod = t_in/out * rate
//
// mod_a: 2.0 =  1.0 * 2
// mod_b: 1.0 =  1.0 * 1
// mod_c: 1.0 =  1.0 * 1
// mod_d (systemc module)
//
// results in cluster execution schedule { A B C B C }

#include <systemc-ams>
#include "test_utilities.h"

//#define TDF_DEBUG

SCA_TDF_MODULE( mod_a )
{
  sca_tdf::sca_out<double> out;
  sca_tdf::sca_de::sca_out<double> out_de;

  void processing()
  {
    std::cout << get_time() << ": A" << std::endl;

    for (int i = 0; i < rate;  i++)
    {
      out.write(val, i);
      out_de.write(val);
#ifdef TDF_DEBUG
      std::cout << out.get_time(i) << ": A.out = " << val << std::endl;
      std::cout << out_de.get_time() << ": A.out_de = " << val << std::endl;
#endif
      val++;
    }
  }

  void set_attributes()
  {
    out.set_timestep(1.0, sc_core::SC_MS);
    out.set_rate(rate);
    set_timestep(2.0, sc_core::SC_MS);
  }

  SCA_CTOR( mod_a ) : out("out"), out_de("out_de"), rate(2), val(5.5)
  {}

 private:
  int rate;
  double val;

};

SCA_TDF_MODULE( mod_b )
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;
  sca_tdf::sca_de::sca_in<double> in_de;

  void processing()
  {
    std::cout << get_time() << ": B" << std::endl;

    double a = in.read();
    double b = in_de.read();
#ifdef TDF_DEBUG
    std::cout << in.get_time() << ": B.in = " << a << std::endl;
    std::cout << in_de.get_time() << ": B.in_de = " << b << std::endl;
#endif
    double val = a + b;
    out.write(val);
#ifdef TDF_DEBUG
    std::cout << out.get_time() << ": B.out = " << val << std::endl;
#endif
  }

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_MS);
  }

  SCA_CTOR( mod_b ) : in("in"), out("out"), in_de("in_de")
  {}
};

SCA_TDF_MODULE( mod_c )
{
  sca_tdf::sca_in<double> in;

  void processing()
  {
    std::cout << get_time() << ": C" << std::endl;

    double val = in.read();
#ifdef TDF_DEBUG
    std::cout << in.get_time() << ": C.in = " << val << std::endl;
#endif
  }

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_MS);
  }

  SCA_CTOR( mod_c ) : in("in") {}
};

SC_MODULE( mod_d )
{
  sc_core::sc_out<double> out;

  double val1, val2;

  void gen_val()
  {
    while (true)
    {
      wait(sc_core::sc_time(1.0, sc_core::SC_MS));
      out.write(val1);
#ifdef TDF_DEBUG
      std::cout << sc_core::sc_time_stamp() << ": D.out = " << val1 << std::endl;
#endif
      wait(sc_core::sc_time(1.0, sc_core::SC_MS));
      out.write(val2);
#ifdef TDF_DEBUG
      std::cout << sc_core::sc_time_stamp() << ": D.out = " << val2 << std::endl;
#endif
    }
  }

  SC_CTOR( mod_d ) : out("out"), val1(1.1), val2(2.2)
  {
    SC_THREAD( gen_val );
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;
  sc_core::sc_signal<double> sig3, sig4;

  mod_a a("a");
   a.out(sig1);
   a.out_de(sig4);

  mod_b b("b");
   b.in(sig1);
   b.in_de(sig3);
   b.out(sig2);

  mod_c c("c");
   c.in(sig2);

  mod_d d("d");
   d.out(sig3);

  sca_util::sca_trace_file* atf = sca_util::sca_create_tabular_trace_file("test04");

  sca_util::sca_trace(atf, sig1, "sig1");
  sca_util::sca_trace(atf, sig2, "sig2");
  sca_util::sca_trace(atf, sig3, "sig3");
  sca_util::sca_trace(atf, sig4, "sig4");

  sc_core::sc_start(5.01, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(atf);

  test_util::check_results("test04", 4);

  TEST_LABEL_END;

  return 0;
}
