/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test01.cpp  - eln regression test with eln_tdf modules

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test01)
{
  // port declaration
  sca_tdf::sca_in<double> inp1;
  sca_tdf::sca_in<double> inp2;
  sca_tdf::sca_in<bool>   inp3;

  // node declaration
  sca_eln::sca_node_ref n0; // reference node
  sca_eln::sca_node n1;
  sca_eln::sca_node n2;
  sca_eln::sca_node n3;
  sca_eln::sca_node n4;
  sca_eln::sca_node n5;
  sca_eln::sca_node n6;

  // component declaration
  sca_eln::sca_tdf_vsource  *vin;
  sca_eln::sca_tdf_rswitch  *tdf_rswitch1;
  sca_eln::sca_tdf_r        *tdf_r1;
  sca_eln::sca_tdf_c        *tdf_c1;
  sca_eln::sca_tdf_l        *tdf_l1, *tdf_l2;
  sca_eln::sca_r            *r1, *r2, *r4;

  struct params // parameter
  {
    params() // default for parameter
    {
    }
  };

  test01( sc_core::sc_module_name nm, params pa = params() ) 
  : inp1("inp1"), inp2("inp2"), inp3("inp3"),
    n0("n0"), n1("n1"), n2("n2"), n3("n3"), n4("n4"), n5("n5"), n6("n6"),
    p(pa)
  {
    architecture(); // generate netlist
  }

  ~test01();

 private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test01::architecture()
{
  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->n(n0);
  vin->p(n1);

  tdf_rswitch1 = new sca_eln::sca_tdf_rswitch("tdf_rswitch1");
  tdf_rswitch1->n(n2);
  tdf_rswitch1->p(n4);
  tdf_rswitch1->ctrl(inp3);
  tdf_rswitch1->ron = 1.0e-3;
  tdf_rswitch1->roff = 2.0e-3;
  tdf_rswitch1->off_state = false;

  tdf_r1 = new sca_eln::sca_tdf_r("tdf_r1");
  tdf_r1->scale = 1.0e3;
  tdf_r1->n(n6);
  tdf_r1->p(n1);
  tdf_r1->inp(inp2);

  tdf_c1 = new sca_eln::sca_tdf_c("tdf_c1");
  tdf_c1->scale = 1.0e-2;
  tdf_c1->n(n0);
  tdf_c1->p(n6);
  tdf_c1->inp(inp2);

  tdf_l1 = new sca_eln::sca_tdf_l("tdf_l1");
  tdf_l1->scale = 1.0e-1;
  tdf_l1->n(n4);
  tdf_l1->p(n5);
  tdf_l1->inp(inp2);

  tdf_l2 = new sca_eln::sca_tdf_l("tdf_l2");
  tdf_l2->scale = 1.0e-2;
  tdf_l2->n(n5);
  tdf_l2->p(n6);
  tdf_l2->inp(inp2);

  r1 = new sca_eln::sca_r("r1");
  r1->value = 1.0e2;
  r1->n(n0);
  r1->p(n5);

  r2 = new sca_eln::sca_r("r2");
  r2->value = 1.0e-6;
  r2->n(n3);
  r2->p(n2);

  r4 = new sca_eln::sca_r("r4");
  r4->value = 1.0e2;
  r4->n(n0);
  r4->p(n4);
}

test01::~test01()
{
  delete vin, tdf_rswitch1, tdf_r1, tdf_c1;
  delete tdf_l1, tdf_l2, r1, r2, r4;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..17" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double i_vin, i1, i2, i4, i5, i6, i7, i8, i9;
    str >> i_vin >> i1 >> i2 >> i4 >> i5 >> i6 >> i7 >> i8 >> i9;

    if (time == 0.02)
    {
      if ( !(test_util::check_value(i5, 0.0)) )
      {
        std::cerr << "Error TD-check failed for i5 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i6, 0.00998048798251)) )
      {
        std::cerr << "Error TD-check failed for i6 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i7, 0.00960457718964)) )
      {
        std::cerr << "Error TD-check failed for i7 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i8, 0.000182797606275)) )
      {
        std::cerr << "Error TD-check failed for i8 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i9, 0.000375910792874)) )
      {
        std::cerr << "Error TD-check failed for i9 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if (time == 0.06)
    {
      if ( !(test_util::check_value(i5, 0.0)) )
      {
        std::cerr << "Error TD-check failed for i5 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i6, 0.132849396356)) )
      {
        std::cerr << "Error TD-check failed for i6 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i7, -0.0672420471503)) )
      {
        std::cerr << "Error TD-check failed for i7 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i8, 0.100078250278)) )
      {
        std::cerr << "Error TD-check failed for i8 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;

      if ( !(test_util::check_value(i9, 0.200091443506)) )
      {
        std::cerr << "Error TD-check failed for i9 at: " << time << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if ( !( (test_util::check_value(i6, i7+i9) && test_util::check_value(i9, i1+i8) && 
             test_util::check_value(i8, i4+i5) && test_util::check_value(-i_vin, i1+i4+i7))) )
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;
  };

  if (test_passed_with_success) std::cout  << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;
  sca_tdf::sca_signal<double> signal2;
  sca_tdf::sca_signal<bool>   signal3;

  test_util::const_src::params src_params;
  src_params.ampl = 10;
  test_util::const_src* src = new test_util::const_src("source", src_params);
  src->outp(signal1);

  test_util::pulse_src2::params src_params2;
  src_params2.val = 1.0;
  test_util::pulse_src2* tdf_src = new test_util::pulse_src2("tdf_source", src_params2);
  tdf_src->outp(signal2);

  test_util::pulse_src* swch = new test_util::pulse_src("sca_sdf_rswitch");
  swch->outp(signal3);

  test01* cir = new test01("circuit");
  cir->inp1(signal1);
  cir->inp2(signal2);
  cir->inp3(signal3);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output3.dat");
  sca_util::sca_trace(tf, cir->vin, "i_through_vin");
  sca_util::sca_trace(tf, cir->r1, "i1");
  sca_util::sca_trace(tf, cir->r2, "i2");
  sca_util::sca_trace(tf, cir->r4, "i4");
  sca_util::sca_trace(tf, cir->tdf_rswitch1, "i5");
  sca_util::sca_trace(tf, cir->tdf_r1, "i6");
  sca_util::sca_trace(tf, cir->tdf_c1, "i7");
  sca_util::sca_trace(tf, cir->tdf_l1, "i8");
  sca_util::sca_trace(tf, cir->tdf_l2, "i9");

  sc_core::sc_start(80, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output3.dat");

  TEST_LABEL_END;

  delete src, tdf_src, swch, cir;
  
  return 0;
}
