/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test01.cpp  - eln regression test
//  test01 with one de-inductor and others

#include <systemc-ams>
#include "test_utilities.h"

SC_MODULE(test01)
{
  // port declaration
  sca_tdf::sca_in<double>     inp1;

  // signal declaration
  sca_tdf::sca_signal<double> sig1;
  sc_core::sc_signal<double>  sig2;

  // node declaration
  sca_eln::sca_node n1;
  sca_eln::sca_node_ref n2;   // reference node
  sca_eln::sca_node n3;
  sca_eln::sca_node n4;

  // component declaration
  sca_eln::sca_tdf_vsource *vin;
  sca_eln::sca_de_l        *de_l1;
  sca_eln::sca_l           *l1;
  sca_eln::sca_c           *c1;
  sca_eln::sca_r           *r1;
  test_util::conv_tdf2de   *conv_tdf2de1;

  struct params // parameters
  {
    params() // default for parameter
    {
    }
  };

  test01( sc_core::sc_module_name nm, params pa = params() ) 
  : inp1("inp1"), sig1("sig1"), sig2("sig2"),
    n1("n1"), n2("n2"), n3("n3"), n4("n4"), p(pa)
  {
    architecture(); // generate netlist
  }

  ~test01();

private:
  params p;
  void architecture(); // method for netlist
};

// netlist implementation

void test01::architecture()
{
  conv_tdf2de1 = new test_util::conv_tdf2de("conf_tdf2de1");
  conv_tdf2de1->sc_o(sig2);
  conv_tdf2de1->tdf_i(sig1);

  vin = new sca_eln::sca_tdf_vsource("vin");
  vin->inp(inp1);
  vin->p(n1);
  vin->n(n2);

  de_l1 = new sca_eln::sca_de_l("de_l1");
  de_l1->scale = 1.0e-3;
  de_l1->phi0 = sca_util::SCA_UNDEFINED;
  de_l1->n(n4);
  de_l1->p(n2);
  de_l1->inp(sig2);

  l1 = new sca_eln::sca_l("l1");
  l1->value = 1.0;
  l1->n(n4);
  l1->p(n3);

  c1 = new sca_eln::sca_c("c1");
  c1->value = 1.0e-2;
  c1->n(n2);
  c1->p(n3);

  r1 = new sca_eln::sca_r("r1");
  r1->value = 1.0e3;
  r1->n(n3);
  r1->p(n1);
}

test01::~test01()
{
  delete conv_tdf2de1,
  delete vin, de_l1, l1, c1, r1;
}

void check_td_results(std::string fname)
{
  bool test_passed_with_success = true;
  std::ifstream tdf(fname.c_str());
  std::string line;
  long test_cnt = 0;

  getline(tdf, line);
  std::cout << "1..9" << std::endl;
  while (getline(tdf, line))
  {
    std::istringstream str(line);

    double time;
    str >> time;

    double i_vin, i1, i2, i3, i4;
    str >> i_vin >> i1 >> i2 >> i3 >> i4;

    if ( !(test_util::check_value(i1, i2+i3) && test_util::check_value(-i_vin, i2+i3)))
    {
      std::cerr << "Error TD-check failed for verification of currents at: " << time << std::endl;
      std::cout << "not ";
      test_passed_with_success = false;
    }
    std::cout << "ok " << ++test_cnt << std::endl;

    if (time == 0.02)
    {
      if ( !(test_util::check_value(i4, -0.000246641622808)))
      {
        std::cerr << "Error TD-check failed for i4 at: " << time
                  << " value: " << i4 << " expected: " << -0.000246641622808 << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }

    if (time == 0.06)
    {
      if ( !(test_util::check_value(i4, -0.00181879304219, 1e-15, 3e-2)))
      {
        std::cerr << "Error TD-check failed for i4 at: " << time
                  << " value: " << i4 << " expected: " << -0.00181879304219 << std::endl;
        std::cout << "not ";
        test_passed_with_success = false;
      }
      std::cout << "ok " << ++test_cnt << std::endl;
    }
  };

  if (test_passed_with_success) std::cout  << "results ok." << std::endl;
}

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> signal1;

  test_util::const_src::params src_params;
  src_params.ampl = 10;

  test_util::const_src* src = new test_util::const_src("const_src", src_params);
  src->outp(signal1);

  test01* cir = new test01("test01");
  cir->inp1(signal1);

  test_util::pulse_src2::params pulse_src_params;
  pulse_src_params.val = 1.0;

  test_util::pulse_src2* tdf_src = new test_util::pulse_src2("pulse_src", pulse_src_params);
  tdf_src->outp(cir->sig1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("output4.dat");
  sca_util::sca_trace(tf, cir->vin, "i_vin");
  sca_util::sca_trace(tf, cir->r1, "i1");
  sca_util::sca_trace(tf, cir->c1, "i2");
  sca_util::sca_trace(tf, cir->l1, "i3");
  sca_util::sca_trace(tf, cir->de_l1, "i4");

  sc_core::sc_start(80, sc_core::SC_MS);

  sca_util::sca_close_tabular_trace_file(tf);

  check_td_results("output4.dat");

  TEST_LABEL_END;

  delete src, cir, tdf_src;

  return 0;
}
