/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test sca_ss filter estimate_next_value

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(ss_filter)
{
  sca_tdf::sca_in<double> in;

  sca_tdf::sca_out<double> out;
  sca_tdf::sca_out<double> out_ref;
  sca_tdf::sca_out<double> out_ref_error;
  sca_tdf::sca_out<double> out_estimated;
  sca_tdf::sca_out<double> out_estimated_error;

  ss_filter( sc_core::sc_module_name nm, double fc_)
  : in("in"), out("out"), out_ref("out_ref"), out_ref_error("out_ref_error"),
  out_estimated("out_estimated"), out_estimated_error("out_estimated_error"),
  ss("ss"), ss_ref("ss_ref"), fc(fc_) 
  {}

  void set_attributes()
  {
    out_estimated.set_delay(1);
    out_estimated_error.set_delay(1);
  }

  void initialize()
  {
    double r_val = 1.0e3;
    double c_val = 1.0/(2.0 * M_PI * fc * r_val);

    // dQ=i=1/r * (vin-vout)  ; vout = 1/c *Q
    // dQ   = -1/r/c * Q    + 1/r  * vin
    // vout =   1/c  * Q    +  0.0 * vin
    A(0, 0) = -1.0/c_val/r_val;
    B(0, 0) = 1.0/r_val;

    C(0, 0) = 1.0/c_val;
    D(0, 0) = 0.0;

    estimated_value = 0.0;
    last_out = 0.0;
    cnt = 0;

    out_estimated.initialize(0.0);
    out_estimated_error.initialize(0.0);
  }
 
  void processing()
  {
    sca_util::sca_vector<double>  tmp_in(1);
    tmp_in(0) = in.read();

    sca_util::sca_vector<double> tmp     = ss(A, B, C, D, tmp_in);
    sca_util::sca_vector<double> tmp_ref = ss_ref(A, B, C, D, tmp_in);

    double tmp_ref_error = tmp(0) - tmp_ref(0);
    double estimation_error = estimated_value - tmp(0);

    out.write(tmp(0));
    out_ref.write(tmp_ref(0));
    out_ref_error.write(tmp_ref_error);
    out_estimated_error.write(estimation_error);

    sca_util::sca_vector<double> tmp_estimated;
    tmp_estimated = ss.estimate_next_value();
    estimated_value = tmp_estimated(0);
    out_estimated.write(estimated_value);

    if (((std::fabs(tmp(0)) > 1e-10) && (std::fabs(tmp_ref_error/tmp(0)) > 1e-13)) ||
       ((std::fabs(tmp(0)) <= 1e-10) && (std::fabs(tmp_ref_error)     > 1e-13)) )
    {
      std::ostringstream str;
      str << "Wrong ltf reference error: " << tmp_ref_error;
      SC_REPORT_ERROR("estimate_next_value", str.str().c_str());
    }

    if ((cnt > 2) && (std::fabs(estimation_error) > fabs(tmp(0) - last_out) / 5.0 ))
    {
      std::ostringstream str;
      str << "Difference estimated to correct value is to high: " << estimation_error;
      SC_REPORT_ERROR("estimate_next_value", str.str().c_str());
    }

    last_out = tmp(0);
    cnt++;
  }

 private:
  sca_tdf::sca_ss ss; // Laplace transfer function
  sca_tdf::sca_ss ss_ref;
  sca_util::sca_matrix<double> A, B, C, D;
  double fc; // 3dB cutoff frequency in Hz
  double estimated_value;
  double last_out;
  unsigned long cnt;
};

SCA_TDF_MODULE(my_sin)
{
  sca_tdf::sca_out<double> out; // output port

  SCA_CTOR(my_sin) : out("out")
  {
  }

  void set_attributes()
  {
    out.set_timestep(10.0, sc_core::SC_US);
  }

  void processing()
  {
    out.write(std::sin(1.0 * M_PI * 1e3 * get_time().to_seconds()));
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;
  sca_tdf::sca_signal<double> sig2_ref;
  sca_tdf::sca_signal<double> sig2_ref_error;
  sca_tdf::sca_signal<double> sig2_estimated;
  sca_tdf::sca_signal<double> sig2_estimation_error;

  my_sin sin1("sin1");
    sin1.out(sig1);

  ss_filter ss1("ss1", 1.0e3);
    ss1.in(sig1);
    ss1.out(sig2);
    ss1.out_ref(sig2_ref);
    ss1.out_ref_error(sig2_ref_error);
    ss1.out_estimated(sig2_estimated);
    ss1.out_estimated_error(sig2_estimation_error);

  // tracing
  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("ltf_filter");
  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");
  sca_util::sca_trace(tf, sig2_ref, "sig2_ref");
  sca_util::sca_trace(tf, sig2_ref_error, "sig2_ref_error");
  sca_util::sca_trace(tf, sig2_estimated, "sig2_estimated");
  sca_util::sca_trace(tf, sig2_estimation_error, "sig2_estimation_error");

  sc_core::sc_start(5.0, sc_core::SC_MS);
  
  sca_util::sca_close_tabular_trace_file(tf);
  
  TEST_LABEL_END;
    
  return 0;
}
