/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

/*
 *   assignment to out via operator using calculate()
 *
 *   Tests:
 *           single rate out / single rate in  (vector,vector,?in?)
 *                                             scalar out / scalar in
 *                                             port out   / scalar in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           single rate out / multi rate in  (vector,vector,?in?)
 *                                             scalar out / vector in
 *                                             port out   / vector in
 *                                             scalar out / port in
 *                                             port out   / port in
 *
 *           multi rate out / single rate in  (vector,vector, sc_time,?in?)
 *           //delay is required due causality (sc_time argument)
 *                                             vector out / scalar in
 *                                             port out   / scalar in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 *           multi rate out / multi rate in  (vector,vector,?in?)
 *                                             vector out / vector in
 *                                             port out   / vector in
 *                                             vector out / port in
 *                                             port out   / port in
 *
 */


#include <systemc-ams>
#include "test_utilities.h"

#include <string>

///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(ss_src)
{
  // single rate
  sca_tdf::sca_out<double> outp_const_1; // constant value (1.0)
  sca_tdf::sca_out<double> outp_ramp_1;  // ramp  y=t
  sca_tdf::sca_out<double> outp_t2_1;    // y=0.5*t*t
  sca_tdf::sca_out<double> outp_sin_1;
  sca_tdf::sca_out<double> outp_t3_1;    // y=0.5/3*t*t*t

  // single rate
  sca_tdf::sca_out<sca_util::sca_vector<double> > outp_vec_1;

  struct params
  {
    int rate;
    sca_core::sca_time sample_time;
    double f_sin;

    params()
    {
      rate = 5;
      sample_time = sca_core::sca_time(1.0, sc_core::SC_US);
      f_sin = 13.3e3;
    }
  };

  void set_attributes()
  {
    set_timestep(p.sample_time);
    does_attribute_changes();
  }

  void initialize();
  void processing();

  void change_attributes()
  {
    switch(activation)
    {
      case 0: // mstep=1ms
        break;

      case 5:
        break;

      case 6:
        break;

      case 7:
        break;

      case 508: // change timestep to 2sec and calculate to end
        set_timestep(2.0, sc_core::SC_US);
        break;

      case 509:
        break;

      case 510:
        break;

      case 511:
        break;

      case 512:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 13:
        break;

      case 514:  // 3 zero timestep module iterations, calculation to end
      case 515:
      case 516:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 517: // 4 zero timestep
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 518: // module timestep 1sec 4 zero tstep iterations
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 519: // calculate to end
        break;

      case 520: // 2 zero module timestep iterations
      case 521:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 522:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 523:
      case 524:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 525:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 526:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 527:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 528:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 529:
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 530:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 531: // reset to begin of case 28
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 532:
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 533: // normal full timestep
        set_timestep(1.0, sc_core::SC_US);
        break;

      case 534: // reset to tstep before last timestep (before last iteration of 32)
        request_next_activation(0.0, sc_core::SC_US);
        break;

      case 535: // repeat as normal full timestep
        request_next_activation(0.0, sc_core::SC_SEC);
        break;

      case 600:
        sc_core::sc_stop();
        break;

      default: break;
    }
  }

  ss_src(sc_core::sc_module_name nm, params pa = params()) 
  : outp_const_1("outp_const_1"), outp_ramp_1("outp_ramp_1"), 
    outp_t2_1("outp_t2_1"), outp_sin_1("outp_sin_1"), outp_t3_1("outp_t3_1"),
    ss1("ss1"), ss3("ss3"), 
    activation(0), p(pa)
  {
  }
  
 private:
  sca_tdf::sca_ss  ss1, ss3;
  sca_util::sca_matrix<double> A1, B1, C1, D1;
  int activation;
  params p;
};

void ss_src::initialize()
{
  A1(0, 0) = 0.0;
  A1(1, 0) = 1.0;  // ds = Integral(x)
  A1(1, 1) = 0.0;
  A1(2, 1) = 1.0;  // ds = integral(Integral(x))
  A1(2, 2) = 0.0;

  B1(0, 0) = 1.0;  // ds = x
  B1(1, 0) = 0.0;
  B1(1, 1) = 0.0;
  B1(2, 1) = 0.0;

  C1(0, 0) = 0.0;
  C1(1, 0) = 1.0;  // y1=t*x
  C1(2, 0) = 0.0;

  C1(2, 1) = 1.0;  // y2=0.5*t*t*x
  C1(3, 1) = 0.0;
  C1(4, 2) = 1.0;  // y2=0.5/3*t*t*t*x

  D1(0, 0) = 1.0;  // y0=x
  D1(2, 0) = 0.0;
  D1(3, 1) = 1.0;  // y=sin(...)
  D1(4, 1) = 0.0;
}

// time domain implementation
void ss_src::processing()
{
  sca_util::sca_vector<double> inp;
  inp(0) = 1.0;
  inp(1) = std::sin(2.0 * M_PI * p.f_sin * get_time().to_seconds());

  //output to vector port sca_tdf::sca_out<sca_util::sca_vector<double> >
  //input vector sca_util::sca_vector<double>
  outp_vec_1 = ss1(A1, B1, C1, D1, inp); // single rate

  sca_util::sca_vector<double> out_vec;

  // output to vector sca_util::sca_vector<double>
  // input vector sca_util::sca_vector<double>
  out_vec = ss3(A1, B1, C1, D1, inp);

  outp_const_1 = out_vec(0);
  outp_ramp_1  = out_vec(1);
  outp_t2_1    = out_vec(2);
  outp_sin_1   = out_vec(3);
  outp_t3_1    = out_vec(4);

  activation++;
}

////////////////////////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE (ss_test)
{
  // single rate
  sca_tdf::sca_in<double> in_const_1; // constant value (1.0)
  sca_tdf::sca_in<double> in_ramp_1;  // ramp  y=t
  sca_tdf::sca_in<double> in_t2_1;    // y=0.5*t*t
  sca_tdf::sca_in<double> in_sin_1;

  sca_tdf::sca_out<double> out_const_1;
  sca_tdf::sca_out<double> out_ramp_1;
  sca_tdf::sca_out<double> out_t2_1;
  sca_tdf::sca_out<double> out_sin_1;
  sca_tdf::sca_out<double> out_t3_1;

  sca_tdf::sca_out<double> out_const_3;
  sca_tdf::sca_out<double> out_ramp_3;
  sca_tdf::sca_out<double> out_t2_3;
  sca_tdf::sca_out<double> out_sin_3;
  sca_tdf::sca_out<double> out_t3_3;

  sca_tdf::sca_out<double> out_const_5;
  sca_tdf::sca_out<double> out_ramp_5;
  sca_tdf::sca_out<double> out_t2_5;
  sca_tdf::sca_out<double> out_sin_5;
  sca_tdf::sca_out<double> out_t3_5;

  sca_tdf::sca_in<sca_util::sca_vector<double> > in_vec_1;

  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_2;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_4;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_6;
  sca_tdf::sca_out<sca_util::sca_vector<double> > out_vec_7;

  struct params // parameter
  {
    unsigned long vlength,vlength2;

    params() // default for parameter
    {
      vlength = 5;
      vlength2 = 3;
    }
  };

  void initialize();     // initialization
  void processing();     // time domain processing method
  void set_attributes();

  // constructor
  ss_test(sc_core::sc_module_name nm, params pa = params()) 
  : in_const_1("in_const_1"), in_ramp_1("in_ramp_1"), in_t2_1("in_t2_1"), in_sin_1("in_sin_1"),
    out_const_1("out_const_1"), out_ramp_1("out_ramp_1"), out_t2_1("out_t2_1"), out_sin_1("out_sin_1"), out_t3_1("out_t3_1"),
    out_const_3("out_const_3"), out_ramp_3("out_ramp_3"), out_t2_3("out_t2_3"), out_sin_3("out_sin_3"), out_t3_3("out_t3_3"),
    out_const_5("out_const_5"), out_ramp_5("out_ramp_5"), out_t2_5("out_t2_5"), out_sin_5("out_sin_5"), out_t3_5("out_t3_5"),
    in_vec_1("in_vec_1"), out_vec_2("out_vec_2"), out_vec_4("out_vec_4"), out_vec_6("out_vec_6"), out_vec_7("out_vec_7"),
    ss1("ss1"), ss2("ss2"), ss3("ss3"), ss4("ss4"), ss5("ss5"), ss6("ss6"), ss7("ss7"), ss_vector("ss_vector"),
    p(pa)
  {}

  // definition of local variables
private:
  sca_tdf::sca_ss ss1, ss2, ss3, ss4, ss5, ss6, ss7;
  sca_tdf::sca_ss ss_vector;
  sca_util::sca_matrix<double> A, B, C, D;
  sca_util::sca_matrix<double> A_vector, B_vector, C_vector, D_vector;
  params p;
};

void ss_test::set_attributes()
{
  accept_attribute_changes();
}

void ss_test::initialize()
{
  A(0, 0) = 0.0;
  B(0, 4) = 1.0;  // ds=x

  C(4, 0) = 1.0;  // y= Integral(x)

  D(0, 0) = 1.0;
  D(1, 1) = 1.0;
  D(2, 2) = 1.0;
  D(3, 3) = 1.0;
  D(4, 4) = 0.0;

  D_vector(0, 0) = 1.0;
  D_vector(1, 1) = 1.0;
  D_vector(2, 2) = 1.0;
  D_vector(3, 3) = 1.0;
  D_vector(4, 4) = 1.0;
}

// time domain implementation
void ss_test::processing()
{
  sca_util::sca_vector<double> x;

  x(0) = in_const_1.read();
  x(1) = in_ramp_1.read();
  x(2) = in_t2_1.read();
  x(3) = in_sin_1.read();
  x(4) = in_t2_1.read();

  //create matrix input
  sca_util::sca_matrix<double> x_mat_vector(D.n_rows(), p.vlength);
  x_mat_vector = ss_vector(A_vector, B_vector, C_vector, D_vector, x);

  ////////// single rate in / single rate out //////////////

  // vector out / vector in
  const sca_util::sca_vector<double> y1 = ss1.calculate(A, B, C, D, x);

  out_const_1 = y1(0);
  out_ramp_1  = y1(1);
  out_t2_1    = y1(2);
  out_sin_1   = y1(3);
  out_t3_1    = y1(4);

  // port out / vector in
  out_vec_2 = ss2.calculate(A, B, C, D, x);

  // vector out / port in
  const sca_util::sca_vector<double> y3 = ss3.calculate(A, B, C, D, in_vec_1);

  out_const_3 = y3(0);
  out_ramp_3  = y3(1);
  out_t2_3    = y3(2);
  out_sin_3   = y3(3);
  out_t3_3    = y3(4);

  // port out / port in
  out_vec_4 = ss4.calculate(A, B, C, D, in_vec_1);

  // vector out / matrix in (length=5)
  //use delayed matrix, due the last vector value corresponds to the out port value
  const sca_util::sca_vector<double> y5 = ss5.calculate(A, B, C, D, x_mat_vector);

  out_const_5 = y5(0);
  out_ramp_5  = y5(1);
  out_t2_5    = y5(2);
  out_sin_5   = y5(3);
  out_t3_5    = y5(4);

  ////////////////////////////////////////////////////////////////////////

  //port out / matrix in (rate=5)
  out_vec_6 = ss6.calculate(A, B, C, D, x_mat_vector);

  //////////////////////////////////////////////////////////////////////

  //matrix out (length=3) / matrix in (length=5)
  sca_util::sca_matrix<double> y7(D.n_rows(), p.vlength2);

  y7 = ss7.calculate(A, B, C, D, x_mat_vector);

  sca_util::sca_vector<double> y7_vec(D.n_rows());

  for (unsigned long i = 0; i < D.n_rows(); i++)
  {
    y7_vec(i) = y7(i, p.vlength2 - 1);
  }
  out_vec_7 = y7_vec;
}

//////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(vector_trace)
{
  sca_tdf::sca_in<sca_util::sca_vector<double> > in_vec;
  sca_tdf::sca_trace_variable<double> c0;
  sca_tdf::sca_trace_variable<double> c1;
  sca_tdf::sca_trace_variable<double> c2;
  sca_tdf::sca_trace_variable<double> c3;
  sca_tdf::sca_trace_variable<double> c4;

  void processing()
  {
    if (in_vec.read().length() != 5)
    {
      SC_REPORT_ERROR("vector_trace", "not supported vector length");
    }

    c0 = in_vec.read()(0);
    c1 = in_vec.read()(1);
    c2 = in_vec.read()(2);
    c3 = in_vec.read()(3);
    c4 = in_vec.read()(4);
  }

  vector_trace( sc_core::sc_module_name, sca_util::sca_trace_file* tf, std::string nm) 
  : in_vec("in_vec"), c0("c0"), c1("c1"), c2("c2"), c3("c3"), c4("c4")
  {
    sca_util::sca_trace(tf, c0, nm + "_0");
    sca_util::sca_trace(tf, c1, nm + "_1");
    sca_util::sca_trace(tf, c2, nm + "_2");
    sca_util::sca_trace(tf, c3, nm + "_3");
    sca_util::sca_trace(tf, c4, nm + "_4");

    accept_attribute_changes();
  }
};

namespace sca_util
{
  void sca_trace(sca_util::sca_trace_file* tf, sca_tdf::sca_signal<sca_util::sca_vector<double> >& sig, std::string nm)
  {
    vector_trace* tfm = new vector_trace(sc_core::sc_gen_unique_name("vector_trace"), tf, nm);
    tfm->in_vec(sig);
  }
}

///////////////////////////////////////////////////////////////////////////////

bool check_val(double val, double e_val, double rel_err, double abs_err)
{
  if ( (std::fabs(val - e_val) > abs_err) && (std::fabs(val - e_val) / e_val > rel_err))
  {
    std::cout << " Expect value: " << e_val << " read value: " << val << "  diff: " << val - e_val << std::endl;
    return true;
  }

  return false;
}

int check_values(double time, double delay, double cval, double rampv, double t2_val, double sin_val,
                 double t3_val, double rel_err = 1e-10, double abs_err = 1e-10)
{
  double ctime = time-delay;
  double e_cval = 0.0;
  double e_rampv = 0.0;
  double e_t2_val = 0.0;
  double e_sin_val = 0.0;
  double e_t3_val = 0.0;

  if (ctime >= 0.0)
  {
    e_cval = 1.0;
    e_rampv = ctime;
    e_t2_val = 0.5 * ctime * ctime;
    e_sin_val = std::sin(2.0 * M_PI * 13.3e3 * ctime);
    e_t3_val = 0.5 / 3.0 * ctime * ctime * ctime;
  }
  else
  {
    return 0;
  }

  if (check_val(cval, e_cval, rel_err, abs_err))
  {
    return 1;
  }

  if (check_val(rampv, e_rampv, rel_err, abs_err))
  {
    return 2;
  }

  if (check_val(t2_val, e_t2_val, rel_err, abs_err))
  {
    return 3;
  }

  if (check_val(t3_val, e_t3_val, rel_err, abs_err))
  {
    return 5;
  }

  return 0;
}

bool check_file(std::string fname, const std::vector<double>& delays)
{
  std::ifstream fin;
  fin.open(fname.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  std::string line_in;
  unsigned long line_cnt = 0;
  if (!std::getline(fin, line_in))
  {
    std::ostringstream str;
    str << "Can't get header line from file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }
  line_cnt++;

  std::vector<std::string> names;
  std::istringstream nistr(line_in);

  std::string name;
  if (!(nistr >> name) && (name != "%time"))
  {
    std::ostringstream str;
    str << "Wrong header line in file: " << fname;
    SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
  }

  while (nistr >> name)
  {
    names.push_back(name);
  }

  while (std::getline(fin, line_in))
  {
    std::istringstream istr(line_in);
    line_cnt++;

    double ctime;
    istr >> ctime;

    if (istr.fail())
    {
      std::ostringstream str;
      str << "Can't get time in file: " << fname << " line: " << line_cnt;
      SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
    }

    for (unsigned long i = 0; i < names.size(); i += 5)
    {
      double cval, rval, t2val, sinval, t3val;

      istr >> cval;
      istr >> rval;
      istr >> t2val;
      istr >> sinval;
      istr >> t3val;

      if (istr.fail())
      {
        std::ostringstream str;
        str << "Failed to read values from file: " << fname << " line: " << line_cnt;
        str << " for: " << names[i] << " " << names[i+1] << " "
            << names[i+2] << " " << names[i+3]<< " " << names[i+4];
        str << " at time: " << ctime;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }

      int err = check_values(ctime, delays[i / 5], cval, rval, t2val, sinval, t3val);
      if (err != 0)
      {
        std::ostringstream str;
        str << "Wrong value in file: " << fname << " line: " << line_cnt;
        str << " for signal: " << names[i+err-1] << " at time " << ctime;
        str << " delay: " << delays[i / 5] << " d index: " << i / 5;
        SC_REPORT_ERROR("ND_CHECK", str.str().c_str());
      }
    }
  }
  return false;
}

int sc_main(int argn, char* argc[])
{
  sc_core::sc_set_time_resolution(1.0, sc_core::SC_FS);

  TEST_LABEL_START;

  // define non-conservative signals
  sca_tdf::sca_signal<double> s_out_const1, s_out_ramp1, s_out_t21, s_out_sin1, s_out_t31;
  sca_tdf::sca_signal<double> s_out_const_1, s_out_ramp_1, s_out_t2_1, s_out_sin_1, s_out_t3_1;
  sca_tdf::sca_signal<double> s_out_const_3, s_out_ramp_3, s_out_t2_3, s_out_sin_3, s_out_t3_3;
  sca_tdf::sca_signal<double> s_out_const_5, s_out_ramp_5, s_out_t2_5, s_out_sin_5, s_out_t3_5;

  sca_tdf::sca_signal<sca_util::sca_vector<double> > s_outvec_1, s_outvec_2, s_outvec_4, s_outvec_6, s_outvec_7;

  // instantiate and connect components

  ss_src* i_src = new ss_src("i_src");
  i_src->outp_const_1(s_out_const1);
  i_src->outp_ramp_1(s_out_ramp1);
  i_src->outp_t2_1(s_out_t21);
  i_src->outp_sin_1(s_out_sin1);
  i_src->outp_t3_1(s_out_t31);

  i_src->outp_vec_1(s_outvec_1);

  ss_test* i_ss_test = new ss_test("i_ss_test");
  i_ss_test->in_const_1(s_out_const1);
  i_ss_test->in_ramp_1(s_out_ramp1);
  i_ss_test->in_t2_1(s_out_t21);
  i_ss_test->in_sin_1(s_out_sin1);

  i_ss_test->in_vec_1(s_outvec_1);

  i_ss_test->out_const_1(s_out_const_1);
  i_ss_test->out_ramp_1(s_out_ramp_1);
  i_ss_test->out_t2_1(s_out_t2_1);
  i_ss_test->out_sin_1(s_out_sin_1);
  i_ss_test->out_t3_1(s_out_t3_1);

  i_ss_test->out_const_3(s_out_const_3);
  i_ss_test->out_ramp_3(s_out_ramp_3);
  i_ss_test->out_t2_3(s_out_t2_3);
  i_ss_test->out_sin_3(s_out_sin_3);
  i_ss_test->out_t3_3(s_out_t3_3);

  i_ss_test->out_const_5(s_out_const_5);
  i_ss_test->out_ramp_5(s_out_ramp_5);
  i_ss_test->out_t2_5(s_out_t2_5);
  i_ss_test->out_sin_5(s_out_sin_5);
  i_ss_test->out_t3_5(s_out_t3_5);

  i_ss_test->out_vec_2(s_outvec_2);
  i_ss_test->out_vec_4(s_outvec_4);
  i_ss_test->out_vec_6(s_outvec_6);
  i_ss_test->out_vec_7(s_outvec_7);

  /////// Tracing -> for each rate a different file

  sca_util::sca_trace_file* tf1 = sca_util::sca_create_tabular_trace_file("tdf_ss_1_1.dat");

  sca_util::sca_trace(tf1, s_out_const1, "s_out_const1");
  sca_util::sca_trace(tf1, s_out_ramp1, "s_out_ramp1");
  sca_util::sca_trace(tf1, s_out_t21, "s_out_t21");
  sca_util::sca_trace(tf1, s_out_sin1, "s_out_sin1");
  sca_util::sca_trace(tf1, s_out_t31, "s_out_t31");

  sca_util::sca_trace(tf1, s_outvec_1, "s_outvec_1");

  sca_util::sca_trace(tf1, s_out_const_1, "s_out_const_1");
  sca_util::sca_trace(tf1, s_out_ramp_1, "s_out_ramp_1");
  sca_util::sca_trace(tf1, s_out_t2_1, "s_out_t2_1");
  sca_util::sca_trace(tf1, s_out_sin_1, "s_out_sin_1");
  sca_util::sca_trace(tf1, s_out_t3_1, "s_out_t3_1");

  sca_util::sca_trace(tf1, s_out_const_3, "s_out_const_3");
  sca_util::sca_trace(tf1, s_out_ramp_3, "s_out_ramp_3");
  sca_util::sca_trace(tf1, s_out_t2_3, "s_out_t2_3");
  sca_util::sca_trace(tf1, s_out_sin_3, "s_out_sin_3");
  sca_util::sca_trace(tf1, s_out_t3_3, "s_out_t3_3");

  sca_util::sca_trace(tf1, s_out_const_5, "s_out_const_5");
  sca_util::sca_trace(tf1, s_out_ramp_5, "s_out_ramp_5");
  sca_util::sca_trace(tf1, s_out_t2_5, "s_out_t2_5");
  sca_util::sca_trace(tf1, s_out_sin_5, "s_out_sin_5");
  sca_util::sca_trace(tf1, s_out_t3_5, "s_out_t3_5");

  sca_util::sca_trace(tf1, s_outvec_2, "s_outvec_2");
  sca_util::sca_trace(tf1, s_outvec_4, "s_outvec_4");
  sca_util::sca_trace(tf1, s_outvec_6, "s_outvec_6");
  sca_util::sca_trace(tf1, s_outvec_7, "s_outvec_7");

  ////////////////////////////////////////////////

  sc_core::sc_start(); //start time domain simulation for 5ms

  sca_util::sca_close_tabular_trace_file(tf1);
  
  std::vector<double> delays;
  delays.resize(20, 0.0);

  // delay is not shown, due we propagate the first value to zero -> thus at time
  // zero the constant is valid -> the ramp starts at zero (begin of integration with 1)
  check_file("tdf_ss_1_1.dat", delays);

  //////////////////////////////////////////////////////////////////////////

  if (sc_core::sc_report_handler::get_count(sc_core::SC_ERROR) <= 0)
  {
    SC_REPORT_INFO("ND_CHECK", "Test was successful");
  }

  TEST_LABEL_END;

  delete i_src, i_ss_test;

  return 0;
}

