/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test ltf_zp for SystemC-AMS 2.0 dtdf -
// correct integration with variable timesteps, test for SC_ZERO_TIME

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE( my_ltf )
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  my_ltf( sc_core::sc_module_name nm )
  : in("in"), out("out"),
    activation(0), tstep(sca_core::sca_max_time()), imax(1)
  {}

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_SEC);
    does_attribute_changes();
    ltf_zp.enable_iterations();
  }

  void initialize()
  {
    // H(s) = 1 / s
    poles(0) = 0.0;
    gain     =  1.0;

    std::cout << "@" << get_time() << ": activation = " << activation
        << " initialize: LTF tstep follows TDF timestep = " << get_timestep() << std::endl;
  }

  void processing()
  {
    double inp = in.read();
    double tmp;

    std::string tstep_str;
    if (tstep == sca_core::sca_max_time()) tstep_str = "sca_max_time";
    else                                   tstep_str = tstep.to_string();

    std::string tstep_it_str;
    if (tstep_it == sca_core::sca_max_time()) tstep_it_str = "sca_max_time";
    else                                      tstep_it_str = tstep_it.to_string();

    // make remainig timestep repeat visible
    if ((activation == 31) || (activation == 34)) inp = 0.0;

    tmp = ltf_zp( zeros, poles, inp, gain, tstep);
    std::cout << "@" << get_time() << ": activation = " << activation << "        mstep: " << get_timestep() << "   tstep: " << tstep_str << "            input = " << inp << "  output = " << tmp << std::endl;

    for (int i = 1; i < imax; i++)
    {
      if (i%2)
      {
        inp = in.read() + 1.0;
      }
      else
      {
        inp = in.read() - 1.0;
      }

      if (i > (imax / 2 * 2 - 2)) inp = in.read();

      tmp = ltf_zp( zeros, poles, inp, gain, tstep_it);
      std::cout << "     @" << get_time() << ": activation = " << activation << "   mstep: " << get_timestep() << "   tstep: " << tstep_it_str << "            input = " << inp << "  output = " << tmp << std::endl;
    }
    out.write(tmp);

    activation++;
  }

  void change_attributes()
  {
    switch(activation)
    {
      case 0: // mstep=1ms tstep=sca_max_time
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 5: // 10 iteration with tstep 0.1 sec, mstep=1sec
        tstep = sca_core::sca_time(0.1, sc_core::SC_SEC);
        imax = 10;
        tstep_it = tstep;
        break;

      case 6: // 6 iteration 0.1sec -> end of calculation 0.6sec
        tstep = sca_core::sca_time(0.1, sc_core::SC_SEC);
        tstep_it = tstep;
        imax = 6;
        break;

      case 7: // mstep=1sec tstep=sca_max_time -> calculate remaining and to end
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 8: // change timestep to 2sec and calculate to end
        set_timestep(2.0, sc_core::SC_SEC);
        imax = 1;
        break;

      case 9: // 10 iterations wit tstep=0.2sec (calculates to end)
        tstep = sca_core::sca_time(0.2, sc_core::SC_SEC);
        tstep_it = tstep;
        imax = 10;
        break;

      case 10: // 1 timestep to end
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 11: // 10 zero timestep iterations
        tstep = sc_core::SC_ZERO_TIME;
        tstep_it = tstep;
        imax = 10;
        break;

      case 12: // change timestep to 1sec and do 4 zero timestep iterations
        tstep = sc_core::SC_ZERO_TIME;
        imax = 4;
        tstep_it = tstep;
        set_timestep(1.0, sc_core::SC_SEC);
        break;

      case 13: // calculate to end
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 14: // 3 zero timestep module iterations, calculation to end
      case 15:
      case 16:
        imax = 1;
        tstep = sca_core::sca_max_time();
        set_timestep(0.0, sc_core::SC_SEC);
        break;

      case 17: // 4 zero timestep and tstep=0 iterations
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sc_core::SC_ZERO_TIME;
        imax = 4;
        tstep_it = tstep;
        break;

      case 18: // module timestep 1sec 4 zero tstep iterations
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sc_core::SC_ZERO_TIME;
        imax = 4;
        tstep_it = tstep;
        break;

      case 19: // calculate to end
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 20: // 2 zero module timestep iterations
      case 21:
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 22:
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sca_core::sca_time(0.2, sc_core::SC_SEC);
        imax = 6;
        tstep_it = sc_core::SC_ZERO_TIME;
        break;

      case 23:
      case 24:
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sca_core::sca_time(0.2, sc_core::SC_SEC);
        imax = 6;
        tstep_it = sc_core::SC_ZERO_TIME;
        break;

      case 25:
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 26:
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 6;
        tstep_it = sc_core::SC_ZERO_TIME;
        break;

      case 27:
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 6;
        tstep_it = sc_core::SC_ZERO_TIME;
        break;

      case 28:
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 29:
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sca_core::sca_time(0.4, sc_core::SC_SEC);
        imax = 6;
        tstep_it = sca_core::sca_time(0.1, sc_core::SC_SEC);
        break;

      case 30:
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sca_core::sca_time(0.4, sc_core::SC_SEC);
        imax = 6;
        tstep_it = sca_core::sca_time(0.1, sc_core::SC_SEC);
        break;

      case 31: // reset to begin of case 28
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sc_core::SC_ZERO_TIME;
        imax = 6;
        tstep_it = sca_core::sca_time(0.1, sc_core::SC_SEC);
        break;

      case 32:
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sc_core::SC_ZERO_TIME;
        imax = 6;
        tstep_it = sca_core::sca_time(0.1, sc_core::SC_SEC);
        break;

      case 33: // normal full timestep
        set_timestep(1.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 34: // reset to tstep before last timestep (before last iteration of 32)
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sc_core::SC_ZERO_TIME;
        imax = 6;
        tstep_it = sca_core::sca_time(0.1, sc_core::SC_SEC);
        break;

      case 35: // repeat as normal full timestep
        set_timestep(0.0, sc_core::SC_SEC);
        tstep = sca_core::sca_max_time();
        imax = 1;
        break;

      case 36:
        set_timestep(1.0, sc_core::SC_SEC);
        sc_core::sc_stop();
        break;

      default:
        imax = 1;
        break;
    }
  }

private:
  sca_tdf::sca_ltf_zp ltf_zp; // Laplace transfer function
  sca_util::sca_vector<sca_util::sca_complex> poles, zeros;
  sca_util::sca_vector<double> state; // numerator and denominator coefficients
  double gain;
  sca_core::sca_time tstep; // LTF time step
  sca_core::sca_time tstep_it;  //tstep during iteration
  int activation;
  int imax;
};

SCA_TDF_MODULE(my_step)
{
  sca_tdf::sca_out<double> out; // output port

  SCA_CTOR(my_step) : out("out")
  {}

  void set_attributes()
  {
    out.set_delay(0);
    accept_attribute_changes();
  }

  void initialize()
  {}

  void processing()
  {
    out.write(1.0);
  }

  void reinitialize()
  {}

  void change_attributes()
  {}
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1;
  sca_tdf::sca_signal<double> sig2;

  my_step step("step");
  my_ltf  ltf1("ltf1");

  step.out(sig1);
  ltf1.in(sig1);
  ltf1.out(sig2);

  sc_core::sc_start();

  TEST_LABEL_END;

  return 0;
}
