/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// test change delay for SystemC-AMS 2.0 dtdf (outport delay)

#include <systemc-ams>
#include "test_utilities.h"

test_util::sorted_stream sstr;

static long g_in_cnt = 0;
static long g_ignore = 0;

std::vector<long> out_ref;

void change_delay_outref(unsigned long old_delay, unsigned long new_delay)
{
  if (new_delay > old_delay)
  {
    std::vector<long> tmp(new_delay - old_delay, 0);
    out_ref.insert(out_ref.end() - old_delay, tmp.begin(), tmp.end());
  }
  else
  {
    std::vector<long>::iterator it_start, it_end;
    it_start = out_ref.end() - old_delay;
    it_end = it_start+old_delay - new_delay;
    out_ref.erase(it_start, it_end);
  }
}

// check trace file

void check_trace_file(std::string name)
{
  std::fstream fin;

  fin.open(name.c_str());

  if (fin.fail())
  {
    std::ostringstream str;
    str << "Can't open trace file: " << name;
    SC_REPORT_ERROR("test", str.str().c_str());
    return;
  }

  std::string header;
  std::getline(fin, header);

  long cnt = 0;

  while (!fin.eof())
  {
    std::string line;

    std::getline(fin, line);
    std::istringstream line_str(line);

    double time;
    long   value;

    line_str >> time;
    if (line_str.fail()) continue; // ignore empty lines

    line_str >> value;

    if (value != out_ref[cnt])
    {
      std::ostringstream str;
      str << "Unexpexted value in trace file at time: " << time;
      str << " expected value: " << out_ref[cnt] << " read value: "
          << value << " cnt: " << cnt;
      SC_REPORT_ERROR("test", str.str().c_str());
      return;
    }

    cnt++;
  }
}

bool is_time_not_equal(const sca_core::sca_time& t1, const sca_core::sca_time& t2)
{
  sca_core::sca_time tdiff;
  if (t1 > t2) tdiff = t1 - t2;
  else         tdiff = t2 - t1;
  return tdiff > sc_core::sc_get_time_resolution();
}


SCA_TDF_MODULE( mod_a )
{
  sca_tdf::sca_out<long> outp;

  void print_status()
  {
    std::cout << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    std::cout << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    std::cout << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    std::cout << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << "   get_timestep:                  " << get_timestep() << std::endl;
    std::cout << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    std::cout << "   get_max_timestep:              ";
    if (get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << get_max_timestep() << std::endl;

    std::cout << "   get_time:                      " << get_time() << std::endl;
    std::cout << std::endl;
    std::cout << "   outp.get_delay()               " << outp.get_delay() << std::endl;
    std::cout << "   outp.get_rate()                " << outp.get_rate() << std::endl;
    std::cout << "   outp.get_time()                " << outp.get_time() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.get_time(" << i << ")               " << outp.get_time(i) << std::endl;
    }

    std::cout << "   outp.get_timestep()            " << outp.get_timestep() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.get_timestep(" << i << ")           " << outp.get_timestep(i) << std::endl;
    }

    std::cout << "   outp.get_last_timestep()       " << outp.get_last_timestep() << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.get_last_timestep(" << i << ")      " << outp.get_last_timestep(i) << std::endl;
    }

    std::cout << "   outp.get_max_timestep()        ";
    if (outp.get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << outp.get_max_timestep() << std::endl;

    std::cout << "   outp.is_timestep_changed()     " << (outp.is_timestep_changed() ? "yes" : "no") << std::endl;
    for (unsigned long i = 0; i < outp.get_rate(); i++)
    {
      std::cout << "   outp.is_timestep_changed(" << i << ")    " << (outp.is_timestep_changed(i) ? "yes" : "no") << std::endl;
    }

    std::cout << "   outp.is_rate_changed()         " << (outp.is_rate_changed() ? "yes" : "no") << std::endl;
    std::cout << "   outp.is_delay_changed()        " << (outp.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    outp.set_rate(3);
    outp.set_delay(5);
    sstr.redirect_cout(this->get_time(), 01);

    std::cout << "set_attributes() mod_a" << std::endl;

    set_timestep(1.0, sc_core::SC_MS);
    accept_attribute_changes();

    std::cout << "      outp.set_rate(3);" << std::endl;
    std::cout << "      outp.set_delay(5);" << std::endl;
    std::cout << "      set_timestep(1.0,sc_core::SC_MS);" << std::endl;
    std::cout << "      accept_attribute_changes();\n" << std::endl;

    sstr.restore_cout();
  }

  void initialize()
  {
    sstr.redirect_cout(this->get_time(), 11);

    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " initialize() mod_a" << std::endl;
    print_status();

    sstr.restore_cout();

    out_cnt = 0;

    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); // for the first timestep the propagated value is returned
    last_port_time = outp.get_time() - outp.get_timestep();
    last_port_timestep = outp.get_timestep();

    for (unsigned long i = 0; i < outp.get_delay(); ++i)
    {
      outp.initialize(out_cnt, i);
      out_ref.push_back(out_cnt);
      out_cnt++;
    }
  }

  void reinitialize()
  {

    sstr.redirect_cout(this->get_time(), 21);

    std::cout << "--------------------------------------------" << std::endl;
    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " reinitialize() mod_a" << std::endl;
    print_status();

    sstr.restore_cout();

    if (reinit)
    {
      for (unsigned long i = 0; i < g_ignore; i++)
      {
        outp.initialize(0, i);
        long old_delay = outp.get_delay() - g_ignore;
        out_ref[out_ref.size() - 1 - old_delay-i] = 0;
      }
      reinit = false;
    }
  }

  void processing()
  {
    sstr.redirect_cout(this->get_time(), 31);

    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " processing() mod_a" << std::endl;
    print_status();

    sstr.restore_cout();

    for (unsigned long i = 0; i < outp.get_rate(); ++i)
    {
      outp.write(long(out_cnt), i);
      out_ref.push_back(out_cnt);
      out_cnt++;
    }

    //// consistency checks ///////////////////////////

    for (unsigned long i = 0; i < outp.get_rate(); ++i)
    {
      if (is_time_not_equal(outp.get_time(i) - last_port_time, outp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << outp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << outp.get_time(i);
        str << " get_timestep: " << outp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_time = outp.get_time(i);

      if (is_time_not_equal(last_port_timestep, outp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " outp.get_last_timestep: " << outp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = outp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time, this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time = this->get_time();

    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {
    cnt++;
    sstr.redirect_cout(this->get_time(), 41);

    std::cout << std::endl << std::endl;
    std::cout << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl;
    std::cout << std::endl;
    std::cout << "--------------------------------------------" << std::endl;
    std::cout << get_time() << " change_attributes() mod_a  cnt: " << cnt << std::endl;

    switch(cnt)
    {
      case 1:
        break;
      case 2:
        // the order has no influence, due consistency check is after
        // change_attribute execution
        does_attribute_changes();
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    does_attribute_changes();" << std::endl;
        break;
      case 3:
        does_no_attribute_changes();
        outp.set_delay(1);  // from 5 -> 1
        change_delay_outref(5, 1);
        g_in_cnt += 4;
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    does_no_attribute_changes();" << std::endl;
        std::cout  << "    outp.set_delay(1);" << std::endl;
        break;
      case 4:
        does_attribute_changes();
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    does_attribute_changes();" << std::endl;
        break;
      case 5:
        request_next_activation(0.5, sc_core::SC_MS);
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    request_next_activation(0.5,sc_core::SC_MS);" << std::endl;
        break;
      case 6:
        does_attribute_changes();
        outp.set_delay(5);  // from 1->5
        change_delay_outref(1, 5);
        reinit = true;
        g_in_cnt -= 4;
        g_ignore = 4;
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    does_attribute_changes();" << std::endl;
        std::cout << "    inp.set_delay(5);"         << std::endl;
        break;
      case 7:
      case 8:
      case 9:
      case 10:
      case 11: break;
      case 12:
        does_attribute_changes();
        outp.set_delay(2);  // from 5-2
        change_delay_outref(5, 2);
        g_in_cnt += 3;
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    does_attribute_changes();" << std::endl;
        std::cout  << "    inp.set_delay(2);" << std::endl;
        break;
      case 13:
        does_attribute_changes();
        outp.set_delay(3);  // from 2->3
        change_delay_outref(2, 3);
        g_in_cnt -= 1;
        g_ignore = 1;
        reinit = true;
        std::cout  << "--------------------------------------------" << std::endl;
        std::cout  << "    does_attribute_changes();" << std::endl;
        std::cout  << "    inp.set_delay(3);" << std::endl;
        break;
      case 14:
      case 15: break;
      case 16:
        does_attribute_changes();
        outp.set_delay(0);  // from 3->0
        change_delay_outref(3, 0);
        g_in_cnt += 3; // remove 3 sample delay
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        std::cout << "    outp.set_delay(0);" << std::endl;
        break;
      case 17:
        does_attribute_changes();
        outp.set_delay(3);  // from 0->3
        change_delay_outref(0, 3);
        g_in_cnt -= 3; // add one sample delay
        g_ignore = 3;
        reinit = true;
        std::cout << "--------------------------------------------" << std::endl;
        std::cout << "    does_attribute_changes();" << std::endl;
        std::cout << "    outp.set_delay(3);" << std::endl;
        break;
      case 18:
      case 19:
      case 20:
      case 21:
      case 22: break;
      default:
        sc_core::sc_stop();
        sstr.restore_cout();
        return;
    }
    this->print_status();
    sstr.restore_cout();
  }

  SCA_CTOR( mod_a ) : outp("outp"), cnt(0), reinit(false), out_cnt(0)
  {}

 private:
  long cnt;
  long out_cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;

  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;

  bool reinit;
};

////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE( mod_b )
{
  sca_tdf::sca_in<long> inp;

  void print_status()
  {
    std::cout << std::setw(50)<< "" << "   are_attribute_changes_allowed: " << (are_attribute_changes_allowed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   is_dynamic:                    " << (is_dynamic() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   are_attributes_changed:        " << (are_attributes_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   is_timestep_changed:           " << (is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   get_timestep:                  " << get_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   get_last_timestep:             " << get_last_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   get_max_timestep:              ";

    if (get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << get_max_timestep() << std::endl;

    std::cout << std::setw(50)<< "" << "   get_time:                      " << get_time() << std::endl;
    std::cout << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_delay()                " << inp.get_delay() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_rate()                 " << inp.get_rate() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_time()                 " << inp.get_time() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_time(1)                " << inp.get_time(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_timestep()             " << inp.get_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_timestep(1)            " << inp.get_timestep(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_last_timestep()        " << inp.get_last_timestep() << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_last_timestep(1)       " << inp.get_last_timestep(1) << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.get_max_timestep()         ";

    if (inp.get_max_timestep() == sca_core::sca_max_time()) std::cout << "sca_core::sca_max_time" << std::endl;
    else                                               std::cout << inp.get_max_timestep() << std::endl;

    std::cout << std::setw(50)<< "" << "   inp.is_timestep_changed()      " << (inp.is_timestep_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.is_timestep_changed(1)     " << (inp.is_timestep_changed(1) ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.is_rate_changed()          " << (inp.is_rate_changed() ? "yes" : "no") << std::endl;
    std::cout << std::setw(50)<< "" << "   inp.is_delay_changed()         " << (inp.is_delay_changed() ? "yes" : "no") << std::endl;
  }

  void set_attributes()
  {
    inp.set_rate(2);

    sstr.redirect_cout(this->get_time(), 02);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << " set_attributes() mod_b" << std::endl;
    std::cout << std::setw(50) << "" << "     inp.set_rate(2);" << std::endl;

    sstr.restore_cout();
  }

  void initialize()
  {
    sstr.redirect_cout(this->get_time(), 12);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " initialize() mod_b" << std::endl;
    print_status();

    sstr.restore_cout();

    last_time_step = this->get_timestep();
    last_time = this->get_time() - this->get_timestep(); // for the first timestep the propagated value is returned
    last_port_time = inp.get_time() - inp.get_timestep();
    last_port_timestep = inp.get_timestep();

    g_in_cnt = -inp.get_delay();
    for (unsigned long i = 0; i < inp.get_delay(); ++i)
    {
      inp.initialize(g_in_cnt, i);
      ++g_in_cnt;
    }
    g_in_cnt = -inp.get_delay();
  }

  void reinitialize()
  {
    sstr.redirect_cout(this->get_time(), 22);

    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " reinitialize() mod_b" << std::endl;
    print_status();

    sstr.restore_cout();
  }

  void processing()
  {
    sstr.redirect_cout(this->get_time(), 32);


    std::cout << std::setw(50) << "" << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50) << "" << get_time() << " processing() mod_b" << std::endl;
    print_status();

    sstr.restore_cout();
    for (unsigned long i = 0; i < inp.get_rate(); ++i, --g_ignore)
    {
      if ((g_ignore <= 0) && (long(inp.read(i)) != g_in_cnt))
      {
        std::ostringstream str;
        str << " read wrong value for sample: " << i << " value: " << inp.read(i)
            << " expected " << g_in_cnt << " cnt: " << cnt << std::endl;
        SC_REPORT_ERROR("test", str.str().c_str());
      }
      ++g_in_cnt;
    }

    for (unsigned long i = 0; i < inp.get_rate(); ++i)
    {
      if (is_time_not_equal(inp.get_time(i) - last_port_time, inp.get_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port timestep expect: " << inp.get_time(i) - last_port_time;
        str << " last_time: " << last_port_time << " current time: " << inp.get_time(i);
        str << " get_timestep: " << inp.get_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());

      }
      last_port_time = inp.get_time(i);

      if (is_time_not_equal(last_port_timestep, inp.get_last_timestep(i)))
      {
        std::ostringstream str;
        str << "Wrong port last timestep expect: " << last_port_timestep;
        str << " outp.get_last_timestep: " << inp.get_last_timestep(i);
        str << " for sample: " << i;
        SC_REPORT_ERROR("Check", str.str().c_str());
      }
      last_port_timestep = inp.get_timestep(i);
    }

    if (is_time_not_equal(this->get_time() - last_time, this->get_timestep()))
    {
      std::ostringstream str;
      str << "Wrong timestep expect: " << this->get_time() - last_time;
      str << " last_time: " << last_time << " current time: " << this->get_time();
      str << " get_timestep: " << this->get_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time = this->get_time();

    // consistency check
    if (is_time_not_equal(last_time_step, this->get_last_timestep()))
    {
      std::ostringstream str;
      str << "Wrong last timestep expect: " << last_time_step;
      str << " get_last_timestep: " << this->get_last_timestep();
      SC_REPORT_ERROR("Check", str.str().c_str());
    }
    last_time_step = this->get_timestep();
  }

  void change_attributes()
  {
    accept_attribute_changes();
    cnt++;
    sstr.redirect_cout(this->get_time(), 42);

    std::cout << std::endl << std::endl;
    std::cout << std::setw(50)  << ""  << "+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++" << std::endl;
    std::cout << std::setw(50)  << ""  << std::endl;
    std::cout << std::setw(50)  << ""  << "--------------------------------------------" << std::endl;
    std::cout << std::setw(50)  << ""  << get_time() << " change_attributes() mod_b  cnt: " << cnt << std::endl;
    print_status();

    sstr.restore_cout();
  }

  SCA_CTOR( mod_b ) : inp("inp"), cnt(0)
  {
    g_ignore = 0;
  }

 private:
  long cnt;

  sca_core::sca_time last_time;
  sca_core::sca_time last_time_step;

  sca_core::sca_time last_port_time;
  sca_core::sca_time last_port_timestep;
};

/////////////////////////////////////////////////////////////////////////////

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sstr.redirect_cout(sc_core::sc_time_stamp(), 0);

  std::cout << "| mod_a outp|3--->2|inp  mod_b |" << std::endl;

  sstr.restore_cout();

  sca_tdf::sca_signal<long> s1;

  mod_a a("a");
    a.outp(s1);

  mod_b b("b");
    b.inp(s1);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("test02.dat");
  sca_util::sca_trace(tf, s1, "s1");

  sc_core::sc_start();

  sstr.print();

  sca_util::sca_close_tabular_trace_file(tf);
  check_trace_file("test02.dat");

  TEST_LABEL_END;

  return 0;
}
