/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example my_tdf2de showing use of converter ports

#include <systemc-ams>
#include "test_utilities.h"

#define DISP(mod, val) \
  std::cout << mod.name() << ": " << val << " at " << sc_core::sc_time_stamp() << std::endl

SCA_TDF_MODULE(my_source)
{
  sca_tdf::sca_out<double> out;

  SCA_CTOR(my_source) : out("out")
  {
    cnt = 0;
    s = true;
  }

  void processing()
  {
    if ( (cnt%4) == 0 ) s = !s;

    double val = s ? 0.0 : 2.1;

    out.write(val);
    DISP(out, val);
    cnt++;

    if (cnt > 100) cnt = 0; // reset if value becomes too high
  }

 private:
  int cnt;
  bool s;
};

SCA_TDF_MODULE(my_tdf2de)
{
  sca_tdf::sca_in<double> in;            // TDF input port
  sca_tdf::sca_de::sca_out<double> outp; // TDF output converter port

  SCA_CTOR(my_tdf2de) : in("in"), outp("outp") {}

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_MS);
  }

  void processing()
  {
    double val = in.read();
    outp.write( val );
    DISP(outp, val);
  }
};

SC_MODULE(my_de_sink)
{
  sc_core::sc_in<double> in;

  SC_CTOR(my_de_sink) : in("in")
  {
    SC_METHOD(justread);
    sensitive << in;
  }

  void justread()
  {
    DISP(in, in.read());
  }
};


int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> tdf_sig;
  sc_core::sc_signal<double> de_sig1;
  sc_core::sc_buffer<double> de_sig2;

  my_source src("src");
    src.out(tdf_sig);

  my_tdf2de tdf2de1("tdf2de1");
    tdf2de1.in(tdf_sig);
    tdf2de1.outp(de_sig1);

  my_tdf2de tdf2de2("tdf2de2");
    tdf2de2.in(tdf_sig);
    tdf2de2.outp(de_sig2);

  my_de_sink sink1("sink1");
    sink1.in(de_sig1);

  my_de_sink sink2("sink2");
    sink2.in(de_sig2);

  sc_core::sc_start(12, sc_core::SC_MS);

  TEST_LABEL_END;

  return 0;
}
