/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example small-signal frequency-domain

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> out; // output port

  sin_src( sc_core::sc_module_name nm, double ampl_= 1.0, double freq_ = 1.0e3,
           sca_core::sca_time Tm_ = sca_core::sca_time(0.125, sc_core::SC_MS) )
  : out("out"), ampl(ampl_), freq(freq_), Tm(Tm_)
  {}

  void set_attributes()
  {
    out.set_timestep(Tm);
  }

  void processing()
  {
    double t = get_time().to_seconds(); // actual time
    out.write(ampl * std::sin( 2.0 * M_PI * freq * t ) ) ;
  }

  void ac_processing()
  {
    sca_ac_analysis::sca_ac(out) = ampl;
  }

 private:
  double ampl; // amplitude
  double freq; // frequency
  sca_core::sca_time Tm; // module time step
};

SCA_TDF_MODULE(flat_response)
{
  // port declarations
  sca_tdf::sca_in<double>  in;
  sca_tdf::sca_out<double> out;

  SCA_CTOR(flat_response) : in("in"), out("out") {}

  void processing()
  {
    out.write( in.read() );
  }

  void ac_processing()
  {
    double h = 1.0;
    sca_util::sca_complex res;

    res = h * sca_ac_analysis::sca_ac(in);
    sca_ac_analysis::sca_ac(out) = res;
  }
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;

  sin_src src("src");
    src.out(sig1);

  flat_response flat("flat");
    flat.in(sig1);
    flat.out(sig2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("flat_response_td");

  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");

  sc_core::sc_start(10, sc_core::SC_MS);

  tf->reopen("flat_response_ac");

  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_REAL_IMAG));

  sca_ac_analysis::sca_ac_start(10.0, 100.0e3, 200, sca_ac_analysis::SCA_LOG);

  sca_util::sca_close_tabular_trace_file(tf);

  std::cout << "Check time domain results..." << std::endl;
  test_util::check_results("flat_response_td", 2, "flat_response");

  std::cout << "Check frequency domain results..." << std::endl;
  test_util::check_results("flat_response_ac", 4, "flat_response");

  TEST_LABEL_END;

  return 0;
}
