/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example of sc_main with combined ac descriptions
 
#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(tdfmod)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;
  
  SCA_CTOR(tdfmod) : in("in"), out("out") {}
  
  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_MS);
  }
  
  void processing() {}

  void ac_processing() {}
};

SC_MODULE(tdf_model)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  tdfmod mod_1;
  tdfmod mod_2;

  SC_CTOR(tdf_model) : in("in"), out("out"), mod_1("mod_1"), mod_2("mod_2")
  {
    mod_1.in(in);
    mod_1.out(sig);

    mod_2.in(sig);
    mod_2.out(out);
  }    

 private:
  sca_tdf::sca_signal<double> sig;
};

SC_MODULE(lsf_model)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  sca_lsf::sca_sub sub1;
  sca_lsf::sca_dot dot1;
  sca_lsf::sca_tdf::sca_source tdf2lsf;
  sca_lsf::sca_tdf::sca_sink lsf2tdf;

  SC_CTOR(lsf_model)
  : in("in"), out("out"), sub1("sub1"), dot1("dot1"), 
    tdf2lsf("tdf2lsf"), lsf2tdf("lsf2tdf"),
    sig1("sig1"), sig2("sig2"), sig3("sig3")
  {
    tdf2lsf.inp(in);
    tdf2lsf.y(sig1);

    sub1.x1(sig1);
    sub1.x2(sig3);
    sub1.y(sig2);

    dot1.x(sig2);
    dot1.y(sig3);

    lsf2tdf.x(sig2);
    lsf2tdf.outp(out);
  }
 private:
  sca_lsf::sca_signal sig1, sig2, sig3;
};

SC_MODULE(eln_model)
{
  sca_eln::sca_terminal p;
  sca_tdf::sca_out<double> out;

  sca_eln::sca_tdf::sca_vsink vout;
  sca_eln::sca_r r1;
  sca_eln::sca_c c1;

  eln_model( sc_core::sc_module_name )
  : p("p"), out("out"), vout("vout"), r1("r1", 1.0e3), c1("c1",
  1.0e-6), n1("n1"), gnd("gnd")
  {
    r1.p(p);
    r1.n(n1);

    c1.p(n1);
    c1.n(gnd);

    vout.p(n1);
    vout.n(gnd);
    vout.outp(out);
  }
 private:
  sca_eln::sca_node n1;
  sca_eln::sca_node_ref gnd;
};

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_eln::sca_node net1;
  sca_eln::sca_node_ref gnd;
  sca_tdf::sca_signal<double> sig1, sig2, sig3;

  // ...  source and sink

  sca_eln::sca_vsource vin("vin");
    vin.p(net1);
    vin.n(gnd);
    vin.ac_amplitude = 1.0;
    vin.amplitude = 0.0;


  eln_model a("a");
    a.p(net1);
    a.out(sig1);

  lsf_model b("b");
    b.in(sig1);
    b.out(sig2);

  tdf_model c("c");
    c.in(sig2);
    c.out(sig3);

  // tracing
  sca_util::sca_trace_file* tf =
  sca_util::sca_create_tabular_trace_file("trace.dat");

  sca_util::sca_trace(tf, net1, "net1");
  sca_util::sca_trace(tf, sig1, "sig1");
  sca_util::sca_trace(tf, sig2, "sig2");
  sca_util::sca_trace(tf, sig3, "sig3");

  // start time-domain simulation
  sc_core::sc_start(10, sc_core::SC_MS);

  tf->reopen("ac_trace");
  tf->set_mode(sca_util::sca_ac_format(sca_util::SCA_AC_MAG_RAD));
 
  // start frequency-domain simulation
  sca_ac_analysis::sca_ac_start(1.0e3, 100.0e4, 4, sca_ac_analysis::SCA_LOG);

  tf->reopen("ac_noise_trace");
  tf->set_mode(sca_util::sca_noise_format(sca_util::SCA_NOISE_ALL));

  // start frequency-domain noise simulation
  sca_ac_analysis::sca_ac_noise_start(1.0e3, 100.0e4, 4, sca_ac_analysis::SCA_LOG);

  sca_util::sca_close_tabular_trace_file(tf);

  std::cout << "Check time domain results..." << std::endl;
  test_util::check_results("ac_trace", 2, "combined_ac");

  std::cout << "Check frequency domain results..." << std::endl;
  test_util::check_results("ac_noise_trace", 4, "combined_ac");

  TEST_LABEL_END;

  return 0;
};
