/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example of sc_main for time domain simulation - with options
 
#include <systemc-ams>
#include "test_utilities.h"
#include "my_source.h"
#include "my_control.h"
#include "my_dut.h"
#include "my_sink.h"

int sc_main(int argc, char* argv[])
{
  // User's guide example sect 6.1.1

  sc_core::sc_set_time_resolution(1.0, sc_core::SC_FS);

  TEST_LABEL_START; // added for regression

  sca_tdf::sca_signal<double> sig1, sig2;
  sc_core::sc_signal<bool> sc_sig;

  my_source i_my_source("i_my_source");
    i_my_source.out(sig1);

  my_control i_my_ctrl("i_my_ctrl");
    i_my_ctrl.out(sc_sig);

  my_dut i_my_dut("i_my_dut");
    i_my_dut.in(sig1);
    i_my_dut.ctrl(sc_sig);
    i_my_dut.out(sig2);

  my_sink i_my_sink("i_my_sink");
    i_my_sink.in(sig2);

  sc_core::sc_start(10.0, sc_core::SC_MS); 

  sc_core::sc_time sim_time(10.0, sc_core::SC_MS);
  sc_core::sc_start(sim_time); 

  // sc_core::sc_start(); // runs forever?...

  // User's guide example sect 6.1.2

  // frequency-domain simulations from 1kHz to 10kHz with 100 points on a linear scale:
  sca_ac_analysis::sca_ac_start(1.0e3, 10.0e3, 100, sca_ac_analysis::SCA_LIN);
  sca_ac_analysis::sca_ac_noise_start(1.0e3, 10.0e3, 100, sca_ac_analysis::SCA_LIN);

  // frequency-domain simulations from 1Hz to 1MHz with 1001 points on a logarithmic scale:
  sca_ac_analysis::sca_ac_start(1.0, 1.0e6, 1001, sca_ac_analysis::SCA_LOG);
  sca_ac_analysis::sca_ac_noise_start(1.0, 1.0e6, 1001, sca_ac_analysis::SCA_LOG);
  
  // User's guide example sect 6.2.1 - VCD

  // open trace file in VCD format
  sca_util::sca_trace_file* atf = sca_util::sca_create_vcd_trace_file( "my_trace.vcd" );
  
  // ... (UG, implemented some tracing below)
  sca_util::sca_trace(atf, sig1, "sig1");
  sc_core::sc_start(sc_core::sc_time(10.0, sc_core::SC_MS)); 
    
  // close the trace file
  sca_util::sca_close_vcd_trace_file( atf );


  // User's guide example sect 6.2.1 - tabular
  
  // open trace file in tabular format
  sca_util::sca_trace_file* atf1 = sca_util::sca_create_tabular_trace_file( "my_trace.dat" );

  // ...(UG, implemented some tracing below)
  sca_util::sca_trace(atf1, sig1, "sig1");
  sc_core::sc_start(sc_core::sc_time(10.0, sc_core::SC_MS)); 
    
  // close the trace file
  sca_util::sca_close_tabular_trace_file( atf1 );

  TEST_LABEL_END; // added for regression

  return 0;
}
