/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example for channel incl noise

#include <systemc-ams>
#include "test_utilities.h"

// the gauss_rand() function returns a gaussian distributed
// random number with variance "variance", centered around 0, using the Marsaglia polar method

#include <cstdlib> // for std::rand
#include <cmath>   // for std::sqrt and std::log

double gauss_rand(double variance)
{   
  double rnd1, rnd2, Q, Q1, Q2; 
     
  do
  { 
    rnd1 = ((double)std::rand()) / ((double)RAND_MAX) ;
    rnd2 = ((double)std::rand()) / ((double)RAND_MAX) ;
   
    Q1 = 2.0 * rnd1 - 1.0 ;          
    Q2 = 2.0 * rnd2 - 1.0 ;
    Q = Q1 * Q1 + Q2 * Q2 ;
     
  } while (Q > 1.0) ;
  
  return ( std::sqrt(variance) * ( std::sqrt( - 2.0 * std::log(Q) / Q) * Q1) );
}  
            
SCA_TDF_MODULE(air_channel_with_noise)  
{ 
  sca_tdf::sca_in<double>  in; 
  sca_tdf::sca_out<double> out; 

  void processing()  
  {  
    out.write( in.read() * attenuation + gauss_rand(variance) ); 
  }
  
  air_channel_with_noise( sc_core::sc_module_name nm, 
                          double attenuation_,
                          double variance_ )
  : in("in"), out("out"), attenuation(attenuation_), variance(variance_) {}
  
 private:
  double attenuation;
  double variance;
};

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> out; // output port

  sin_src( sc_core::sc_module_name nm, double ampl_= 1.0, double freq_ = 1.0e3,
           sca_core::sca_time Tm_ = sca_core::sca_time(0.125, sc_core::SC_MS) )
  : out("out"), ampl(ampl_), freq(freq_), Tm(Tm_)
  {}

  void set_attributes()
  {
    out.set_timestep(Tm);
  }

  void processing()
  {
    double t = get_time().to_seconds(); // actual time
    out.write(ampl * std::sin( 2.0 * M_PI * freq * t ) ) ;
  }

 private:
  double ampl; // amplitude
  double freq; // frequency
  sca_core::sca_time Tm; // module time step
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;

  sin_src src("src");
    src.out(sig1);  

  air_channel_with_noise ch("ch", 100, 1);
    ch.in(sig1);
    ch.out(sig2);
    
  sc_core::sc_start(10, sc_core::SC_MS);
  
  TEST_LABEL_END;
    
  return 0;
}
