/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide amplifier example

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> out; // output port

  sin_src( sc_core::sc_module_name nm, double ampl_= 1.0, double freq_ = 1.0e3,
           sca_core::sca_time Tm_ = sca_core::sca_time(0.125, sc_core::SC_MS) )
  : out("out"), ampl(ampl_), freq(freq_), Tm(Tm_)
  {}

  void set_attributes()
  {
    out.set_timestep(Tm);
  }

  void processing()
  {
    double t = get_time().to_seconds(); // actual time
    out.write(ampl * std::sin( 2.0 * M_PI * freq * t ) ) ;
  }

 private:
  double ampl; // amplitude
  double freq; // frequency
  sca_core::sca_time Tm; // module time step
};

SCA_TDF_MODULE(amplifier)
{
  sca_tdf::sca_in<double>  in;
  sca_tdf::sca_out<double> out;

  amplifier( sc_core::sc_module_name, double gain_     = 100.0,
                                      double dom_pole_ = 5.0e8,
                                      double limit_    = 5.0 )
  : in("in"), out("out"), gain(gain_), dom_pole(dom_pole_), limit(limit_) {}

  void initialize()
  {
    // no zeros defined
    poles(0) = sca_util::sca_complex( -2.0 * M_PI * dom_pole, 0.0 );
    k = gain * 2.0 * M_PI * dom_pole;
  }

  void processing()
  {
    // time-domain implementation of amplifier behavior as function of frequency
    double internal = ltf_zp( zeros, poles, state, in.read(), k );

    // limiting
    if (internal > limit) internal = limit;
    else if (internal < -limit) internal = -limit;

    out.write(internal);
  }

 private:
  double gain;     // DC gain
  double dom_pole; // 3dB cutoff frequency in Hz
  double limit;    // limiter value
  double k;        // filter gain
  sca_tdf::sca_ltf_zp ltf_zp; // Laplace transfer function
  sca_util::sca_vector<sca_util::sca_complex> poles, zeros; // poles and zeros as complex values
  sca_util::sca_vector<double> state; // state vector
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> sig1, sig2;

  sin_src src("src");
    src.out(sig1);

  amplifier amp("a1");
    amp.in(sig1);
    amp.out(sig2);

  sc_core::sc_start(10, sc_core::SC_MS);

  TEST_LABEL_END;

  return 0;
}

