/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// user's guide example eln_in_tdf

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> out; // output port

  sin_src( sc_core::sc_module_name nm, double ampl_= 1.0, double freq_ = 1.0e3,
           sca_core::sca_time Tm_ = sca_core::sca_time(0.125, sc_core::SC_MS) )
  : out("out"), ampl(ampl_), freq(freq_), Tm(Tm_)
  {}

  void set_attributes()
  {
    out.set_timestep(Tm);
  }

  void processing()
  {
    double t = get_time().to_seconds(); // actual time
    out.write(ampl * std::sin( 2.0 * M_PI * freq * t ) ) ;
  }

 private:
  double ampl; // amplitude
  double freq; // frequency
  sca_core::sca_time Tm; // module time step
};

SC_MODULE(eln_in_tdf)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  sca_eln::sca_tdf::sca_vsource vin;
  sca_eln::sca_tdf::sca_vsink vout;
  sca_eln::sca_r r;
  sca_eln::sca_c c;

  eln_in_tdf( sc_core::sc_module_name, double r_val, double c_val )
  : in("in"), out("out"), vin("vin"), vout("vout"), r("r", r_val), c("c",
  c_val), n1("n1"), n2("n2"), gnd("gnd")
  {
    vin.inp(in);
    vin.p(n1);
    vin.n(gnd);

    r.p(n1);
    r.n(n2);

    c.p(n2);
    c.n(gnd);

    vout.p(n2);
    vout.n(gnd);
    vout.outp(out);
  }
 private:
  sca_eln::sca_node n1, n2;
  sca_eln::sca_node_ref gnd;
};

int sc_main( int, char*[] )
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> s1, s2;

  sin_src src("src");
  src.out(s1);

  eln_in_tdf elnemb("elnemb", 1.0e3, 1.0e-6);
    elnemb.in(s1);
    elnemb.out(s2);

  sca_util::sca_trace_file* tf = sca_util::sca_create_tabular_trace_file("eln_in_tdf");

  sca_util::sca_trace(tf, s1, "s1");
  sca_util::sca_trace(tf, s2, "s2");
  
  sc_core::sc_start(5, sc_core::SC_MS);
  
  sca_util::sca_close_tabular_trace_file(tf);

  test_util::check_results("eln_in_tdf", 2, 1e-14);

  TEST_LABEL_END;
    
  return 0;
}
