/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test_tracing.cpp --

#include <string>
#include <systemc-ams>
#include "test_utilities.h"

// Simple DE source switching constantly between two values of the
// specified data type.
template<typename T>
class de_bistate_source : public sc_core::sc_module
{
 public:
  sc_core::sc_out<T> out;

  void write_thread()
  {
    for (;;)
    {
      out.write(VAL_1_);
      wait(0.5 * T_PERIOD_);
      out.write(VAL_2_);
      wait(0.5 * T_PERIOD_);
    }
  }

  SC_HAS_PROCESS(de_bistate_source);

  de_bistate_source(sc_core::sc_module_name name,
                    const T& val_1, const T& val_2,
                    const sc_core::sc_time t_period = sc_core::sc_time(10.0, sc_core::SC_NS))
  : sc_core::sc_module(name), out("out"),
    VAL_1_(val_1), VAL_2_(val_2),
    T_PERIOD_(t_period)
  {
    SC_THREAD(write_thread);
  }

 private:
  const T VAL_1_, VAL_2_;            // Output values.
  const sc_core::sc_time T_PERIOD_;  // Time period between sending values.
};


// Simple DE sink reading all values of the specified data type.
template<typename T>
class de_sink : public sc_core::sc_module
{
 public:
  sc_core::sc_in<T> in;

  void read_method()
  {
    val_ = in.read();
  }

  SC_HAS_PROCESS(de_sink);
  
  de_sink(sc_core::sc_module_name name)
  : sc_core::sc_module(name), in("in")
  {
    SC_METHOD(read_method);
    sensitive << in;
  }

 private:
  T val_; // Last read value.
};


// Module to test the tracing of DE input ports, output ports, and
// signal of the specified data type.
template<typename T>
class de_trace_test : public sc_core::sc_module
{
 public:
  de_bistate_source<T> src;
  de_sink<T> sink;

  sc_core::sc_signal<T> sig;

  de_trace_test(sc_core::sc_module_name name,
                const T& val_1, const T& val_2,
                const sc_core::sc_time t_period = sc_core::sc_time(10.0, sc_core::SC_NS))
  : sc_core::sc_module(name),
    src("src", val_1, val_2, t_period),
    sink("sink"), sig("sig")
  {
    // Connectivity.
    src.out(sig);
    sink.in(sig);
  }

  void trace(sc_core::sc_trace_file* tf)
  {
    using namespace sc_core;
    sc_trace(tf, src.out, src.out.name());
    sc_trace(tf, sink.in, sink.in.name());
    sc_trace(tf, sig, sig.name());
  }

  void trace(sca_util::sca_trace_file* tf)
  {
    using namespace sca_util;
    sca_trace(tf, src.out, src.out.name());
    sca_trace(tf, sink.in, sink.in.name());
    sca_trace(tf, sig, sig.name());
  }

};

template<typename T>
class tdf_bistate_source : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_out<T> out;

  tdf_bistate_source(sc_core::sc_module_name name,
                     const T& val_1, const T& val_2,
                     const sc_core::sc_time t_period = sc_core::sc_time(10.0, sc_core::SC_NS))
  : sca_tdf::sca_module(name), out("out"),
    VAL_1_(val_1), VAL_2_(val_2),
    T_PERIOD_(t_period)
  {}

  virtual void set_attributes()
  {
    out.set_rate(2);
    out.set_delay(0);
    out.set_timestep(0.5 * T_PERIOD_);
  }

  virtual void processing()
  {
    out.write(VAL_1_, 0);
    out.write(VAL_2_, 1);
  }

 private:
  const T VAL_1_, VAL_2_;            // Output values.
  const sc_core::sc_time T_PERIOD_;  // Time period between sending values.
};


template<typename T>
class tdf_sink : public sca_tdf::sca_module
{
 public:
  sca_tdf::sca_in<T> in;

  tdf_sink(sc_core::sc_module_name name)
  : sca_tdf::sca_module(name), in("in")
  {}

  virtual void set_attributes()
  {
    in.set_rate(1);
    in.set_delay(0);
  }

  virtual void processing()
  {
    val = in.read();
  }

 private:
  T val; // Last read value.
};


// Module to test the tracing of DE input ports, output ports, and
// signal of the specified data type.
template<typename T>
class tdf_trace_test : public sc_core::sc_module
{
 public:
  tdf_bistate_source<T> src;
  tdf_sink<T> sink;

  sca_tdf::sca_signal<T> sig;

  tdf_trace_test(sc_core::sc_module_name name,
                 const T& val_1, const T& val_2,
                 const sc_core::sc_time t_period = sc_core::sc_time(10.0, sc_core::SC_NS))
  : sc_core::sc_module(name),
    src("src", val_1, val_2, t_period),
    sink("sink"), sig("sig")
  {
    // Connectivity.
    src.out(sig);
    sink.in(sig);
  }

  // // Support for tracing to a SystemC trace file seems to have been
  // // removed in the SystemC AMS extensions LRM 1.0
  // void trace(sc_core::sc_trace_file* tf) {
  //   using namespace sc_core;
  //   sc_trace(tf, src.out, src.out.name());
  //   sc_trace(tf, sink.in, sink.in.name());
  //   sc_trace(tf, sig, sig.name());
  // }

  void trace(sca_util::sca_trace_file* tf)
  {
    using namespace sca_util;
    sca_trace(tf, src.out, src.out.name());
    sca_trace(tf, sink.in, sink.in.name());
    sca_trace(tf, sig, sig.name());
  }

};

int sc_main(int argc, char* argv[])
{
  using namespace std;
  using namespace sc_core;
  using namespace sca_util;

  const std::string TEST_NAME("test_tracing");

  const sc_core::sc_time TIME_UNIT(1.0, sc_core::SC_NS);
  const sc_core::sc_time T_PERIOD(10.0, sc_core::SC_NS);
  const sc_core::sc_time T_STOP(80.0, sc_core::SC_NS);
  
  // Open trace files for all three tracing mechanisms.
  sc_trace_file* vcd_tf = sc_create_vcd_trace_file((TEST_NAME + "_sc_vcd").c_str());
  vcd_tf->set_time_unit(TIME_UNIT.to_double(), sc_core::SC_PS);
  sca_trace_file* tab_ams_tf = sca_create_tabular_trace_file((TEST_NAME + "_sca_tab.dat").c_str());
  sca_trace_file* vcd_ams_tf = sca_create_vcd_trace_file((TEST_NAME + "_sca_vcd.vcd").c_str());

  sc_core::sc_clock clk("clk", T_PERIOD);
  sc_trace(vcd_tf, clk, clk.name());
  sca_trace(tab_ams_tf, clk, clk.name());
  sca_trace(vcd_ams_tf, clk, clk.name());

  // Instantiate trace test modules.
  de_trace_test<bool> de_trace_bool("de_trace_bool", true, false, T_PERIOD);
  de_trace_bool.trace(vcd_tf);
  de_trace_bool.trace(tab_ams_tf);
  de_trace_bool.trace(vcd_ams_tf);

  tdf_trace_test<bool> tdf_trace_bool("tdf_trace_bool", true, false, T_PERIOD);
  tdf_trace_bool.trace(tab_ams_tf);
  tdf_trace_bool.trace(vcd_ams_tf);

  de_trace_test<int> de_trace_int("de_trace_int", 1, -1, T_PERIOD);
  de_trace_int.trace(vcd_tf);
  de_trace_int.trace(tab_ams_tf);
  de_trace_int.trace(vcd_ams_tf);

  tdf_trace_test<int> tdf_trace_int("tdf_trace_int", 1, -1, T_PERIOD);
  tdf_trace_int.trace(tab_ams_tf);
  tdf_trace_int.trace(vcd_ams_tf);

  de_trace_test<double> de_trace_double("de_trace_double", 1.8, 0.0, T_PERIOD);
  de_trace_double.trace(vcd_tf);
  de_trace_double.trace(tab_ams_tf);
  de_trace_double.trace(vcd_ams_tf);

  tdf_trace_test<double> tdf_trace_double("tdf_trace_double", 1.8, 0.0, T_PERIOD);
  tdf_trace_double.trace(tab_ams_tf);
  tdf_trace_double.trace(vcd_ams_tf);

  TEST_LABEL_START;

  // Start simulation.
  sc_start(T_STOP);

  // Close trace files.
  sc_close_vcd_trace_file(vcd_tf);
  sca_close_tabular_trace_file(tab_ams_tf);
  sca_close_vcd_trace_file(vcd_ams_tf);

  TEST_LABEL_END;

  return 0;
}
