/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  Example demonstrating the basic language constructs of the SystemC AMS
//  extensions as defined in the Draft Standard SystemC AMS Extensions
//  Language Reference Manual, Draft #1, December 3 2008

#include <systemc-ams>
#include <cmath>
#include <cstdlib>
#include "test_utilities.h"

test_util::sorted_stream sstr;

#ifndef M_PI
const double M_PI = std::acos(-1);
#endif // M_PI

/*****************************************************************************
  file: lp_filter_lsf.h
 *****************************************************************************/

SC_MODULE(lp_filter_lsf)  // Hierarchical models use SC_MODULE class
{
  sca_tdf::sca_in<double> in;         // Input/outputs in data flow MoC
  sca_tdf::sca_out<double> out;       //

  sca_lsf::sca_sub* sub1;             // Subtractor
  sca_lsf::sca_dot* dot1;             // Differentiator
  sca_lsf::sca_tdf_source* tdf2lsf1;  // TDF -> LSF converter
  sca_lsf::sca_tdf_sink*   lsf2tdf1;  // LSF -> TDF converter

  sca_lsf::sca_signal in_lsf, sig, out_lsf;  // LSF signals

  // Constructor with parameters
  lp_filter_lsf(sc_core::sc_module_name, double fc_ = 1.0e3)
  : in("in"), out("out"), in_lsf("in_lsf"), sig("sig"), out_lsf("out_lsf"), fc(fc_)
  { architecture(); }

  ~lp_filter_lsf();
  
 private:
  double fc;
  void architecture();
};

/*****************************************************************************
    file: lp_filter_lsf.cpp
 *****************************************************************************/

void lp_filter_lsf::architecture()
{

  tdf2lsf1 = new sca_lsf::sca_tdf_source("tdf2lsf1");  // TDF->LSF converter
    tdf2lsf1->inp(in);    // Port to signal binding like in SystemC
    tdf2lsf1->y(in_lsf);

  sub1 = new sca_lsf::sca_sub("sub1");
    sub1->x1(in_lsf);
    sub1->x2(sig);
    sub1->y(out_lsf);

  dot1 = new sca_lsf::sca_dot("dot1", 1.0 / (2.0 * M_PI * fc));  // M_PI=3.1415...
    dot1->x(out_lsf);
    dot1->y(sig);

  lsf2tdf1 = new sca_lsf::sca_tdf_sink("lsf2tdf1");     // LSF->TDF converter
    lsf2tdf1->x(out_lsf);
    lsf2tdf1->outp(out);
}

lp_filter_lsf::~lp_filter_lsf()
{
  delete tdf2lsf1, sub1, dot1, lsf2tdf1;
}

/*****************************************************************************
    file: mixer.h
 *****************************************************************************/

SCA_TDF_MODULE(mixer) // TDF primitive module definition
{
 public:

  sca_tdf::sca_in<double>  rf_in, lo_in;  // TDF in ports
  sca_tdf::sca_out<double> if_out;        // TDF out ports

  void set_attributes();

  void processing();

  SCA_CTOR(mixer) : rf_in("rf_in"), lo_in("lo_in"), if_out("if_out") {}

};

/*****************************************************************************
    file: mixer.cpp
 *****************************************************************************/

void mixer::set_attributes()
{
  // set_timestep(1.0, sc_core::SC_US);  // time between activations
}

void mixer::processing()  // executed at each activation
{
  if_out.write(rf_in.read() * lo_in.read());
}

/*****************************************************************************
    file: lp_filter_tdf.h
 *****************************************************************************/

SCA_TDF_MODULE(lp_filter_tdf)
{
  sca_tdf::sca_in<double>  in;
  sca_tdf::sca_out<double> out;
  sca_tdf::sc_in<double>   gain;  // converter port for SystemC input

  sca_tdf::sca_ltf_nd ltf;                // computes transfer function
  sca_util::sca_vector<double> num, den;  // coefficients

  void initialize();
  void processing();

  SCA_CTOR(lp_filter_tdf) : in("in"), out("out"), gain("gain") {}
};


/*****************************************************************************
    file: lp_filter_tdf.cpp
 *****************************************************************************/

void lp_filter_tdf::initialize()
{
  num(0) = 1.0;
  den(0) = 1.0; den(1) = 1.0/(2.0 * M_PI * 10.0e3);  // M_PI=3.1415...
}

void lp_filter_tdf::processing()
{
  out.write(ltf(num, den, in.read()) * gain.read());
}

/*****************************************************************************
    file: agc_ctrl.h
 *****************************************************************************/

SC_MODULE(agc_ctrl)
{

 public:
  sc_core::sc_in<sc_dt::sc_bv<3> > config;
  sc_core::sc_out<double> out;

  SC_CTOR(agc_ctrl);

 private:
  void do_agc_ctrl();
};

/*****************************************************************************
    file: agc_ctrl.cpp
 *****************************************************************************/

agc_ctrl::agc_ctrl( sc_core::sc_module_name )
: config("config"), out("out")
{
  SC_METHOD(do_agc_ctrl);
  sensitive << config;
}

void agc_ctrl::do_agc_ctrl()
{
  out.write(100.0 * static_cast<double>(config.read().to_uint()));
}

/*****************************************************************************
    file: frontend.h
 *****************************************************************************/

SC_MODULE(frontend)  // SC_MODULES used for hierarchical structure
{
  sca_tdf::sca_in<double> rf, loc_osc;  // use TDF ports to connect with
  sca_tdf::sca_out<double> if_out;      // TDF ports/signals in hierarchy
  sc_core::sc_in<sc_dt::sc_bv<3> > ctrl_config;  // SystemC input agc_ctrl
                                                 // configuration
  sca_tdf::sca_signal<double> if_sig;   // TDF internal signal
  sc_core::sc_signal<double> ctrl_gain; // SystemC internal signal

  mixer* mixer1;
  lp_filter_tdf* lpf1;
  agc_ctrl* ctrl1;

  SC_CTOR(frontend) 
  : rf("rf"), loc_osc("loc_osc"), if_out("if_out"), ctrl_config("ctrl_config"),
    if_sig("if_sig"), ctrl_gain("ctrl_gain")
  { architecture(); }

   ~frontend();

 private:
   void architecture();
};

/*****************************************************************************
    file: frontend.cpp
 *****************************************************************************/

void frontend::architecture() {

  mixer1 = new mixer("mixer1");
    mixer1->rf_in(rf);
    mixer1->lo_in(loc_osc);
    mixer1->if_out(if_sig);

  lpf1 = new lp_filter_tdf("lpf1");
    lpf1->in(if_sig);
    lpf1->out(if_out);
    lpf1->gain(ctrl_gain);

  ctrl1 = new agc_ctrl("ctrl1");  // SystemC module
    ctrl1->config(ctrl_config);
    ctrl1->out(ctrl_gain);
}

frontend::~frontend()
{
  delete mixer1, lpf1, ctrl1;
}

/*****************************************************************************
    file: ad_converter.h
 *****************************************************************************/

SCA_TDF_MODULE(ad_converter)  // Very simple AD converter
{
  sca_tdf::sca_in<double> in_tdf;              // TDF port
  sca_tdf::sc_out<sc_dt::sc_int<12> > out_de;  // converter port to DE domain

  void processing();

  SCA_CTOR(ad_converter) : in_tdf("in_tdf"), out_de("out_de") {}
};

/*****************************************************************************
    file: ad_converter.cpp
 *****************************************************************************/

void ad_converter::processing()
{
  out_de.write(static_cast<sc_dt::sc_int<12> >(in_tdf.read()));
}

/*****************************************************************************
    file: lp_filter_eln.h
 *****************************************************************************/

SC_MODULE(lp_filter_eln)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<double> out;

  sca_eln::sca_node in_node, out_node;  // node declarations
  sca_eln::sca_node_ref gnd;            // reference node

  sca_eln::sca_r *r1;                   // resistor
  sca_eln::sca_c *c1;                   // capacitor
  sca_eln::sca_tdf_vsource *v_in;       // converter TDF -> voltage
  sca_eln::sca_tdf_vsink   *v_out;      // converter voltage -> TDF


  SC_CTOR(lp_filter_eln) 
  : in("in"), out("out"), in_node("in_node"), out_node("out_node"), gnd("gnd")
  { architecture(); }

  ~lp_filter_eln();

 private:
   void architecture();

};

/*****************************************************************************
    file: lp_filter_eln.cpp
 *****************************************************************************/

void lp_filter_eln::architecture() {

  v_in = new sca_eln::sca_tdf_vsource("v_in", 1.0); // scale factor 1.0
    v_in->inp(in);     // TDF input
    v_in->p(in_node);  // is converted to voltage
    v_in->n(gnd);

  r1 = new sca_eln::sca_r("r1", 10e3);    // 10 kOhm resistor
    r1->p(in_node);
    r1->n(out_node);

  c1 = new sca_eln::sca_c("c1", 100e-6);  // 100 uF capacitor
    c1->p(out_node);
    c1->n(gnd);

  v_out = new sca_eln::sca_tdf_vsink("v_out", 1.0);  // scale factor 1.0
    v_out->p(out_node);  // filter output as voltage
    v_out->n(gnd);
    v_out->outp(out);    // here converted to TDF signal
}

lp_filter_eln::~lp_filter_eln()
{
  delete v_in, r1, c1, v_out;
}

/*****************************************************************************
    file: bask_demodulator.h
 *****************************************************************************/

SCA_TDF_MODULE(bask_demodulator)
{
  sca_tdf::sca_in<double> in;
  sca_tdf::sca_out<bool> out;  // converter port to SystemC

  void set_attributes();

  void processing();

  SCA_CTOR(bask_demodulator)
  : in("in"), out("out") {}

};

/*****************************************************************************
    file: bask_demodulator.cpp
 *****************************************************************************/

void bask_demodulator::set_attributes()
{
  in.set_rate(20000);                // port in has 20000 samples/timestep
  out.set_timestep(0.1, sc_core::SC_MS);
}

void bask_demodulator::processing()  // maps 20000 samples to 1 symbol
{
  const double THRESHOLD = 10000.0;
  double val = 0.0;
  for (unsigned long i = 0; i < in.get_rate(); i++)
    val += abs(in.read(i));

    if (val>THRESHOLD) out.write(true);
    else out.write(false);
}

/*****************************************************************************
    file: ser2par.h
 *****************************************************************************/

template<int N>
SCA_TDF_MODULE(ser2par)
{
  sca_tdf::sca_in<bool> in;
  sca_tdf::sca_out<sc_dt::sc_bv<N> > out;

  sc_dt::sc_bv<N> intern;

  void set_attributes();

  void processing();

  SCA_CTOR(ser2par)
    : in("in"), out("out"), intern(0)
  {}
};

/*****************************************************************************
    file: ser2par.cpp
 *****************************************************************************/

template<int N>
void ser2par<N>::set_attributes()
{
  in.set_rate(N);
}

template<int N>
void ser2par<N>::processing()
{
  for (int i = (N-1); i >= 0; i--) {
    intern[i] = in.read(i);
  }
  out.write(intern);
}

/*****************************************************************************
    file: par2ser.h
 *****************************************************************************/

template<int N>
SCA_TDF_MODULE(par2ser)
{
  sca_tdf::sca_in<sc_dt::sc_bv<N> > in;
  sca_tdf::sca_out<bool> out;

  sc_dt::sc_bv<N> intern;

  void set_attributes();

  void processing();

  SCA_CTOR(par2ser)
  : in("in"), out("out")
  {}
};

/*****************************************************************************
    file: par2ser.cpp
 *****************************************************************************/

template<int N>
void par2ser<N>::set_attributes()
{
  out.set_rate(N);
}

template<int N>
void par2ser<N>::processing()
{
  intern = in.read();

  for (int i = (N-1); i >= 0; i--) {
    out.write(intern.get_bit(i), i);
  }
}

/*****************************************************************************
    file: ber_tester.h
 *****************************************************************************/

template<int N>
SCA_TDF_MODULE(ber_tester)
{
  sca_tdf::sca_in<sc_dt::sc_bv<N> > data_in;
  sca_tdf::sca_out<sc_dt::sc_bv<N> > data_out;

  sc_dt::sc_uint<N> count;

  void set_attributes();
  void initialize();
  void processing();

  SCA_CTOR(ber_tester)
  : data_in("data_in"), data_out("data_out")
  {}

};

/*****************************************************************************
    file: ber_tester.cpp
 *****************************************************************************/

template<int N>
void ber_tester<N>::set_attributes()
{
  data_in.set_delay(1);
}

template<int N>
void ber_tester<N>::initialize()
{
  count = 1;
}

template<int N>
void ber_tester<N>::processing()
{
  using namespace std;

  data_out.write(count);

  sstr.redirect_cout(this->get_time(), 1);

  std::cout << "sent:     " << sc_dt::sc_bv<N>(count)
            << " - received: " << data_in.read();
  if (data_in.read() != count-1)
  {
    std::cout << " (biterror)";
  }
  std::cout << std::endl;

  sstr.restore_cout();

  count++;
}

/*****************************************************************************
    file: bask_modulator.h
 *****************************************************************************/

SCA_TDF_MODULE(bask_modulator)
{
  sca_tdf::sca_in<bool> in;
  sca_tdf::sca_out<double> out;

  double freq;
  double ampl;
  int rate;

  void set_attributes();

  void processing();

  bask_modulator(sc_core::sc_module_name name,
                 double freq_ = 100.0e6, double ampl_ = 1.0,
                 int rate_ = 20000)
  : in("in"), out("out"), freq(freq_), ampl(ampl_), rate(rate_) {}
};

/*****************************************************************************
    file: bask_modulator.cpp
 *****************************************************************************/

void bask_modulator::set_attributes()
{
  out.set_rate(rate);
}

void bask_modulator::processing()
{
  double carrier;
  for (int i = 0; i < rate; i++)
  {
    carrier = ampl * cos(2.0 * M_PI * freq *
                          (get_time().to_seconds()
                           + out.get_timestep().to_seconds() * i));
    if (in.read())
    {
       out.write(carrier, i);
    } else {
       out.write(0.0, i);
    }
  }
}

/*****************************************************************************
    file: oscillator.h
 *****************************************************************************/

SCA_TDF_MODULE(oscillator)
{
  sca_tdf::sca_out<double> out;

  double freq;
  double ampl;

  void processing();

  oscillator(sc_core::sc_module_name name,
             double freq_ = 100.0e6, double ampl_ = 1.0)
  : out("out"), freq(freq_), ampl(ampl_) {}
};

/*****************************************************************************
    file: oscillator.cpp
 *****************************************************************************/

void oscillator::processing()
{
  double wave;
  wave = ampl * cos(2.0 * M_PI * freq * get_time().to_seconds());
  out.write(wave);
}

/*****************************************************************************
    file: testbench.h
 *****************************************************************************/

// testbench parameters
const sc_core::sc_time WORD_PERIOD = sc_core::sc_time(800.0, sc_core::SC_US);
const int NBITS = 8;               // word length
const int RF_TOKEN_RATE = 20000;   // RF token rate
const double MOD_FREQ = 1.0e6;     // BASK modulation frequency / MHz
const double MOD_AMPL = 0.25;      // BASK modulator output amplitude
const double LOC_OSC_FREQ = 0.8e6; // oscillator frequency / MHz
const double LOC_OSC_AMPL = 1.0;   // oscillator amplitude

SC_MODULE(testbench)
{
  sca_tdf::sca_out<double> rf;
  sca_tdf::sca_out<double> loc_osc;
  sc_core::sc_out<sc_dt::sc_bv<3> > ctrl;
  sca_tdf::sca_in<bool> symbol;

  sca_tdf::sca_signal<sc_dt::sc_bv<NBITS> > snd_word;
  sca_tdf::sca_signal<bool> snd_bit;
  sca_tdf::sca_signal<sc_dt::sc_bv<NBITS> > rcv_word;

  ber_tester<NBITS> *ber_tester1;
  par2ser<NBITS> *par2ser1;
  oscillator *osc1;
  bask_modulator *bask_mod1;
  ser2par<NBITS> *ser2par1;

  // Thread to change the ctrl bits over times, which are controlling
  // the amplification in the BASK receiver frontend
  void ctrl_thread();

  SC_CTOR(testbench)
    : rf("rf"), loc_osc("loc_osc"), ctrl("ctrl"), symbol("symbol"),
      snd_word("snd_word"), snd_bit("snd_bit"), rcv_word("rcv_word")
  {
    architecture();

    SC_THREAD(ctrl_thread);
  }

  ~testbench();

  private:
   void architecture();
};

/*****************************************************************************
    file: testbench.h
 *****************************************************************************/

void testbench::architecture()
{
  ber_tester1 = new ber_tester<NBITS>("ber_tester1");
    ber_tester1->data_out(snd_word);
    ber_tester1->set_timestep(WORD_PERIOD);
    ber_tester1->data_in(rcv_word);

  par2ser1 = new par2ser<NBITS>("par2ser1");
    par2ser1->in(snd_word);
    par2ser1->out(snd_bit);

  bask_mod1 = new bask_modulator("bask_mod1", MOD_FREQ, MOD_AMPL,
                                  RF_TOKEN_RATE);
    bask_mod1->in(snd_bit);
    bask_mod1->out(rf);

  osc1 = new oscillator("osc1", LOC_OSC_FREQ, LOC_OSC_AMPL);
    osc1->out(loc_osc);

  ser2par1 = new ser2par<NBITS>("ser2par1");
    ser2par1->in(symbol);
    ser2par1->out(rcv_word);
}

testbench::~testbench()
{
  delete ber_tester1, par2ser1, bask_mod1, osc1, ser2par1;
}

void testbench::ctrl_thread()
{
  unsigned counter = 4;
  for (;;) {
    ctrl.write(counter);

    sstr.redirect_cout();
    std::cout << "amplitude ctrl = " << counter << std::endl;
    sstr.restore_cout();

    wait((pow(NBITS, 2.0) / 8) * WORD_PERIOD);
    counter = (counter - 1) % 8;
  }
}

/*****************************************************************************
    file: main.cpp
 *****************************************************************************/

int sc_main(int argc, char* argv[])
{
  // added lines below for regression test
  TEST_LABEL_START;
  // end regression test block
  
  sca_tdf::sca_signal<double> rf, loc_osc, d_in;
  sca_tdf::sca_signal<bool> symbol;
  sc_core::sc_signal<sc_dt::sc_bv<3> > ctrl_config;

  frontend fe1("fe1");
    fe1.loc_osc(loc_osc);
    fe1.ctrl_config(ctrl_config);
    fe1.rf(rf);
    fe1.if_out(d_in);

  bask_demodulator bask1("bask1");
    bask1.in(d_in);
    bask1.out(symbol);

  testbench tb1("tb1");
    tb1.rf(rf);
    tb1.loc_osc(loc_osc);
    tb1.ctrl(ctrl_config);
    tb1.symbol(symbol);

  sca_util::sca_trace_file *tfp =
    sca_util::sca_create_vcd_trace_file("systemc_ams_example.vcd");
    sca_util::sca_trace(tfp, rf, "rf");
    sca_util::sca_trace(tfp, loc_osc, "loc_osc");
    sca_util::sca_trace(tfp, fe1.if_sig, "fe1.if_sig");
    sca_util::sca_trace(tfp, ctrl_config, "ctrl_config");
    sca_util::sca_trace(tfp, fe1.ctrl_gain, "fe1.ctrl_gain");
    sca_util::sca_trace(tfp, d_in, "d_in");
    sca_util::sca_trace(tfp, symbol, "symbol");

  sc_core::sc_start(2.0, sc_core::SC_MS);

  sca_util::sca_close_vcd_trace_file(tfp);

  sstr.print();

  // added lines below for regression test
  TEST_LABEL_END;
  // end regression test block
  
  return 0;
}
