/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#include <systemc-ams>
#include "test_utilities.h"

// ---- ELN model -----------------------------------------------------------
class eln_model : public sc_core::sc_module {

public:
  sc_core::sc_in<double> in;    // DE input port
  sc_core::sc_out<double> out;  // DE output port

  // Internal ELN primitives
  sca_eln::sca_de::sca_vsource vsrc1; // DE-ELN converter module
  sca_eln::sca_r R1;                  // LSF resistor primitive
  sca_eln::sca_de::sca_vsink vsink1;  // ELN-DE converter module

  // ELN nodes
  sca_eln::sca_node n1;
  sca_eln::sca_node_ref gnd;

  eln_model(sc_core::sc_module_name nm, double r_val)
  : in("in"), out("out"), vsrc1("vsrc1"), R1("R1", r_val), vsink1("vsink1"),
    n1("n1"), gnd("gnd")
  {
    // Specify connectivity
    vsrc1.inp(in);
    vsrc1.p(n1);
    vsrc1.n(gnd);

    R1.p(n1);
    R1.n(gnd);
    R1.set_timestep(1, sc_core::SC_MS); // Ts required at least in a ELN prim.

    vsink1.p(n1);
    vsink1.n(gnd);
    vsink1.outp(out);
  }

}; // class eln_model


//---- DE source module -------------------------------------------------------
class de_source : public sc_core::sc_module {

public:
  sc_core::sc_out<double> out; // DE output port

  SC_HAS_PROCESS(de_source);

  de_source(sc_core::sc_module_name nm, sc_core::sc_time wt, double st_init = 1)
  : out("out"), wt_(wt), st_(st_init)
  {
    SC_THREAD(source);
    out.initialize(st_init);
  }

  void source()
  {
    while (true)
    {
      wait(wt_);
      out = ++st_;
    }
  }

private:
  sc_core::sc_time wt_; // wait time
  double st_; // status
}; // class de_source


//---- DE sink module -------------------------------------------------------
class de_sink : public sc_core::sc_module {

public:
  sc_core::sc_in<double> in; // DE input port

  SC_HAS_PROCESS(de_sink);

  de_sink(sc_core::sc_module_name nm)
  : in("in")
  {
    SC_METHOD(sink);
    sensitive << in;
  }

  void sink()
  {
    in.read();
  }
}; // class de_sink

// ---- main
int sc_main(int argc, char* argv[])
{

  // Attributes of ELN_modules
  double R_val = 3000;

  sc_core::sc_time t_wait(2.5, sc_core::SC_MS); // wait time
  sc_core::sc_time t_sim(25.0, sc_core::SC_MS); // simulation time

  // DE signals
  sc_core::sc_signal <double> sig_de1;
  sc_core::sc_signal <double> sig_de2;

  // Specify connectivity
  de_source s1("s1", t_wait);
  s1.out(sig_de1);

  eln_model e1("M1", R_val);
  e1.in(sig_de1);
  e1.out(sig_de2);

  de_sink s2("Y");
  s2.in(sig_de2);

  sca_util::sca_trace_file* tfp_vcd;
  tfp_vcd = sca_util::sca_create_vcd_trace_file("test_eln");
  sca_trace(tfp_vcd, sig_de1, "sig_de1");
  sca_trace(tfp_vcd, sig_de2, "sig_de2");

  // Simulation
  try
  {
    sc_core::sc_start(t_sim);
  } catch (const std::exception& e)
  {
    std::cerr << e.what() << std::endl;
  }

  sca_util::sca_close_vcd_trace_file(tfp_vcd);
  sc_core::sc_stop();

  // This line is never reached due to segfault in the post solve
  // statics member function of the linear solver called during
  // end_of_simulation() callback, which execution is triggered by
  // sc_stop().

  TEST_LABEL_START;

  std::cout << "No segfault happened" << std::endl;

  TEST_LABEL_END;

  return sc_core::sc_report_handler::get_count(sc_core::SC_ERROR);
}

// vim: expandtab : tabstop=2 : softtabstop=2 : shiftwidth=2 : textwidth=79
