/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

//  test_close_trace_file_wo_analysis.cpp --
//  problem reported by Sumit Adhikari

#include <systemc-ams>
#include "test_utilities.h"

SCA_TDF_MODULE(tdf_src)
{
  sca_tdf::sca_out<double > out;

  SCA_CTOR(tdf_src)
  : out("out")
  {}

  void set_attributes()
  {
    set_timestep(1.0, sc_core::SC_US);
  }

  void processing()
  {
    double x = std::sin(2.0 * M_PI * 1.0e3 * get_time().to_seconds());
    out.write(x);
  }
};

/////////////////////////////////////////////////

SCA_TDF_MODULE(tdf2de)
{
  sca_tdf::sca_in<double>          inp;
  sca_tdf::sca_de::sca_out<double> de_out;
  sca_tdf::sca_de::sca_in<double>  de_in;

  sca_tdf::sca_de::sca_out<bool>   de_out_bool;
  sca_tdf::sca_de::sca_in<bool>    de_in_bool;

  sca_tdf::sca_de::sca_out<sc_dt::sc_logic> de_out_logic;
  sca_tdf::sca_de::sca_in<sc_dt::sc_logic>  de_in_logic;

  SCA_CTOR(tdf2de)
  : inp("inp"), de_out("de_out"),
    de_in("de_in"), de_out_bool("de_out_bool"),
    de_in_bool("de_in_bool"), de_out_logic("de_out_logic"),
    de_in_logic("de_in_logic")
  {
    bstate = false;
  }

  void set_attributes()
  {
    de_in.set_delay(1);
    de_in_bool.set_delay(1);
    de_in_logic.set_delay(1);

    de_in.set_rate(2);
    de_in_bool.set_rate(2);
    de_in_logic.set_rate(2);
  }

  void processing()
  {
    de_out.write(inp.read());
    de_out_bool.write(bstate);
    bstate = !bstate;
    de_out_logic.write(sc_dt::sc_logic(bstate));
  }

  bool bstate;
};

/////////////////////////////////////////////////

int sc_main(int argc, char* argv[])
{
  TEST_LABEL_START;

  sca_tdf::sca_signal<double> src_out;
  tdf_src U_source("U_source");
  U_source.out(src_out);

  sc_core::sc_signal<double> inf_out;
  sc_core::sc_signal<bool>   inf_out_bool;
  sc_core::sc_signal<sc_dt::sc_logic> inf_out_logic;

  tdf2de U_INTF("U_INTF");
    U_INTF.inp(src_out);
    U_INTF.de_out(inf_out);
    U_INTF.de_in(inf_out);

    U_INTF.de_out_bool(inf_out_bool);
    U_INTF.de_in_bool(inf_out_bool);

    U_INTF.de_out_logic(inf_out_logic);
    U_INTF.de_in_logic(inf_out_logic);

  //  Tracing
  sca_util::sca_trace_file *tf = sca_util::sca_create_vcd_trace_file("tdfde_port_tracing");
  sca_util::sca_trace(tf, inf_out, "inf_out");
  sca_util::sca_trace(tf, U_INTF.de_out, "U_INTF.de_out");
  sca_util::sca_trace(tf, U_INTF.de_in, "U_INTF.de_in");
  sca_util::sca_trace(tf, U_INTF.de_out_bool, "U_INTF.de_out_bool");
  sca_util::sca_trace(tf, U_INTF.de_in_bool, "U_INTF.de_in_bool");
  sca_util::sca_trace(tf, U_INTF.de_out_logic, "U_INTF.de_out_logic");
  sca_util::sca_trace(tf, U_INTF.de_in_logic, "U_INTF.de_in_logic");

  sca_util::sca_trace_file *tft = sca_util::sca_create_tabular_trace_file("tdfde_port_tracing.dat");
  sca_util::sca_trace(tft, inf_out, "inf_out");
  sca_util::sca_trace(tft, U_INTF.de_out, "U_INTF.de_out");
  sca_util::sca_trace(tft, U_INTF.de_in, "U_INTF.de_in");
  sca_util::sca_trace(tft, U_INTF.de_out_bool, "U_INTF.de_out_bool");
  sca_util::sca_trace(tft, U_INTF.de_in_bool, "U_INTF.de_in_bool");
  sca_util::sca_trace(tft, U_INTF.de_out_logic, "U_INTF.de_out_logic");
  sca_util::sca_trace(tft, U_INTF.de_in_logic, "U_INTF.de_in_logic");

  sc_core::sc_start(10.0, sc_core::SC_MS);

  sca_util::sca_close_vcd_trace_file(tf);
  sca_util::sca_close_tabular_trace_file(tft);

  std::cout << std::endl << "---------------------------------------------------------";
  std::cout << std::endl << "Check results of time domain simulation ..." << std::endl;

  test_util::check_results("tdfde_port_tracing", 7, "converterport_tracing");

  TEST_LABEL_END;

  return (EXIT_SUCCESS);
}
