/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

// Example of usage:
//
//    test_util::sorted_stream sstr; // global declaration
//
//    somewhere in the model
//
//     sstr.redirect_cout(get_time());
//
//    SC_REPORT_INFO("Simple_TDF", str.str().c_str());
//
//    sstr.restore_cout();
//------------------------------------
//
//   //inside TDF
//   sstr.set_time(this->get_time(), 0);
//   sstr << mod.name() << ": " << val << " at " << mod.get_time() << std::endl;
//   sstr.flush_stamp()
//
//  in sc_module's set_time is not required (current time is used and priority is 0)
//   sstr << mod.name() << ": " << val << " at " << sc_core::sc_time_stamp() << std::endl;
//   sstr.flush_stamp()
//
//  when all printouts are collected, they can be sorted and printed
//
//    sstr.print();

#ifndef TEST_PRINT_SORTER_H_
#define TEST_PRINT_SORTER_H_

#include <systemc-ams>
#include <algorithm>

// test utilities in dedicated namespace
// to avoid conflict with local definitions in tests
namespace test_util {

/**
 * this class sorts printouts regarding time, priority and alphabetical to
 * allow simple comparisons also for the case the printout order is scrambled
 * due e.g. the scheduling order is different
 */
class printout_sorter
{
public:
  /**
   * adds a stamp - if no time or sc_core::sc_max_time() given the current SystemC time is used
   * @param time
   * @param printout
   * @param priority
   */
  void add(const sca_core::sca_time &time, const std::string &printout, int priority = 0);

  /**
   * sorts and prints out all stamps
   * stamps are sorted regarding there time, than priority (lowest first) and alphabetically
   */
  void print(std::ostream &ostr = std::cout);

private:
  std::map<sca_core::sca_time, std::vector<std::pair<int, std::string>>> printout_map;
};

/**
 * adds a stamp - if no time or sc_core::sc_max_time() given the current SystemC time is used
 * @param time
 * @param printout
 * @param priority
 */
void printout_sorter::add(const sca_core::sca_time &time, const std::string &printout, int priority)
{
  printout_map[time].push_back({priority, printout});
}

/**
 * sorts and prints out all stamps
 * stamps are sorted regarding there time, than priority (lowest first) and alphabetically
 */
void printout_sorter::print(std::ostream &ostr)
{
  for (auto &timepoint : printout_map)
  {
    std::sort(timepoint.second.begin(), timepoint.second.end(),
          [](const std::pair<int, std::string> &a, const std::pair<int, std::string> &b)
          {
            if (a.first == b.first)
            {
              return a.second < b.second;
            }

            return a.first < b.first;
          });

    for (const auto &entry : timepoint.second)
    {
      ostr << entry.second;
    }
  }

  printout_map.clear();
}

/////////////////////////////////////////

class sorted_stream : public std::streambuf, public std::ostream
{
public:
  sorted_stream() : std::ostream(this) {}

  /**
   * sets the time stamp and optional priority (lowest first)
   * @param time
   * @param cpriority
   */
  void set_time(const sca_core::sca_time &time, int cpriority = 0)
  {
    ctime = time;
    priority = cpriority;
  }

  /**
   * sets the priority only
   * @param cpriority
   */
  void set_priority(int cpriority)
  {
    priority = cpriority;
  }

  /**
   * finishes a stamp (one entry for sorting)
   */
  void flush_stamp()
  {
    this->flush();
    if (ctime == sc_core::sc_max_time())
    {
      ctime = sc_core::sc_time_stamp();
    }

    if (!ostr.str().empty())
    {
      sorter.add(ctime, ostr.str(), priority);
    }

    ctime = sc_core::sc_max_time();
    priority = 0;
    ostr.str("");
  }

  /**
   * prints out the collected stamps
   */
  void print(std::ostream &ostr = std::cout)
  {
    this->flush_stamp();
    sorter.print(ostr);
  }

  /**
   * redirects cout and if time argument given it sets the time stamp
   * @param time
   * @param priority
   */
  void redirect_cout(const sca_core::sca_time &time = sc_core::sc_max_time(), int priority = 0)
  {
    std::cout.flush();
    cout_rdbuf = std::cout.rdbuf();
    std::cout.rdbuf(this);

    if (time != sc_core::sc_max_time())
    {
      this->set_time(time, priority);
    }
  }

  /**
   * restores cout and flushes the stamp
   */
  void restore_cout()
  {
    std::cout.flush();
    this->flush_stamp();
    std::cout.rdbuf(cout_rdbuf);
    cout_rdbuf = NULL;
  }

private:
  std::ostringstream ostr;

  printout_sorter sorter;

  sca_core::sca_time ctime = sc_core::sc_max_time();
  int priority = 0;

  std::streambuf *cout_rdbuf = NULL;

  std::streambuf::int_type overflow(std::streambuf::int_type c) override
  {
    ostr << (char)c;
    return 0;
  }
};

} // namespace test_util

#endif /* TEST_PRINT_SORTER_H_ */
