/*****************************************************************************

  Licensed to Accellera Systems Initiative Inc. (Accellera) under one or
  more contributor license agreements.  See the NOTICE file distributed
  with this work for additional information regarding copyright ownership.
  Accellera licenses this file to you under the Apache License, Version 2.0
  (the "License"); you may not use this file except in compliance with the
  License.  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
  implied.  See the License for the specific language governing
  permissions and limitations under the License.

 *****************************************************************************/

#ifndef TEST_PREDEFINED_MODULES_H_
#define TEST_PREDEFINED_MODULES_H_

#include <systemc-ams>
#include <string>
#include "test_check_results.h"

// test utilities in dedicated namespace
// to avoid conflict with local definitions in tests
namespace test_util {

///////////////////////////////////////////////////////////////////////////////
// non-conservative sinusoidal source
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(sin_src)
{
  sca_tdf::sca_out<double> outp;

  struct params
  {
    double freq;
    double ampl;
    double offset;
    sc_core::sc_time sample_time;

    params()
    {
      ampl   = 1.0;
      freq   = 1e03;
      offset = 0.0;
      sample_time = sc_core::sc_time(0.25, sc_core::SC_MS);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  sin_src(sc_core::sc_module_name nm, params pa = params()) : p(pa) {}

private:
  params p;
};

// frequency domain implementation
void sin_src::ac_processing()
{
  // we use for ac-domain for all frequencies the same amplitude and a phase=0
  sca_ac_analysis::sca_ac(outp) = p.ampl;
}

// time domain implementation
void sin_src::processing()
{
  outp = p.ampl * std::sin(2.0 * M_PI * p.freq * get_time().to_seconds()) + p.offset;
}

///////////////////////////////////////////////////////////////////////////////
// non-conservative sinusoidal source with discrete-event output
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(sin_src_de)
{
  sca_tdf::sca_de::sca_out<double> outp;

  struct params
  {
    double freq;
    double ampl;
    double offset;
    sc_core::sc_time sample_time;

    params()
    {
      ampl   = 1.0;
      freq   = 1e03;
      offset = 1.1;
      sample_time = sc_core::sc_time(0.25, sc_core::SC_US);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void processing()
  {
    outp = p.ampl*std::sin(2.0 * M_PI * p.freq * sc_core::sc_time_stamp().to_seconds()) + p.offset;
  }

  sin_src_de( sc_core::sc_module_name nm, params pa = params() )
  : p(pa)
  {
    outp.initialize_de_signal(1.0);
  }

 private:
  params p;
};

///////////////////////////////////////////////////////////////////////////////
// non-conservative constant source
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(const_src)
{
  sca_tdf::sca_out<double> outp;

  struct params
  {
    double ampl;
    sc_core::sc_time sample_time;

    params()
    {
      ampl = 1.0;
      sample_time = sc_core::sc_time(10.0, sc_core::SC_MS);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  const_src(sc_core::sc_module_name nm, params pa = params()) : p(pa) {}

private:
  params p;
};

// frequency domain implementation
void const_src::ac_processing()
{
  sca_ac_analysis::sca_ac(outp) = p.ampl;
}

// time domain implementation
void const_src::processing()
{
  outp = p.ampl;
}

///////////////////////////////////////////////////////////////////////////////
// non-conservative sinusoidal source implemented to test the capacitor
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(sin2_src)
{
  sca_tdf::sca_out<double> outp;

  struct params
  {
    double freq;
    double ampl;
    sc_core::sc_time sample_time;

    params()
    {
      ampl   = 1.0;
      freq   = 1e02;
      sample_time = sc_core::sc_time(0.25, sc_core::SC_MS);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  sin2_src(sc_core::sc_module_name nm, params pa = params()) : p(pa) {}

private:
  params p;
};

// frequency domain implementation
void sin2_src::ac_processing()
{
  // we use for ac-domain for all frequencies the same amplitude and a phase=0
  sca_ac_analysis::sca_ac(outp) = p.ampl;
}

// time domain implementation
void sin2_src::processing()
{
  //3.141592653590
  if (get_time().to_seconds() <= 0.01)
    outp = p.ampl * std::sin(2.0 * M_PI * 100 * get_time().to_seconds());
  else
    if ((get_time().to_seconds() > 0.01) && (get_time().to_seconds() <= 0.034)  )
      outp = p.ampl * std::sin(2.0 * M_PI * 41.6666666667 * get_time().to_seconds());
    else
      outp = p.ampl * std::sin(2.0 * M_PI * 25 * get_time().to_seconds());
}

///////////////////////////////////////////////////////////////////////////////
// non-conservative constant source implemented to test the capacitor
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(const2_src)
{
  sca_tdf::sca_out<double> outp;

  struct params1
  {
    double c1;
    sc_core::sc_time sample_time;

    params1()
    {
      c1 = 7.95774715459e-07;
      sample_time = sc_core::sc_time(0.25, sc_core::SC_MS);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  const2_src(sc_core::sc_module_name nm, params1 pa = params1()) : p(pa) {}

private:
  params1 p;
};

// frequency domain implementation
void const2_src::ac_processing()
{
  sca_ac_analysis::sca_ac(outp) = p.c1;
}

// time domain implementation
void const2_src::processing()
{
  /*   outp = p.c1; */
  if (get_time().to_seconds() <= 0.01)
    p.c1 = 7.95774715459e-07;
  else
  {
    if ((get_time().to_seconds() > 0.01) && (get_time().to_seconds() <= 0.034)  )
      p.c1 = 1.9098593171e-06;
    else
      p.c1 = 3.18309886184e-06;
  }
  outp = p.c1;
}

///////////////////////////////////////////////////////////////////////////////
// non-conservative pulse source2
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(pulse_src2)
{
  sca_tdf::sca_out<double> outp;

  struct params
  {
    double val;
    sc_core::sc_time sample_time;

    params()
    {
      val = 1e12;
      sample_time = sc_core::sc_time(10.0, sc_core::SC_MS);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  pulse_src2(sc_core::sc_module_name nm, params pa = params()) : p(pa), cnt(0)
  {
  }

private:
  params p;
  int cnt;
};

// frequency domain implementation
void pulse_src2::ac_processing()
{
  sca_ac_analysis::sca_ac(outp) = p.val;
}

// time domain implementation
void pulse_src2::processing()
{
  outp = p.val;
  ++cnt;

  if (cnt == 4)
    p.val = 1e-12;
  else
    if (cnt == 8)
    {
      p.val = 1e12;
      cnt = 0;
    }
}

///////////////////////////////////////////////////////////////////////////////
// non-conservative pulse source
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(pulse_src)
{
  sca_tdf::sca_out<bool> outp;

  struct params
  {
    double init_val;
    sc_core::sc_time sample_time;

    params()
    {
      init_val = 0;
      sample_time = sc_core::sc_time(10.0, sc_core::SC_MS);
    }
  };

  void set_attributes()
  {
    outp.set_timestep(p.sample_time);
  }

  void ac_processing();
  void processing();

  pulse_src(sc_core::sc_module_name nm, params pa = params()) : p(pa), cnt(0)
  {}

private:
  params p;
  int cnt;
};

// frequency domain implementation
void pulse_src::ac_processing()
{
  sca_ac_analysis::sca_ac(outp) = p.init_val;
}

// time domain implementation
void pulse_src::processing()
{
  outp = p.init_val;
  ++cnt;
  if (cnt == 3)
    p.init_val = !p.init_val;
  else if (cnt == 8)
    cnt = 0;
}

///////////////////////////////////////////////////////////////////////////////
// conv_tdf2de
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(conv_tdf2de)
{
  sca_tdf::sca_in<double > tdf_i;
  sca_tdf::sc_out<double > sc_o;

  struct params
  {
    params()
    {
    }
  } p;

  void processing()
  {
    sc_o.write(double(tdf_i.read()));
  }

  conv_tdf2de(sc_core::sc_module_name nm, params pa = params() )
  : tdf_i("tdf_i"), sc_o("sc_o"), p(pa)
  {}
};

///////////////////////////////////////////////////////////////////////////////
// ramp
///////////////////////////////////////////////////////////////////////////////

SCA_TDF_MODULE(ramp)
{
  sca_tdf::sca_out<double> output; // Declaration of Ports of the TDF module

  struct params
  {
    double inclination;
    double cutoff; // Parameters of the function ramp (wich will change the value of the capacitance)
    sc_core::sc_time sample_time;

    params()
    {
    }
  } p;

  void set_attributes()
  {
    output.set_timestep(p.sample_time);
  }

  void processing()
  {
    output = p.inclination * get_time().to_seconds() + p.cutoff; // Definition of the time behavior of the module
  }

  ramp( sc_core::sc_module_name nm, params pa = params() )
  : output("output"), p(pa) {}
};

} // namespace test_util

#endif // TEST_PREDEFINED_MODULES_H_
