/**
 *
 * @file task.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: task.hh 1420 2010-09-07 12:51:24Z lehton87 $
 *
 */


#ifndef SCTG_TASK_HH
#define SCTG_TASK_HH

#include "trigger.hh"
#include "configuration.hh"
#include "tg_token.hh"
#include "common.hh"

#include <boost/property_tree/ptree.hpp>
#include <string>
#include <vector>

#include <systemc>



namespace sctg
{
   class Trigger;
   
   class Task
   {
   public:        
      
      //* Constructor
      Task(const boost::property_tree::ptree& pt,
	   Configuration& config);
      
      //* Destructor
      ~Task();
      
      const std::string& getName();
      const std::string& getClass();
      unsigned long int  getId();
      
      //* True if this task has in_port with that id
      bool hasInPort(unsigned long int id);

      //* True if this task has out_port with that id
      bool hasOutPort(unsigned long int id);
      
      //* Returns tasks current operation
      OperationType getOperation();
      
      //* True if its new operation (not yet partially consumed)
      bool isNewOperation();
      
      //* Returns current amount what is left from current operation
      unsigned long int getAmount();
      
      //* Return current operation's out_port_id
      unsigned long int getOutPort();
      
      //* Consumes amount of operations (or bytes for SENDs)
      //* @return How many operations are left
      unsigned long int consume(unsigned long int amount);
      
      //* Returns task's state
      State getState();
      
      //* Task receives a packet (token)
      void receive(tgToken token);
      
      //* Changes task's state 
      void changeState(State newState);
      
      //* Returns buffered bytes
      unsigned long int getBufferUsage() const;

      //* Returns how many times task has been triggered
      unsigned long int getTimesTriggered();

      //* Returns sent local bytes
      unsigned long int getLocalBytesSent();

      //* Adds to sent local bytes
      void addLocalBytesSent(unsigned long int b);

      //* Returns sent inter PE bytes
      unsigned long int getRemoteBytesSent();

      //* Adds to sent remote bytes
      void addRemoteBytesSent(unsigned long int b);
      
      //* Returns how many cycles task has executed so far
      unsigned long int getCyclesExecuted();

      //* Returns last latency
      const sc_core::sc_time& getLastLatency() const;

      //* Returns last delay
      const sc_core::sc_time& getLastDelay() const;

      //* Returns last response time
      const sc_core::sc_time& getLastResponse() const;

      //* Set last response time
      void setLastResponse(const sc_core::sc_time& resp);

      //* Increase trigger time counter
      void incTriggered(void);

   private:            
      
      std::string           _name;
      unsigned long int     _id;
      std::string           _class;
      State                 _state;
      
      std::vector<unsigned long int> _inPorts;
      std::vector<unsigned long int> _outPorts;
      
      std::vector<sctg::Trigger*> _triggers;
      unsigned int                _currentTrigger;    
      
      Configuration&  _config;

      unsigned long int _localBytesSent;
      unsigned long int _remoteBytesSent;

      sc_core::sc_time _lastLatency;
      sc_core::sc_time _lastDelay;
      sc_core::sc_time _lastResponseTime;
      
      unsigned long int _timesTriggered;

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
