/**
 *
 * @file configuration.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: configuration.hh 1420 2010-09-07 12:51:24Z lehton87 $
 *
 */


#ifndef SCTG_CONFIGURATION_HH
#define SCTG_CONFIGURATION_HH

#include "noc_conf_if.hh"
#include "non_sc_conf_if.hh"

#include <boost/property_tree/ptree.hpp>
#include <boost/random.hpp>
#include <boost/optional.hpp>
#include <systemc>
#include <string>
#include <vector>
#include <map>
#include <iostream>
#include <fstream>
#include <queue>

namespace sctg
{
   class ProcessingElement;
   class Task;
   class Buffer;
   class BufferInterface;

#ifdef SCTG_USE_EXECMON
   class TcpServerIf;
#endif

   class Configuration : public NocConfIf
#ifdef SCTG_USE_EXECMON
		       , public NonScConfIf
#endif
   {
   public :
    
      typedef boost::lagged_fibonacci19937 RealEngineType; 
      typedef boost::mt19937               IntEngineType;

      //* Constructor
      Configuration(const boost::property_tree::ptree& pt,
		    std::string& saveDir);

      //* Destructor
      ~Configuration();
    
      //* Returns simulation's length
      const sc_core::sc_time& getSimLength() const;

      //* Returns simulation's resolution
      const sc_core::sc_time& getSimResolution() const;

      //* Returns processing element libs filename
      const std::string& getPeLibFile() const;

      //* Returns random number generator engines
      RealEngineType& getRealRNGEngine();
      IntEngineType& getIntRNGEngine();

      //* Returns seed
      const unsigned long int getSeed() const;

      //* Returns number between 0.0 and 1.0
      double random();

      //* Add task connection from source to destination port
      void addTaskConnection(unsigned long int source,
			     unsigned long int destination);

      //* Returns destination port 
      unsigned long int getDestination(unsigned long int source);

      //* Maps id to PE
      void addPeMap(unsigned long int id, ProcessingElement* pe);
    
      //* Returns PE with whis id
      ProcessingElement* getPe(unsigned long int id);

      //* Maps id to Task
      void addTaskMap(unsigned long int id, Task* task);
    
      //* Returns Task with this id
      Task* getTask(unsigned long int id);

      //* Maps Task id to pe
      void addTaskToPeMap(unsigned long int id, ProcessingElement* pe);

      //* Returns PE with this task
      ProcessingElement* getPeByTask(unsigned long int id);

      //* Maps in_port id to Task
      void addTaskToInPortMap(unsigned long int id, Task* task);

      //* Returns Task which has this in_port
      Task* getTaskByInPort(unsigned long int id);

      //* Returns buffer by tasks in port
      Buffer* getBufferByPe(unsigned long int pe);

      //* Returns unique id every time its called
      unsigned int getTokenId();

      //* Add buffer to pe
      void addBufferToPe(Buffer* buffer, unsigned long int pe);

      //* Returns buffer interface by agent (PE)
      BufferInterface* getBufferIf(unsigned long int agent);

      //* Add task to group mapping
      void addTaskToGroup(unsigned long int task,
			  unsigned long int group);

      //* Add group to pe mapping
      void addGroupToPe(unsigned long int group,
			unsigned long int pe);

      //* Get task's group
      unsigned long int getGroup(unsigned long int task);

      //* Get group's PE
      unsigned long int getPeByGroup(unsigned long int group);

      //* Returns measurement interval time
      const sc_core::sc_time& getMeasurementInterval() const;

      //* Returns ostream for logging packets
      boost::optional<std::ostream*>& getPacketStream();

      //* Returns ostream for logging tokens
      boost::optional<std::ostream*>& getTokenStream();

      //* Returns ostream for logging applications
      boost::optional<std::ostream*>& getAppStream();

      //* Returns ostream for logging PEs
      boost::optional<std::ostream*>& getPeStream();

      //* Returns ostream for Summary
      boost::optional<std::ostream*>& getSummaryStream();

      //* Returns ostream for Execution Monitor log
      boost::optional<std::ostream*>& getExecMonStream();

      const std::string& getNocClass();
      const std::string& getNocType();
      const std::string& getNocSubType();

      void setNocClass(std::string& s);
      void setNocType(std::string& s);
      void setNocSubType(std::string& s);

      std::map<unsigned long int, ProcessingElement*>::iterator getPeBegin();
      std::map<unsigned long int, ProcessingElement*>::iterator getPeEnd();
      std::map<unsigned long int, unsigned long int>::iterator getGroupPeBegin();
      std::map<unsigned long int, unsigned long int>::iterator getGroupPeEnd();
      std::map<unsigned long int, unsigned long int>::iterator getTaskGroupBegin();
      std::map<unsigned long int, unsigned long int>::iterator getTaskGroupEnd();

      void useExecMon(bool use);
      bool useExecMon();

#ifdef SCTG_USE_EXECMON
      //* Sets pointer to TCP server
      void setTcpServer(TcpServerIf* server);
      //* Return pointer to TCP server
      TcpServerIf* getTcpServer();
#endif

      std::vector<std::string>& getCostFunctions();

      void addTokenLatency(unsigned long int src,
			   unsigned long int dst,
			   sc_core::sc_time& latency);


      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
      getTokenLatency();

      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
      getTokenLatencyMax();

      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
      getTokenLatencyMin();

      std::map<unsigned long int, std::map<unsigned long int, unsigned long int> >&
      getTokenCount();

      void addSendTime(unsigned long int port);
      void addReceiveTime(unsigned long int port);

      std::map<std::string, sc_core::sc_time>&  getTotPathLat();
      std::map<std::string, sc_core::sc_time>&  getMaxPathLat();
      std::map<std::string, sc_core::sc_time>&  getMinPathLat();
      std::map<std::string, unsigned long int>& getPathCount();

      void taskTriggered(unsigned long int task, unsigned long int times);

      std::map<std::string, double>& getCostVariables();
      
   private:

      //* Creates sc_time_unit from string
      const sc_core::sc_time_unit parseUnit(std::string unit);

      sc_core::sc_time         _simResolution;
      sc_core::sc_time         _simLength;    
      std::string              _peLibFile;
      std::vector<std::string> _costFunctions;
      // RNG engines 
      RealEngineType          _realEngine;
      IntEngineType           _intEngine;
      unsigned long int       _seed;

      unsigned int       _nextTokenId;

      sc_core::sc_time _measurementInterval;

      bool _useExecMon;

      boost::uniform_real<double>  _random;
    
      std::map<unsigned long int, unsigned long int> _taskConnections;
      std::map<unsigned long int, ProcessingElement*> _peMap;
      std::map<unsigned long int, ProcessingElement*> _peTaskMap;
      std::map<unsigned long int, Task*> _taskMap;
      std::map<unsigned long int, Task*> _taskInPortMap;
      std::map<unsigned long int, Buffer*> _peBufferMap;
      std::map<unsigned long int, unsigned long int> _taskGroupMap;
      std::map<unsigned long int, unsigned long int> _groupPeMap;

      boost::optional<std::ostream*> _pktOutFile;
      boost::optional<std::ostream*> _tokenOutFile;
      boost::optional<std::ostream*> _peOutFile;
      boost::optional<std::ostream*> _appOutFile;
      boost::optional<std::ostream*> _summaryOutFile;
      boost::optional<std::ostream*> _execMonOutFile;

      std::string _nocClass;
      std::string _nocType;
      std::string _nocSubType;

      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >
      _tokenLatency;
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >
      _tokenLatencyMax;
      std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >
      _tokenLatencyMin;

      std::map<unsigned long int, std::map<unsigned long int, unsigned long int> >
      _tokenCount;

      std::multimap<unsigned long int, unsigned long int>             _pathsFwd;
      // Above:       source port,      destination port
      std::multimap<unsigned long int, unsigned long int>             _pathsRev;
      // Above:      destination port,     source port

      // Below: string "src_dst", ...
      std::map<std::string, std::queue<sc_core::sc_time> > _portSendTime;

      std::map<std::string, sc_core::sc_time>  _totPathLat;
      std::map<std::string, sc_core::sc_time>  _maxPathLat;
      std::map<std::string, sc_core::sc_time>  _minPathLat;
      std::map<std::string, unsigned long int> _pathCount;

      std::multimap<unsigned long int, unsigned long int> _taskTriggerTimes;

      std::map<std::string, double> _costVariables;

#ifdef SCTG_USE_EXECMON
      TcpServerIf* _serverIf;
#endif

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
