/**
 *
 * @file sctg_to_ocp_tl3.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: sctg_to_osci_tlm.hh 90 2010-06-23 14:35:10Z lehton87 $
 *
 */

#ifndef SCTG_TO_OCP_TL3_HH
#define SCTG_TO_OCP_TL3_HH

#include "noc_conf_if.hh"
#include "buffer_if.hh"
#include "tg_packet.hh"

#include "tlm.h"
#include "ocpip.h"

#ifdef MTI_SYSTEMC
#include "peq_with_get.h"
#else
#include "tlm_utils/peq_with_get.h"
#endif

#include <systemc>

#include <stdexcept>
#include <iostream>


namespace sctg
{

   template<unsigned int N_AGENTS,
	    unsigned int DATA_WIDTH>
   class SctgToOcpTl3 : public sc_core::sc_module
   {
   public:

      SC_HAS_PROCESS(SctgToOcpTl3);

      ocpip::ocp_master_socket_tl3<DATA_WIDTH, 0>* masterSocket;
      ocpip::ocp_slave_socket_tl3<DATA_WIDTH, 0>* slaveSocket;
      

      //* Constructor
      SctgToOcpTl3(sc_core::sc_module_name name,
		    sctg::NocConfIf* confIf)
	 : sc_module(name),
	   _confIf(confIf),
	   _cycleTime(20, sc_core::SC_NS)
      {

	 // OCP-IP parameters
	 _ocpParameters.byteen = false;	 

	 masterSocket = new ocpip::ocp_master_socket_tl3<DATA_WIDTH, 0>
	    ("master",
	     ocpip::ocp_master_socket_tl3<DATA_WIDTH, 0>::mm_txn_with_data());
	 
	 slaveSocket = new ocpip::ocp_slave_socket_tl3<DATA_WIDTH, 0>
	    ("slave");
	 

	 masterSocket->set_ocp_config(_ocpParameters);
	 slaveSocket->set_ocp_config(_ocpParameters);

	 masterSocket->register_nb_transport_bw
	    (this, &SctgToOcpTl3::nb_transport_bw);
	 slaveSocket->register_nb_transport_fw
	    (this, &SctgToOcpTl3::nb_transport_fw);

	 for(unsigned int i = 0; i < N_AGENTS; ++i)
	 {
	    // Generating PEQ and threads for every agent

	    std::ostringstream oss;
	    oss << "rxPeq_" << i;
	    _rxPeq[i] = new tlm_utils::peq_with_get<tlm::tlm_generic_payload>
	       (oss.str().c_str());	    
	    sc_spawn(sc_bind(&SctgToOcpTl3::sender, this, i));
	    sc_spawn(sc_bind(&SctgToOcpTl3::receiver, this, i));
	    
	 }
      }

      //* Destructor
      ~SctgToOcpTl3()
      {
	 std::cout << "SctgToOcpTl3 destructor" << std::endl;
	 for(unsigned int i = 0; i < N_AGENTS; ++i)
	 {	    
	    delete _rxPeq[i]; _rxPeq[i] = 0;
	 }
	 delete masterSocket; masterSocket = 0;
	 delete slaveSocket; slaveSocket = 0;
      }

   private:

      // Disable operator= and copy constructor
      SctgToOcpTl3& operator=(const SctgToOcpTl3& rhs);
      SctgToOcpTl3(const SctgToOcpTl3& rhs);

      void sender(unsigned int agent)
      {	
	 // Get pointer to buffer between IP -> NoC
	 sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);

	 // If buffer doesn't point anywhere there is no agent with this ID
	 if(!buffer)
	 { return; }

	 // struct we get from TG
	 tgPacket* packet = 0;


	 // TLM structures we're sending to NoC
	 tlm::tlm_generic_payload* trans = 0;
	 tlm::tlm_phase            phase;
	 sc_core::sc_time          delay;

	 tlm::tlm_sync_enum        retval;
	 	 	 
	 while(true)
	 {

	    // Wait for packets to send
	    if(!buffer->txPacketAvailable())
	    {	       
	       sc_core::wait(*(buffer->txGetPacketAvailableEvent()));
	    }
	    	    
	    // Read packet from buffer
	    packet = buffer->txGetPacket();
	    	    
	    trans = masterSocket->get_transaction();

	    // Initialize generic payload
	    trans->set_command(tlm::TLM_WRITE_COMMAND);
	    trans->set_data_length(packet->size);
	    trans->set_byte_enable_ptr(0);
	    trans->set_streaming_width(DATA_WIDTH / 8);

	    trans->set_address(packet->address); 	    
	    trans->set_data_ptr(packet->data); 
	    trans->set_dmi_allowed(false);	    
	    phase = tlm::BEGIN_REQ;
	    delay = sc_core::SC_ZERO_TIME;

	    // Send packet
	    retval = (*masterSocket)[agent]->nb_transport_fw(*trans, phase, delay);
	    
	    if(retval == tlm::TLM_COMPLETED)
	    {
	       if(delay != sc_core::SC_ZERO_TIME)
	       {
		  wait(delay);
	       }
	    }
	    else if(retval == tlm::TLM_UPDATED || retval == tlm::TLM_ACCEPTED)
	    {
	       if(phase == tlm::BEGIN_REQ)
	       {

		  // Wait for END_REQ or BEGIN_RESP
		  wait(_txCompleteEvent[agent]);
	       }
	       else if(phase == tlm::END_REQ)
	       {
		  
		  // BEGIN_RESP will come via backward path
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
		  
		  wait(_txCompleteEvent[agent]);

	       }
	       else if(phase == tlm::BEGIN_RESP)
	       {
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
		  
		  // Send END_RESP immediately
		  trans->set_dmi_allowed(false);
		  phase = tlm::END_RESP;
		  delay = sc_core::SC_ZERO_TIME;
		  retval = (*masterSocket)[agent]->
		     nb_transport_fw(*trans, phase, delay);


		  // Transaction completed
		  
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "Sender " << agent << " got invalid TLM_UPDATED";
		  throw std::runtime_error(oss.str().c_str());
	       }

	    }
	    
	    masterSocket->release_transaction(trans);

	    // Free memory used by the packet
	    packet->data = 0;
	    delete   packet;       packet       = 0;

	 } // End of while(true)

      }

      void receiver(unsigned int agent)
      {	 
	 // Get pointer to buffer between NoC -> IP
	 sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);

	 tlm::tlm_generic_payload* trans;
	 tlm::tlm_phase            phase = tlm::END_REQ;
	 sc_core::sc_time          delay = sc_core::SC_ZERO_TIME;

	 tlm::tlm_sync_enum        retval;

	 tgPacket* packet = 0;
	 
	 // If buffer doesn't point anywhere there is no agent with this ID
	 if(!buffer)
	 { return; }

	 
	 while(true)
	 {
	    if((trans = _rxPeq[agent]->get_next_transaction()) == 0)
	    {	       
	       wait(_rxPeq[agent]->get_event());
	       trans = _rxPeq[agent]->get_next_transaction();
	    }	    

	    packet = new tgPacket;
	    packet->address = trans->get_address();
	    packet->size    = trans->get_data_length();
	    packet->data    = trans->get_data_ptr();

	    unsigned int flits = 
	       trans->get_data_length() / trans->get_streaming_width();


	    // Wait until PE has space to receive packet
	    while(buffer->rxSpaceLeft() < packet->size)
	    {
	       wait(*(buffer->rxGetReadEvent()));
	    }
	    
	    

	    buffer->rxPutPacket(packet);
	    packet = 0; // buffer handles freeing memory

	    

	    phase = tlm::END_REQ;
	    delay = sc_core::SC_ZERO_TIME;

	    trans->set_response_status( tlm::TLM_OK_RESPONSE );
	    retval = (*slaveSocket)[agent]->nb_transport_bw(*trans, phase, delay);
	    
	    trans->release();

	    

	    // Assuming it returns TLM_COMPLETED as we are not using responces
	    if(retval != tlm::TLM_COMPLETED)
	    {
	       std::ostringstream oss;
	       oss << "SctgToOcpTl3::receiver " << agent
		   << ": was expecting TLM_COMPLETE";
	       throw std::runtime_error(oss.str().c_str());
	    }

	 } // End of while(true)

	 
      }

      tlm::tlm_sync_enum nb_transport_fw(unsigned int id,
					 tlm::tlm_generic_payload &trans,
					 tlm::tlm_phase           &phase,
					 sc_core::sc_time         &delay)
      {

	 // Only write command is supported
	 if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	 {
	    std::ostringstream oss;
	    oss << "SctgToOcpTl3::nb_tranport_fw " << id 
		<< ": only write command is supported";
	    throw std::runtime_error(oss.str().c_str());
	 }
	 
	 if(phase == tlm::BEGIN_REQ)
	 {
	    trans.acquire();
	    _rxPeq[id]->notify(trans, delay);
	 }
	 
	 trans.set_response_status( tlm::TLM_OK_RESPONSE );
	 return tlm::TLM_ACCEPTED;
      }
      
      tlm::tlm_sync_enum nb_transport_bw(unsigned int id,
					 tlm::tlm_generic_payload &trans,
					 tlm::tlm_phase           &phase,
					 sc_core::sc_time         &delay)
      {
	 

	 if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	 {
	    std::ostringstream oss;
	    oss << "nb_tranport_bw " << id << " got wrong phase";
	    std::cout << oss.str() << std::endl;
	    throw std::runtime_error(oss.str().c_str());
	 }
	 
	 _txCompleteEvent[id].notify(delay);
	 	 
	 return tlm::TLM_COMPLETED;
      }

      // Pointer to configuration interface
      sctg::NocConfIf* _confIf;

      sc_core::sc_event _txCompleteEvent[N_AGENTS];
      
      tlm_utils::peq_with_get<tlm::tlm_generic_payload>* _rxPeq[N_AGENTS];

      sc_core::sc_time _cycleTime;
      ocpip::ocp_parameters _ocpParameters;

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:

