/**
 *
 * @file processing_element.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: processing_element.cc 1948 2011-08-19 08:17:42Z ege $
 *
 */

#include "processing_element.hh"
#include "memory_model.hh"
#include <iostream>
#include <iomanip>
#include <exception>
#include <algorithm>

namespace sctg
{
   ProcessingElement::ProcessingElement(sc_core::sc_module_name name, 
					const boost::property_tree::ptree& pt,
					const boost::property_tree::ptree& peLib,
					sctg::Configuration& config)
      : sctg::Resource(name, pt, config),
	_config(config),
	_measureStart(),
	_measureEnd(),
	_iCacheMissInterval(0),
	_dCacheMissInterval(0),
	_iLineSize(0),
	_dLineSize(0),
	_iMemId(0),
	_dMemId(0),
	_nextIMiss(0),
	_nextDMiss(0)
   {
      _currentTask = 0;
      _currentState = IDLE;
      _dmaEnabled = true;
    
      _intraGroupRxCost = 0;
      _intraGroupTxCost = 0;
      _interGroupRxCost = 0;
      _interGroupTxCost = 0;
      _interPeRxCost = 0;
      _interPeTxCost = 0;
      _cycleLength =  sc_core::sc_time(1.0 / getFrequency(), sc_core::SC_US);
      using boost::property_tree::ptree;
      ptree node = peLib.get_child("processing_element_lib");
      // Search for correct pe type
      ptree::const_iterator iter;
      for(iter = node.begin(); iter != node.end(); ++iter)
      {	
	 if((*iter).first == "processing_element" &&
	    (*iter).second.get<std::string>("<xmlattr>.type") == getType() )
	 {
	    break; // Found, stop search
	 }
      }
      if(iter == node.end())
      {
	 // Did not find
	 std::string err = "Did not find processing element \"" + getType() +
	    "\" from pe_lib";
	 throw std::runtime_error(err.c_str());
      }
      _intOps = (*iter).second.get<double>("<xmlattr>.int_ops");
      _floatOps = (*iter).second.get<double>("<xmlattr>.float_ops");
      _memOps = (*iter).second.get<double>("<xmlattr>.mem_ops");

      // Add communication costs if they exists
      boost::optional<const ptree&> costs = (*iter).second.get_child_optional
	 ("communication_costs");
    
      boost::optional<const ptree&> temp;
      if(costs)
      {
	 for(ptree::const_iterator i = (*costs).begin(); i != (*costs).end(); ++i)
	 {
	    if((*i).first == "intragroup")
	    {
	       temp = (*i).second.get_child_optional("receive");
	       if(temp) {_intraGroupRxCost = 
		     new Amount<unsigned long int>((*temp), _config);}
	       temp = (*i).second.get_child_optional("send");
	       if(temp) {_intraGroupTxCost = 
		     new Amount<unsigned long int>((*temp), _config);}
	    }
	    else if((*i).first == "intergroup")
	    {
	       temp = (*i).second.get_child_optional("receive");
	       if(temp) {_interGroupRxCost = 
		     new Amount<unsigned long int>((*temp), _config);}
	       temp = (*i).second.get_child_optional("send");
	       if(temp) {_interGroupTxCost = 
		     new Amount<unsigned long int>((*temp), _config);}
	    }
	    else if((*i).first == "interpe")
	    {
	       temp = (*i).second.get_child_optional("receive");
	       if(temp) {_interPeRxCost = 
		     new Amount<unsigned long int>((*temp), _config);}
	       temp = (*i).second.get_child_optional("send");
	       if(temp) {_interPeTxCost = 
		     new Amount<unsigned long int>((*temp), _config);}
	    }
	 }
      }
    
      std::cout << getName() << ": int (" << _intOps << ") float (" << _floatOps
		<< ") mem (" << _memOps << ")" << std::endl;
      if(costs)
	 std::cout << " Found costs" << std::endl;

      
      // Parse cache misses
      using boost::property_tree::ptree;
      for(ptree::const_iterator iter = pt.begin(); iter != pt.end(); ++iter)
      {
	 if((*iter).first == "cache")
	 {	    
	    for(ptree::const_iterator iter2 = (*iter).second.begin(); 
		iter2 != (*iter).second.end(); ++iter2)
	    {
	       if((*iter2).first == "i_miss")
	       {
		  _iLineSize = (*iter2).second.get<unsigned long int>
		     ("<xmlattr>.line_size");
		  _iMemId = (*iter2).second.get<unsigned long int>
		     ("<xmlattr>.mem_area_id");
		  _iCacheMissInterval = new Amount<double>((*iter2).second,
							   config);
	       }
	       
	       if((*iter2).first == "d_miss")
	       {
		  _dLineSize = (*iter2).second.get<unsigned long int>
		     ("<xmlattr>.line_size");
		  _dMemId = (*iter2).second.get<unsigned long int>
		     ("<xmlattr>.mem_area_id");
		  _dCacheMissInterval = new Amount<double>((*iter2).second,
							   config);
	       }
	    }  
	 }
      }

      
    
      SC_THREAD(thread);

      if(_dmaEnabled)
      {
	 SC_THREAD(txDma);
	 SC_THREAD(rxDma);
      }
   }

   ProcessingElement::~ProcessingElement()
   {
      if(_intraGroupRxCost) { delete _intraGroupRxCost; _intraGroupRxCost = 0; }
      if(_intraGroupTxCost) { delete _intraGroupTxCost; _intraGroupTxCost = 0; }
      if(_interGroupRxCost) { delete _interGroupRxCost; _interGroupRxCost = 0; }
      if(_interGroupTxCost) { delete _interGroupTxCost; _interGroupTxCost = 0; }
      if(_interPeRxCost) { delete _interPeRxCost; _interPeRxCost = 0; }
      if(_interPeTxCost) { delete _interPeTxCost; _interPeTxCost = 0; }

      updateMeasurements();

      double util = 0.0;

      if(_measurements.busyCycles != 0.0)
      {
	 util = double(_measurements.busyCycles) / 
	    double(_measurements.idleCycles + _measurements.busyCycles);
      }

      if(_config.getSummaryStream())
      {
	 **_config.getSummaryStream()
	    << "- PE " << getName() 
	    << " average utilization during simulation was "
	    << util << std::endl
	    << "  Clock cycles" << " idle: " 
	    << std::setw(10) << _measurements.idleCycles
	    << "  exec: " << std::setw(10) << _measurements.busyCycles
	    << "  total: " << std::setw(10)
	    << (_measurements.idleCycles + _measurements.busyCycles)
	    << std::endl
	    << "  Inter PE TX " 
	    << std::setw(10) << _buffer->getMeasurements().txBytes
	    << " bytes" << std::endl
	    << "  Inter PE RX " 
	    << std::setw(10) << _buffer->getMeasurements().rxBytes - 
	    _buffer->getMeasurements().internalBytes
	    << " bytes" << std::endl
	    << "  Intra PE TX " 
	    << std::setw(10) << _buffer->getMeasurements().internalBytes
	    << " bytes"
	    << std::endl;
      }

   }

   void ProcessingElement::thread()
   {
      double waitTime = 1.0;

      // Schedule first task
      bool sched = true;
      bool sendStalled = false;
    
      while(true)
      {
	 if(_currentTask == 0 || sched || _currentTask->getState() != READY)
	 {
	    Task* old = _currentTask;
	    _currentTask = schedule();    
	    sched = false;
	    if(old != _currentTask)
	    {
	       sendStalled = false;
	    }
	 }
	
	 /*
	  * 1. If something is coming in handle it
	  * 2. Otherwise execute task
	  */
	 if(_buffer->rxTokenAvailable())
	 {
// 	    std::cout << "token available " 
// 			    << getName() << std::endl;
	    tgToken tokenIn = _buffer->rxGetToken();
	    
	    _config.addReceiveTime(tokenIn.destination);
	    
	    Task* receiver = dynamic_cast<Task*>
	       (_config.getResourceUserByInPort(tokenIn.destination));
	    
	    if(!receiver)
	    {
	       std::ostringstream oss;
	       oss << "No task with in_port " << tokenIn.destination
		   << " found on PE " << getName();
	       std::cout << oss.str() << std::endl;
	       throw std::runtime_error(oss.str().c_str());
	    }
	    
	    if(_currentTask && _currentTask->getOperation() == WAIT_RESP 
	       && _currentTask == receiver)
	    {
//   	       std::cout << "WAIT_RESP consuming " << tokenIn.size 
//   			 << " bytes" << std::endl;
	       _currentTask->consume(tokenIn.size);
	       _buffer->removeToken(tokenIn.size);
	    }
	    else
	    {
//  	       std::cout << "OTHERS consuming " << tokenIn.size 
//   			 << " bytes" << std::endl;
	       receiver->receive(tokenIn);
	       _fifoScheduler.push(receiver);
	    }


#ifdef SCTG_USE_EXECMON	    
	    if(!tokenIn.isEvent)
	    {
	       _tokenQueue.push(tokenIn);
	    }
#endif

	 }
	 else if(_currentTask == 0 || sendStalled) // No tasks to execute
	 {
// 	    std::cout << "nothing to execute " 
// 			    << getName() << std::endl;
	    sendStalled = false;
	    _currentState = IDLE;
	    //_measureStart = sc_core::sc_time_stamp();
	    wait(_eventEvent 
		 | *(_buffer->rxGetTokenAvailableEvent())
		 | *(_buffer->txGetReadEvent()));
	    updateMeasurements();	    
	 }
	 else
	 {
	    // Check what is current operation
	    /*
	     * 1. If sending send packets one by one
	     * 2. Otherwise execute until all executed or incoming packet
	     */
	    tgToken token;
	    tgPacket* packet;
	    unsigned long int port;
	    unsigned long int task;
	    unsigned long int size;
	    Resource* pe;
	    unsigned long int execcycles = _currentTask->getAmount();

	    switch(_currentTask->getOperation())
	    {
	       case EXECUTE:
		  		  
		  
		  if(_iCacheMissInterval && _dCacheMissInterval)
		  {		     

		     if(_nextIMiss < 1)
		     {
			_measurements.iCacheMisses++;
			_currentState = I_CACHE_MISSING;
			// Calculate next time I-cache misses
			_nextIMiss = _iCacheMissInterval->value();

			// Send cache miss read packet
			token.size = 8;
			token.source = getId();
			token.destination = _iMemId;
			token.returnPort = 0;
			token.respSize = _iLineSize;
			token.burstSize = _iLineSize;
			token.id = _config.getTokenId();
			token.isEvent = false;
			token.type = CACHE_MISS;
			
			_config.getBufferByResource
			   (_config.getResourceByResourceUser(_iMemId)->getId())
			   ->expectToken(token);
			
			// Create packet
			packet = new tgPacket;
			packet->size = 8;
			packet->id = token.id;
			pe = _config.getResourceByResourceUser(_iMemId);
			packet->address = 
			   pe->getTerminalAddress(pe->getTerminal());
			
			while(_buffer->txSpaceLeft() < packet->size)
			{			
			   wait(*(_buffer->txGetReadEvent()));
			}
			
			_buffer->txPutPacket(packet);
			
			// Wait for it to come back
			long int waitBytes = _iLineSize;
			sc_core::sc_time t = sc_core::sc_time_stamp();
//  			std::cout << "EXECUTING I-CACHE MISS, WAITING FOR "
//  				  << waitBytes << " BYTES " 
//  				  << sc_core::sc_time_stamp().value()
//  				  << std::endl;
			while(waitBytes > 0)
			{
			   
			   wait(*(_buffer->rxGetCacheAvailableEvent()));
			   tgPacket* rcv = _buffer->rxGetCache();   
			   waitBytes -= rcv->size;
			   _buffer->removeToken(rcv->size);
			   delete rcv->data;
			   delete rcv;			   
			}
//  			std::cout << "CACHE READ in "
//  				  << (sc_core::sc_time_stamp() - t).value()
//  				  << std::endl;
			updateMeasurements();
		     }

		     if(_nextDMiss < 1)
		     {
			_measurements.dCacheMisses++;
			_currentState = D_CACHE_MISSING;
			// Calculate next time I-cache misses
			_nextDMiss = _dCacheMissInterval->value();

			// Send cache miss read packet
			token.size = 8;
			token.source = getId();
			token.destination = _dMemId;
			token.returnPort = 0;
			token.respSize = _dLineSize;
			token.burstSize = _dLineSize;
			token.id = _config.getTokenId();
			token.isEvent = false;
			token.type = CACHE_MISS;
			
			_config.getBufferByResource
			   (_config.getResourceByResourceUser(_dMemId)->getId())
			   ->expectToken(token);
			
			// Create packet
			packet = new tgPacket;
			packet->size = 8;
			packet->id = token.id;
			pe = _config.getResourceByResourceUser(_dMemId);
			packet->address = 
			   pe->getTerminalAddress(pe->getTerminal());
			

			while(_buffer->txSpaceLeft() < packet->size)
			{			
			   wait(*(_buffer->txGetReadEvent()));
			}
			
			_buffer->txPutPacket(packet);
			
			// Wait for it to come back
			long int waitBytes = _dLineSize;

// 			std::cout << "EXECUTING D-CACHE MISS, WAITING FOR "
//  				  << waitBytes << " BYTES " 
//  				  << sc_core::sc_time_stamp().value()
//  				  << std::endl;

			while(waitBytes > 0)
			{
			   wait(*(_buffer->rxGetCacheAvailableEvent()));
			   tgPacket* rcv = _buffer->rxGetCache();
			   waitBytes -= rcv->size;
			   delete rcv->data;
			   delete rcv;
			   _buffer->removeToken(rcv->size);
			}
			updateMeasurements();
		     }

		     execcycles = std::min(execcycles, _nextIMiss);
		     execcycles = std::min(execcycles, _nextDMiss);
		     
		     _nextIMiss -= execcycles;
		     _nextDMiss -= execcycles;
		  }

		  _currentState = EXECUTING;
		  _currentTask->changeState(RUN);
		  /*std::cout << "At " << sc_core::sc_time_stamp() 
		    << " Task " << _currentTask->getName()
		    << " has " << _currentTask->getAmount()
		    << " clock cycles left to execute" << std::endl;
		  */
		  //waitTime = _cycleLength * _currentTask->getAmount();
		  //std::cout << " Time to wait " << waitTime << std::endl;
		  wait(_cycleLength * execcycles);
		  //_measurements.execCycles += _currentTask->getAmount();
		  updateMeasurements();
		  _currentTask->consume(execcycles);	    
		  _currentTask->changeState(WAIT);
		  break;

	       case WAIT_RESP:
//  		  std::cout << "waiting response " 
//  			    << getName() << std::endl;
		  _currentState = READING;
		  wait(_eventEvent 
		       | *(_buffer->rxGetTokenAvailableEvent())
		       | *(_buffer->txGetReadEvent()));
		  updateMeasurements();
//  		  std::cout << "done waiting response " 
//  			    << getName() << std::endl;
		  break;

	       case SEND:
	       case READ:
		  _currentTask->changeState(RUN);
		  if(_currentTask->getOperation() == SEND)
		  { 
		     _currentState = SENDING; 
// 		     std::cout << "PE " << getName() << " sending" << std::endl;
		  }
		  else
		  { 
		     _currentState = READING; 
//  		     std::cout << "PE " << getName() << " reading" << std::endl;
		  }
		  
// 		  std::cout << "At " << sc_core::sc_time_stamp() 
// 			    << " Task " << _currentTask->getName()
// 			    << " has " << _currentTask->getAmount()
// 			    << " bytes left to send" << std::endl;
		  
		  // Construct token if this is first time for this send
		  if(_currentTask->isNewOperation())
		  {
		     token.destination = 
			_config.getDestination(_currentTask->getOutPort());
		     
		     
		     if(_currentTask->getOperation() == SEND)
		     { 
			token.size = _currentTask->getAmount();
		     }
		     else
		     { 
			
			token.size = dynamic_cast<MemoryModel*>
			(_config.getResourceByResourceUser
			 (_config.getResourceUserByInPort
			  (token.destination)->getId())
			   )->getReqSize();
		  }		
		  

		  token.source = _currentTask->getOutPort();
		  token.timeSent = sc_core::sc_time_stamp();
		  token.id = _config.getTokenId();
		  if(_packetSize != 0)
		  {
		     token.packets = ((token.size - 1) / _packetSize) + 1;
		  }
		  else
		  {
		     token.packets = 1;
		  }
		  
		  token.burstSize  = _currentTask->getBurstAmount();
		  

		  if(_currentTask->getOperation() == READ)
		  {
		     token.type = READ_CMD; 
		     token.returnPort = _currentTask->getRespPort();
		     token.respSize   = _currentTask->getReadAmount();
		     
		     unsigned int reqs = 
			((token.respSize - 1) / token.burstSize) + 1;
		     token.size *= reqs;
		     
		  }

		     task = 
			_config.getResourceUserByInPort(token.destination)
			->getId();
		     _config.getBufferByResource
			(_config.getResourceByResourceUser(task)->getId())
			->expectToken(token);
		     _sendTokens[_currentTask->getId()] = token;

		     _config.addSendTime(token.source);
		  }
		    
		  // Create packet
		  packet = new tgPacket;
		  if(_packetSize > 0) // Zero packet_size means "don't cut tokens"
		  {
		     size = 
			_sendTokens[_currentTask->getId()].size < _packetSize ?
			_sendTokens[_currentTask->getId()].size : _packetSize;
		  }
		  else
		  {
		     size = _sendTokens[_currentTask->getId()].size;
		  }
		  size = size < 4 ? 4 : size; // Four bytes is minimum
					      // packet size
		  packet->size = size;
		  // std::cout << "packet size " << packet->size << std::endl;
		  port = _sendTokens[_currentTask->getId()].destination;
		  // std::cout << "packet port " << port << std::endl;
		  task = _config.getResourceUserByInPort(port)->getId();
		  //std::cout << "packet task " << task << std::endl;
		  pe = _config.getResourceByResourceUser(task);
		  packet->address = pe->getTerminalAddress(pe->getTerminal());
		  //std::cout << "packet address " << std::hex 
		  //	      << packet->address << std::dec << std::endl;
		  packet->id   = _sendTokens[_currentTask->getId()].id;
		  //packet->data = new unsigned char[packet->size];
		  //*(reinterpret_cast<unsigned int*>(packet->data)) =
		  //   _sendTokens[_currentTask->getId()].id;

		  // Check where packet is going
		  if(pe->getId() == getId())
		  {		       
		     // Packet is for this PE
		     if(_buffer->rxSpaceLeft() < packet->size)
		     {
			/*std::cout << "No space for intra_tx (S:" 
				  << size << "B,P:" << port
				  << ",T:"<< task <<  ")" << std::endl;*/
			sched = true;
			_currentTask->changeState(WAIT);
			_fifoScheduler.push(_currentTask);
			//delete packet->data; packet->data = 0;
			delete packet; packet = 0;
			break;
		     }
		     
		     while(_buffer->rxSpaceLeft() < packet->size)
		     {
			_currentState = INTRA_TX_WAIT;
			wait(*(_buffer->rxGetReadEvent()));
			updateMeasurements();
		     }
		     		     
		     _currentTask->addLocalBytesSent(packet->size);

		     packet->data = new unsigned char[packet->size];
		     *(reinterpret_cast<unsigned int*>(packet->data)) =
		        _sendTokens[_currentTask->getId()].id;

		     if(!_dmaEnabled)
		     {
			_buffer->rxReserveSpace(packet->size);
			_currentState = SENDING;
			waitTime =
			   (1.0 / getFrequency()) * 
			   (size / (getTerminalWidth(getTerminal())/8));
			
			wait(waitTime, sc_core::SC_US);
			updateMeasurements();
			_buffer->rxFreeSpace(packet->size);

			_buffer->addInternalBytes(packet->size);
			_buffer->rxPutPacket(packet);
		     }
		     else
		     {			
			_rxDmaQueue.push(packet);
			_rxDmaEvent.notify(sc_core::SC_ZERO_TIME);
		     }		     
		     _sendTokens[_currentTask->getId()].size -= packet->size;
		  }
		  else
		  {
		     // Packet is heading for another PE
		     if(_buffer->txSpaceLeft() < packet->size)
		     {
			sched = true;
			sendStalled = true;
			_fifoScheduler.push(_currentTask);
			//delete packet->data; packet->data = 0;
			delete packet; packet = 0;
			break;
		     }

		     // while(_buffer->txSpaceLeft() < packet->size)
		     // {			
		     // 	_currentState = TX_WAIT;
		     // 	wait(*(_buffer->txGetReadEvent()));			
		     // 	updateMeasurements();
		     // }

		     _currentTask->addRemoteBytesSent(packet->size);
		
		     if(!_dmaEnabled)
		     {
			_buffer->txReserveSpace(packet->size);
			_currentState = SENDING;
			waitTime =
			   (1.0 / getFrequency()) * 
			   (size / (getTerminalWidth(getTerminal())/8));
			
			wait(waitTime, sc_core::SC_US);
			updateMeasurements();
			_buffer->txFreeSpace(packet->size);
			_buffer->txPutPacket(packet);
		     }
		     else
		     {
			_txDmaQueue.push(packet);
			_txDmaEvent.notify(sc_core::SC_ZERO_TIME);
		     }
       		     _sendTokens[_currentTask->getId()].size -= packet->size;
		  }		  
		    

		  

		  _currentTask->consume(size);
		  // unsigned long int bytesLeft = _currentTask->consume(size);
		  // if(bytesLeft == 0)
		  // {
		  //    _sendTokens[_currentTask->getId()]
		  // }
		  _currentTask->changeState(WAIT);	
// 		  std::cout << "done sending/reading " 
// 			    << getName() << std::endl;
		  break;

	       default:
		  std::cout << "At " << sc_core::sc_time_stamp() 
			    << " Task " << _currentTask->getName()
			    << " operation " << _currentTask->getOperation()
			    << " FAILURE" << std::endl;
		  sc_core::sc_stop();
		  wait(10, sc_core::SC_NS);
		  break;
	    }		
	 }	    	    
      }
    
   }

   double ProcessingElement::getIntOps()
   { return _intOps; }
  
   double ProcessingElement::getFloatOps()
   { return _floatOps; }

   double ProcessingElement::getMemOps()
   { return _memOps; }

   unsigned long int ProcessingElement::getIntraGroupRxCost()
   { return (_intraGroupRxCost == 0 ? 0 : _intraGroupRxCost->value()); }

   unsigned long int ProcessingElement::getIntraGroupTxCost()
   { return (_intraGroupTxCost == 0 ? 0 : _intraGroupTxCost->value()); }

   unsigned long int ProcessingElement::getInterGroupRxCost()
   { return (_interGroupRxCost == 0 ? 0 : _interGroupRxCost->value()); }

   unsigned long int ProcessingElement::getInterGroupTxCost()
   { return (_interGroupTxCost == 0 ? 0 : _interGroupTxCost->value()); }

   unsigned long int ProcessingElement::getInterPeRxCost()
   { return (_interPeRxCost == 0 ? 0 : _interPeRxCost->value()); }

   unsigned long int ProcessingElement::getInterPeTxCost()
   { return (_interPeTxCost == 0 ? 0 : _interPeTxCost->value()); }

   sctg::Task* ProcessingElement::schedule()
   {
      Task* retval = 0;
      while(!_fifoScheduler.empty())
      {
	 retval = _fifoScheduler.front();
	 _fifoScheduler.pop();
	 if(retval->getState() == READY || retval->getState() == RUN)
	 {	    
	    return retval;
	 }
      }

      // loop over all tasks
      for(unsigned int i = 0; i < _tasks.size(); ++i)
      {
	 /*std::cout << "At " << sc_core::sc_time_stamp() 
	   << " Task (" << i << ") " << _tasks.at(i)->getName()
	   << "'s state is " 
	   << stateToString(_tasks.at(i)->getState())
	   << std::endl;*/
	 if(_tasks.at(i)->getState() == READY || _tasks.at(i)->getState() == RUN)
	 { return _tasks.at(i); }
      }
      return 0;
   }
   

   bool ProcessingElement::hasInPort(unsigned long int port)
   {
      for(unsigned int i = 0; i < _tasks.size(); ++i)
      {
	 if(_tasks.at(i)->hasInPort(port)) {return true;}
      }
      return false;
   }
  
   void ProcessingElement::receiveEvent(tgToken token)
   {
      /*std::cout << "PE " << getName() << " got token from event to port "
	<< token.destination << std::endl;*/
      // Forward token to all tasks with correct in_port

      _config.addSendTime(token.source);

      for(unsigned int i = 0; i < _tasks.size(); ++i)
      {
	 if(_tasks.at(i)->hasInPort(token.destination))
	 {
	    _tasks.at(i)->receive(token);
	    _fifoScheduler.push(_tasks.at(i));
	 }
      }
      _eventEvent.notify(sc_core::SC_ZERO_TIME);
   }

   const sctg::PeMeasurements& ProcessingElement::getMeasurements()
   {
      updateMeasurements();

      return _measurements;
   }

   void ProcessingElement::updateMeasurements()
   {
      _measureEnd   = sc_core::sc_time_stamp();

      switch(_currentState)
      {
	 case IDLE:	
	    _measurements.idleTime += 
	       _measureEnd - _measureStart;	
	    break;
	 case EXECUTING:	
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.execTime += 
	       _measureEnd - _measureStart;
	    break;
	 case INTRA_TX_WAIT:	    
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.intraTxWait +=
	       _measureEnd - _measureStart;
	    break;
	 case TX_WAIT:
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.txWaitTime +=
	       _measureEnd - _measureStart;
	    break;
	 case RX_WAIT:
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.rxWaitTime +=
	       _measureEnd - _measureStart;
	    break;
	 case I_CACHE_MISSING:
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.iCacheTime +=
	       _measureEnd - _measureStart;
	    break;
	 case D_CACHE_MISSING:
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.dCacheTime +=
	       _measureEnd - _measureStart;
	    break;
	 case READING:
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.readingTime +=
	       _measureEnd - _measureStart;
	    break;
	 case SENDING:
	    _measurements.busyTime += 
	       _measureEnd - _measureStart;
	    _measurements.sendingTime +=
	       _measureEnd - _measureStart;
	    break;
	 default:
	    break;
      }

      _measurements.cycleLength = _cycleLength;

      _measurements.idleCycles = 
	 static_cast<unsigned long int>(_measurements.idleTime/_cycleLength);
      _measurements.busyCycles = 
	 static_cast<unsigned long int>(_measurements.busyTime/_cycleLength);
    
      _measureStart = _measureEnd;

   }

   double ProcessingElement::getAvgUtilization()
   {
      if(_measurements.busyTime+_measurements.idleTime != sc_core::SC_ZERO_TIME)
      {
	 return (_measurements.busyTime / 
		 (_measurements.busyTime + _measurements.idleTime));
      }
      else
      {
	 return 0.0;
      }
   }

   void ProcessingElement::txDma()
   {
      tgPacket* packet = 0;
      double waitTime  = 0.0;

      while(true)
      {
	 if(_txDmaQueue.empty())
	 {
	    wait(_txDmaEvent);
	 }

	 while(!_txDmaQueue.empty())
	 {
	    packet = _txDmaQueue.front();
	    _txDmaQueue.pop();

	    while(_buffer->txSpaceLeft() < packet->size)
	    {				       
	       wait(*(_buffer->txGetReadEvent()));			
	    }	    	    

	    _buffer->txReserveSpace(packet->size);

	    waitTime =
	       (1.0 / getFrequency()) * 
	       (packet->size / (getTerminalWidth(getTerminal())/8));
	    
	    wait(waitTime, sc_core::SC_US);

	    _buffer->txFreeSpace(packet->size);
	    _buffer->txPutPacket(packet);
	    
	 }
      }
   }

   void ProcessingElement::rxDma()
   {
      tgPacket* packet = 0;
      double waitTime  = 0.0;

      while(true)
      {
	 if(_rxDmaQueue.empty())
	 {
	    wait(_rxDmaEvent);
	 }

	 while(!_rxDmaQueue.empty())
	 {
	    packet = _rxDmaQueue.front();
	    _rxDmaQueue.pop();
	    
	    while(_buffer->rxSpaceLeft() < packet->size)
	    {
	       wait(*(_buffer->rxGetReadEvent()));
	    }

	    //std::cout << "rxDma" << std::endl;
	    
	    _buffer->rxReserveSpace(packet->size);
	    
	    waitTime =
	       (1.0 / getFrequency()) * 
	       (packet->size / (getTerminalWidth(getTerminal())/8));
	    
	    wait(waitTime, sc_core::SC_US);
	    
	    _buffer->rxFreeSpace(packet->size);
	       
	    _buffer->addInternalBytes(packet->size);
	    _buffer->rxPutPacket(packet);
	    
	 }
      }
   }

   std::queue<tgToken>& ProcessingElement::getTokenQueue()
   {
      return _tokenQueue;
   }

   void ProcessingElement::updateUnfinishedTokensLatency()
   {
      _buffer->updateUnfinishedTokensLatency();
   }

   void ProcessingElement::mapResourceUser(ResourceUser* task)
   {
      const int prLen = 12;
      std::string taskNameForPrint = task->getName().substr(0,prLen-1);
      while (taskNameForPrint.length() < prLen) {
	 taskNameForPrint.append(" ");
      }


//      std::cout << "Task " << task->getName() << " mapped to PE " //ES
      std::cout << "Task " << taskNameForPrint << " mapped to PE " //ES
		<< getName() << std::endl;
      _tasks.push_back(dynamic_cast<Task*>(task));
   }

}



// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
