/**
 *
 * @file configuration.cc
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: configuration.cc 1953 2011-08-22 08:38:32Z lehton87 $
 *
 */


#include "configuration.hh"
#include "buffer.hh"
#include <boost/lexical_cast.hpp>
#include <iostream>
#include <sstream>
#include <exception>
#include <ctime>
#include <iostream>

namespace sctg
{
   Configuration::Configuration(const boost::property_tree::ptree& ptroot,
				std::string& saveDir)
   {
      //using boost::property_tree::ptree;
      boost::property_tree::ptree pt = ptroot.get_child("constraints");
      boost::property_tree::ptree nocTree = 
	 ptroot.get_child("platform.noc.<xmlattr>");

      // Parse NoC generics
      for(boost::property_tree::ptree::const_iterator iter = nocTree.begin();
 	  iter != nocTree.end(); ++iter)
      {
	 
	 try
	 {
	    _generics[(*iter).first] = 
	       boost::lexical_cast<int>((*iter).second.data());
	 }
	 catch(boost::bad_lexical_cast &)
	 {
	 }
      }
      
      // Parse and set simulation's resolution
      double time = pt.get<double>("sim_resolution.<xmlattr>.time");
      std::string unit = pt.get<std::string>("sim_resolution.<xmlattr>.unit", 
					     "ns");
      sc_core::sc_set_time_resolution(time, parseUnit(unit));
      _simResolution = sc_core::sc_time(time, parseUnit(unit));

      // Parse simulation's length
      time = pt.get<double>("sim_length.<xmlattr>.time");
      unit = pt.get<std::string>("sim_length.<xmlattr>.unit", "ns");
      _simLength = sc_core::sc_time(time, parseUnit(unit));
    
      // Parse measurement interval
      time = pt.get<double>("measurements.<xmlattr>.time", 0.0);
      unit = pt.get<std::string>("measurements.<xmlattr>.unit", "ms");
      _measurementInterval = sc_core::sc_time(time, parseUnit(unit));

      _useExecMon = false;

      // Parse processing element's filename
      _peLibFile = pt.get<std::string>("pe_lib.<xmlattr>.file");

      // Parse cost functions
      for(boost::property_tree::ptree::const_iterator iter = pt.begin();
	  iter != pt.end(); ++iter)
      {
	 if((*iter).first == "cost_function")
	 {
	    _costFunctions.push_back((*iter).second.
				     get<std::string>("<xmlattr>.func"));
	 }
      }

      // Parse paths
      for(boost::property_tree::ptree::const_iterator iter = pt.begin();
	  iter != pt.end(); ++iter)
      {
	 if((*iter).first == "path_measurement")
	 {
	    unsigned long int src = 
	       (*iter).second.get<unsigned long int>("<xmlattr>.src");
	    unsigned long int dst = 
	       (*iter).second.get<unsigned long int>("<xmlattr>.dst");

	    _pathsFwd.insert(std::pair<unsigned long int, unsigned long int>
			     (src, dst));
	    _pathsRev.insert(std::pair<unsigned long int, unsigned long int>
			     (dst, src));	    
	 }
      }

      // Parse Task count measurements
      for(boost::property_tree::ptree::const_iterator iter = pt.begin();
	  iter != pt.end(); ++iter)
      {
	 if((*iter).first == "task_measurement")
	 {
	    unsigned long int task = 
	       (*iter).second.get<unsigned long int>("<xmlattr>.task");
	    unsigned long int count = 
	       (*iter).second.get<unsigned long int>("<xmlattr>.count");

	    _taskTriggerTimes.insert(std::pair<unsigned long int, unsigned long int>
				     (task, count));
	 }
      }

      // Parse RNG seed, using time if not available
      boost::optional<unsigned long int> seed = 
	 pt.get_optional<unsigned long int>("rng_seed.<xmlattr>.value");
    
      if(seed)
      { _seed = *seed; }
      else
      { _seed = static_cast<unsigned long int>(std::time(0)); }
    
      _intEngine.seed(_seed);
      _realEngine.seed(_seed);

      _nextTokenId = 0;
    
    
      // Parse log file names and open them if available
      boost::optional<std::string> pktFile =
	 pt.get_optional<std::string>("log_packet.<xmlattr>.file");
      boost::optional<std::string> tokenFile =
	 pt.get_optional<std::string>("log_token.<xmlattr>.file");
      boost::optional<std::string> summaryFile =
	 pt.get_optional<std::string>("log_summary.<xmlattr>.file");
      boost::optional<std::string> peFile =
	 pt.get_optional<std::string>("log_pe.<xmlattr>.file");
      boost::optional<std::string> memFile =
	 pt.get_optional<std::string>("log_mem.<xmlattr>.file");
      boost::optional<std::string> appFile =
	 pt.get_optional<std::string>("log_app.<xmlattr>.file");
      boost::optional<std::string> execMonFile =
	 pt.get_optional<std::string>("log_exec_mon.<xmlattr>.file");

      if(saveDir.size() > 0 
	 && saveDir.at(saveDir.size()-1) != '/'
	 && saveDir.at(saveDir.size()-1) != '\\')
      {
	 saveDir.append("/");
      }

      if(pktFile)     
      {
	 (*pktFile) = saveDir + (*pktFile);
	 _pktOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*pktFile).c_str())); 
      }
      if(tokenFile)   
      {
	 (*tokenFile) = saveDir + (*tokenFile);
	 _tokenOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*tokenFile).c_str()));
      }
      if(summaryFile) 
      {
	 (*summaryFile) = saveDir + (*summaryFile);
	 _summaryOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*summaryFile).c_str()));
      }
      if(peFile)      
      {
	 (*peFile) = saveDir + (*peFile);
	 _peOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*peFile).c_str()));
      }
      if(memFile)      
      {
	 (*memFile) = saveDir + (*memFile);
	 _memOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*memFile).c_str()));
      }
      if(appFile)
      {
	 (*appFile) = saveDir + (*appFile);
	 _appOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*appFile).c_str()));
      }
      if(execMonFile)
      {
	 (*execMonFile) = saveDir + (*execMonFile);
	 _execMonOutFile = boost::optional<std::ostream*>
	    (new std::ofstream((*execMonFile).c_str()));
      }

#ifdef SCTG_USE_EXECMON
      _serverIf = 0;
#endif
    
   }

   Configuration::~Configuration()
   {
      if(_pktOutFile)     {delete (*_pktOutFile);     *_pktOutFile     = 0;}
      if(_tokenOutFile)   {delete (*_tokenOutFile);   *_tokenOutFile   = 0;}
      if(_summaryOutFile) {delete (*_summaryOutFile); *_summaryOutFile = 0;}
      if(_peOutFile)      {delete (*_peOutFile);      *_peOutFile      = 0;}
      if(_appOutFile)     {delete (*_appOutFile);     *_appOutFile     = 0;}
   }
  
   const sc_core::sc_time& Configuration::getSimLength() const
   { return _simLength; }

   const sc_core::sc_time& Configuration::getSimResolution() const
   { return _simResolution; }

   const std::string& Configuration::getPeLibFile() const
   { return _peLibFile; }

   Configuration::RealEngineType& Configuration::getRealRNGEngine()
   { return _realEngine; }

   Configuration::IntEngineType& Configuration::getIntRNGEngine()
   { return _intEngine; }

   const unsigned long int Configuration::getSeed() const
   { return _seed; }

   double Configuration::random()
   { return _random(_realEngine); }

   const sc_core::sc_time_unit Configuration::parseUnit(std::string unit)
   {
      if(unit == "ns")
      {	return sc_core::SC_NS; }
      else if(unit == "ps")
      {	return sc_core::SC_PS; }
      else if(unit == "us")
      {	return sc_core::SC_US; }
      else if(unit == "ms")
      {	return sc_core::SC_MS; }
      else if(unit == "fs")
      {	return sc_core::SC_FS; }
      else if(unit == "s")
      {	return sc_core::SC_SEC;}
      else
      {
	 std::string errstr = "Unregonized time unit " + unit;
	 throw std::runtime_error(errstr.c_str());
      }
   }

   void Configuration::addPortConnection(unsigned long int source,
					 unsigned long int destination)
   {
      _portConnections[source] = destination;
      _revPortConnections[destination] = source;
   }

   unsigned long int Configuration::getDestination(unsigned long int source)
   {
      if(_portConnections.find(source) == _portConnections.end())
      {
	 std::ostringstream oss;
	 oss << "There's no destination for out_port " << source;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }

      return _portConnections[source];
   }

   unsigned long int Configuration::getSource(unsigned long int dest)
   {
      if(_revPortConnections.find(dest) == _revPortConnections.end())
      {
	 std::ostringstream oss;
	 oss << "There's no source for in_port " << dest;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }

      return _revPortConnections[dest];
   }

   void Configuration::addResourceMap(unsigned long int id, Resource* pe)
   {
      _resourceMap[id] = pe;
   }
    
  
   Resource* Configuration::getResource(unsigned long int id)
   {
      if(_resourceMap.find(id) == _resourceMap.end())
      {
	 std::ostringstream oss;
	 oss << "There's no resource with id " << id;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _resourceMap[id];
   }

   void Configuration::addResourceUserMap(unsigned long int id, 
					  ResourceUser* task)
   {
      _resourceUserMap[id] = task;
   }

   ResourceUser* Configuration::getResourceUser(unsigned long int id)
   {
      if(_resourceUserMap.find(id) == _resourceUserMap.end())
      {
	 std::ostringstream oss;
	 oss << "There's no task or mem_area with id " << id;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _resourceUserMap[id];
   }

   void Configuration::addResourceUserToResourceMap(unsigned long int id, 
						    Resource* pe)
   {
      _resourceResourceUserMap[id] = pe;
   }
    
  
   Resource* Configuration::getResourceByResourceUser(unsigned long int id)
   {
      if(_resourceResourceUserMap.find(id) == _resourceResourceUserMap.end())
      {
	 std::ostringstream oss;
	 oss << "Task id:" << id << " not mapped to any resource";
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _resourceResourceUserMap[id];
   }
    
   void Configuration::addResourceUserToInPortMap(unsigned long int id, 
						  ResourceUser* task)
   {
      _resourceUserInPortMap[id] = task; 
   }

   ResourceUser* Configuration::getResourceUserByInPort(unsigned long int id)
   {
      if(_resourceUserInPortMap.find(id) == _resourceUserInPortMap.end())
      {
	 std::ostringstream oss;
	 oss << "There's no task or mem_area with in_port " << id;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _resourceUserInPortMap[id];
   }

   Buffer* Configuration::getBufferByResource(unsigned long int pe)
   {
      if(_resourceBufferMap.find(pe) == _resourceBufferMap.end())
      {
	 std::ostringstream oss;
	 oss << "No buffer found for resource " << pe;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _resourceBufferMap[pe];
   }
  
   unsigned int Configuration::getTokenId()
   {
      return _nextTokenId++;
   }
      
   void Configuration::addBufferToResource(Buffer* buffer, unsigned long int pe)
   {
      _resourceBufferMap[pe] = buffer;
   }

   BufferInterface* Configuration::getBufferIf(unsigned long int agent)
   {
      if(_resourceBufferMap.find(agent) == _resourceBufferMap.end())
      {
	 std::ostringstream oss;
	 oss << "No buffer found for agent " << agent;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _resourceBufferMap[agent];
   }

   void Configuration::addTaskToGroup(unsigned long int task,
				      unsigned long int group)
   {
      _taskGroupMap[task] = group;
   }
   
   void Configuration::addGroupToPe(unsigned long int group,
				    unsigned long int pe)
   {
      _groupPeMap[group] = pe;
   }

   unsigned long int Configuration::getGroup(unsigned long int task)
   {
      if(_taskGroupMap.find(task) == _taskGroupMap.end())
      {
	 std::ostringstream oss;
	 oss << "No group found for task " << task;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _taskGroupMap[task];
   }

   unsigned long int Configuration::getPeByGroup(unsigned long int group)
   {
      if(_groupPeMap.find(group) == _groupPeMap.end())
      {
	 std::ostringstream oss;
	 oss << "No PE found for group " << group;
	 std::cout << oss.str() << std::endl;
	 throw std::runtime_error(oss.str().c_str());
      }
      return _groupPeMap[group];
   }

   const sc_core::sc_time& Configuration::getMeasurementInterval() const
   {
      return _measurementInterval;
   }

   boost::optional<std::ostream*>& Configuration::getPacketStream()
   {
      return _pktOutFile;
   }

   boost::optional<std::ostream*>& Configuration::getTokenStream()
   {
      return _tokenOutFile;
   }

   boost::optional<std::ostream*>& Configuration::getSummaryStream()
   {
      return _summaryOutFile;
   }

   boost::optional<std::ostream*>& Configuration::getPeStream()
   {
      return _peOutFile;
   }

   boost::optional<std::ostream*>& Configuration::getMemStream()
   {
      return _memOutFile;
   }
   
   boost::optional<std::ostream*>& Configuration::getAppStream()
   {
      return _appOutFile;
   }

   boost::optional<std::ostream*>& Configuration::getExecMonStream()
   {
      return _execMonOutFile;
   }

   const std::string& Configuration::getNocClass()
   {
      return _nocClass;
   }
   
   const std::string& Configuration::getNocType()
   {
      return _nocType;
   }

   const std::string& Configuration::getNocSubType()
   {
      return _nocSubType;
   }

   const std::map<std::string, int>& Configuration::getGenerics()
   {
      return _generics;
   }

   void Configuration::setNocClass(std::string& s)
   {
      _nocClass = s;
   }
   
   void Configuration::setNocType(std::string& s)
   {
      _nocType = s;
   }

   void Configuration::setNocSubType(std::string& s)
   {
      _nocSubType = s;
   }

   std::map<unsigned long int, Resource*>::iterator 
   Configuration::getResourceBegin()
   {
      return _resourceMap.begin();
   }

   std::map<unsigned long int, Resource*>::iterator 
   Configuration::getResourceEnd()
   {
      return _resourceMap.end();
   }

   std::map<unsigned long int, unsigned long int>::iterator 
   Configuration::getGroupPeBegin()
   {
      return _groupPeMap.begin();
   }

   std::map<unsigned long int, unsigned long int>::iterator 
   Configuration::getGroupPeEnd()
   {
      return _groupPeMap.end();
   }

   std::map<unsigned long int, unsigned long int>::iterator 
   Configuration::getTaskGroupBegin()
   {
      return _taskGroupMap.begin();
   }

   std::map<unsigned long int, unsigned long int>::iterator 
   Configuration::getTaskGroupEnd()
   {
      return _taskGroupMap.end();
   }

   void Configuration::useExecMon(bool use)
   {
      _useExecMon = use;
   }

   
   bool Configuration::useExecMon()
   {
      return _useExecMon;
   }

#ifdef SCTG_USE_EXECMON      
   void Configuration::setTcpServer(TcpServerIf* server)
   {
      _serverIf = server;
   }

   TcpServerIf* Configuration::getTcpServer()
   {
      return _serverIf;
   }
#endif

   std::vector<std::string>& Configuration::getCostFunctions()
   {
      return _costFunctions;
   }

   void Configuration::addTokenLatency(unsigned long int src,
				       unsigned long int dst,
				       sc_core::sc_time& latency)
   {
      std::map<unsigned long int, 
	       std::map<unsigned long int, sc_core::sc_time> >::iterator outer;
      std::map<unsigned long int, sc_core::sc_time>::iterator inner;

      _tokenLatency[src][dst] += latency;
      _tokenCount[src][dst]   += 1;
      if(_tokenLatencyMax[src][dst] < latency)
      {
	 _tokenLatencyMax[src][dst] = latency;
      }
      
      if(_tokenLatencyMin.find(src) != _tokenLatencyMin.end() &&
	 _tokenLatencyMin[src].find(dst) != _tokenLatencyMin[src].end())
      {
	 if(_tokenLatencyMin[src][dst] > latency)
	 {
	    _tokenLatencyMin[src][dst] = latency;
	 }
      }
      else
      {
	 _tokenLatencyMin[src][dst] = latency;
      }

   }

   std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
   Configuration::getTokenLatency()
   {
      return _tokenLatency;
   }

   std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
   Configuration::getTokenLatencyMax()
   {
      return _tokenLatencyMax;
   }

   std::map<unsigned long int, std::map<unsigned long int, sc_core::sc_time> >&
   Configuration::getTokenLatencyMin()
   {
      return _tokenLatencyMin;
   }

   std::map<unsigned long int, std::map<unsigned long int, unsigned long int> >&
   Configuration::getTokenCount()
   {
      return _tokenCount;
   }

   void Configuration::addSendTime(unsigned long int port)
   {
      // if(_pathsFwd.find(port) == _pathsFwd.end())
      // {
      // 	 return; // No path declared
      // }

      for(std::multimap<unsigned long int, unsigned long int>::iterator iter =
	     _pathsFwd.lower_bound(port); iter != _pathsFwd.upper_bound(port); 
	  ++iter)
      {
	 std::ostringstream oss;
	 oss << port << "_" << (*iter).second;
	 _portSendTime[oss.str()].push(sc_core::sc_time_stamp());      
      }
      
   }

   void Configuration::addReceiveTime(unsigned long int port)
   {
      if(_pathsRev.find(port) == _pathsRev.end())
      {
	 return; // NO path declared
      }
      
      for(std::multimap<unsigned long int, unsigned long int>::iterator iter =
	     _pathsRev.lower_bound(port); iter != _pathsRev.upper_bound(port);
	  ++iter)
      {
	 std::ostringstream oss;
	 oss << (*iter).second << "_" << port;

	 if(_portSendTime[oss.str()].empty())
	 {
	    std::ostringstream ossi;
	    ossi << "Path measurements: destination port"
		 << " reached but no send time for source port for path " 
		 << oss.str()
		 << " (check that there's only one token received per"
		 << " every sent one)";
	    std::cout << ossi.str() << std::endl;
	    throw std::runtime_error(ossi.str().c_str());
	 }

	 sc_core::sc_time lat = sc_core::sc_time_stamp() - 
	    _portSendTime[oss.str()].front();
      
	 _portSendTime[oss.str()].pop();

	 _pathCount[oss.str()] += 1;
	 _totPathLat[oss.str()] += lat;

	 if(_maxPathLat[oss.str()] < lat)
	 { _maxPathLat[oss.str()] = lat; }
      
	 if(_minPathLat.find(oss.str()) == _minPathLat.end() ||
	    _minPathLat[oss.str()] > lat)
	 { _minPathLat[oss.str()] = lat; }


      }


      // unsigned long int source = _pathsRev[port];

      // if(_portSendTime[source].empty())
      // {
      // 	 std::ostringstream oss;
      // 	 oss << "Path measurements: destination port with id " << port
      // 	     << " reached but no send time for source port " << source
      // 	     << " (check that there's only one token received per"
      // 	     << " every sent one)";
      // 	 std::cout << oss.str() << std::endl;
      // 	 throw std::runtime_error(oss.str().c_str());
      // }

      // sc_core::sc_time lat = sc_core::sc_time_stamp() - 
      // 	 _portSendTime[source].front();
      
      // _portSendTime[source].pop();

      // _pathCount[port] += 1;
      // _totPathLat[port] += lat;

      // if(_maxPathLat[port] < lat)
      // { _maxPathLat[port] = lat; }
      
      // if(_minPathLat.find(port) == _minPathLat.end() ||
      // 	 _minPathLat[port] > lat)
      // { _minPathLat[port] = lat; }

   }
   
   std::map<std::string, sc_core::sc_time>& 
   Configuration::getTotPathLat()
   {
      return _totPathLat;
   }

   std::map<std::string, sc_core::sc_time>& 
   Configuration::getMaxPathLat()
   {
      return _maxPathLat;
   }

   std::map<std::string, sc_core::sc_time>& 
   Configuration::getMinPathLat()
   {
      return _minPathLat;
   }

   std::map<std::string, unsigned long int>& 
   Configuration::getPathCount()
   {
      return _pathCount;
   }

   void Configuration::taskTriggered
   (unsigned long int task, unsigned long int times)
   {
      for(std::multimap<unsigned long int, unsigned long int>::iterator iter =
	     _taskTriggerTimes.lower_bound(task); 
	  iter != _taskTriggerTimes.upper_bound(task); ++iter)
      {
	 if((*iter).second == times)
	 {
	    std::ostringstream oss;
	    oss << "tt_" << task << "_" << times;
	    _costVariables[oss.str()] = sc_core::sc_time_stamp() / 
	       sc_core::sc_time(1000.0, sc_core::SC_MS);
	 }
      }
   }
   
   std::map<std::string, double>& Configuration::getCostVariables()
   { 
      return _costVariables;
   }

}


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
