/**
 *
 * @file common.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it
 *  and/or modify it under the terms of the Lesser GNU General Public
 *  License as published by the Free Software Foundation, either
 *  version 3 of the License, or (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be
 *  useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *  See the Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public
 *  License along with Transaction Generator.  If not, see
 *  <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: common.hh 1916 2011-07-06 12:44:26Z lehton87 $
 *
 */

#ifndef SCTG_COMMON_HH
#define SCTG_COMMON_HH

#include <string>
#include <systemc>

namespace sctg
{
   /** Defines the state of processing element
    */
   enum OperationType  {EXECUTE, SEND, READ, WAIT_RESP, POLL};

   /** Defines the state of a task
    */
   enum State {RUN, READY, WAIT, FREE};

   /** Measurement data from processing elements
    */
   struct PeMeasurements
   {
      unsigned long int idleCycles;   ///< Number of idle clock cycles
      unsigned long int busyCycles;   ///< Number of cycles used for execution
      unsigned long int iCacheMisses; ///< How many I-cache misses has happened
      unsigned long int dCacheMisses; ///< How many D-cache misses has happened
      sc_core::sc_time idleTime;      ///< Time idled
      sc_core::sc_time busyTime;      ///< Time used for executing programs
      sc_core::sc_time execTime;      ///< Time used for executing programs
      sc_core::sc_time readingTime;   ///< Time spent for reading
      sc_core::sc_time sendingTime;   ///< Time spent for sending data
      sc_core::sc_time writingTime;   ///< Time spent for writing data to mem
      sc_core::sc_time rxWaitTime;    ///< Time used to wait read response
      sc_core::sc_time txWaitTime;    ///< Time waited for tx buffer to empty
      sc_core::sc_time intraTxWait;   ///< Time used for intra PE communication
      sc_core::sc_time cacheTime;     ///< Time used for cache reading
      sc_core::sc_time iCacheTime;    ///< Time used because of I-cache misses
      sc_core::sc_time dCacheTime;    ///< Time used because of D-cache misses
      sc_core::sc_time cycleLength;   ///< Length of PE's clock cycle

      
      PeMeasurements()
	 : idleCycles(0),
	   busyCycles(0),
	   iCacheMisses(0),
	   dCacheMisses(0)
      {
      }
   };

   /** Measurement data from buffers
    */
   struct BufferMeasurements
   {
      unsigned long int rxBytes;       ///< Bytes received
      unsigned long int txBytes;       ///< Bytes sent
      unsigned long int internalBytes; ///< Bytes sent in PE
      unsigned long int rxUsed;        ///< Receive buffer used
      unsigned long int txUsed;        ///< Transfer buffer used

      BufferMeasurements()
      {
	 rxBytes = 0;
	 txBytes = 0;
	 internalBytes = 0;
	 rxUsed = 0;
	 txUsed = 0;
      };
   };

   /** Converts State enum values to strings
    */
   std::string stateToString(State state);

}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:
