/**
 *
 * @file sctg_to_osci_tlm.hh
 * @author Lasse Lehtonen
 *
 *
 */

/*
 * Copyright 2010 Tampere University of Technology
 * 
 *  This file is part of Transaction Generator.
 *
 *  Transaction Generator is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Transaction Generator is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Transaction Generator.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Id: sctg_to_osci_tlm.hh 1399 2010-08-26 13:56:45Z lehton87 $
 *
 */

#ifndef SCTG_TO_OSCI_TLM_HH
#define SCTG_TO_OSCI_TLM_HH

#include "noc_conf_if.hh"
#include "buffer_if.hh"
#include "tg_packet.hh"

#include "tlm.h"
#ifdef MTI_SYSTEMC
#include "simple_initiator_socket.h"
#include "simple_target_socket.h"
#include "peq_with_get.h"
#else
#include "tlm_utils/simple_initiator_socket.h"
#include "tlm_utils/simple_target_socket.h"
#include "tlm_utils/peq_with_get.h"
#endif
#include <systemc>

#include <stdexcept>
#include <iostream>


namespace sctg
{
   template<unsigned int N_AGENTS,
	    unsigned int DATA_WIDTH>
   class SctgToOsciTlm : public sc_core::sc_module, public tlm::tlm_mm_interface
   {
   public:

      SC_HAS_PROCESS(SctgToOsciTlm);
      
      tlm_utils::simple_initiator_socket_tagged
      <SctgToOsciTlm, DATA_WIDTH> initSockets[N_AGENTS];
      tlm_utils::simple_target_socket_tagged
      <SctgToOsciTlm, DATA_WIDTH> targetSockets[N_AGENTS];      

      void free(tlm::tlm_generic_payload* pl)
      {
	 delete pl;
      }

      //* Constructor
      SctgToOsciTlm(sc_core::sc_module_name name,
		    sctg::NocConfIf* confIf)
	 : sc_module(name),
	   _confIf(confIf),
	   _cycleTime(20, sc_core::SC_NS)
      {

	 for(unsigned int i = 0; i < N_AGENTS; ++i)
	 {
	    // Generating PEQ and threads for every agent

	    std::ostringstream oss;
	    oss << "rxPeq_" << i;
	    _rxPeq[i] = new tlm_utils::peq_with_get<tlm::tlm_generic_payload>
	       (oss.str().c_str());
	    initSockets[i].register_nb_transport_bw
	       (this, &SctgToOsciTlm::nb_transport_bw, i);
	    targetSockets[i].register_nb_transport_fw
	       (this, &SctgToOsciTlm::nb_transport_fw, i);
	    sc_spawn(sc_bind(&SctgToOsciTlm::sender, this, i));
	    sc_spawn(sc_bind(&SctgToOsciTlm::receiver, this, i));
	    
	 }
      }

      //* Destructor
      ~SctgToOsciTlm()
      {
	 for(unsigned int i = 0; i < N_AGENTS; ++i)
	 {	    
	    delete _rxPeq[i]; _rxPeq[i] = 0;
	 }
      }

   private:

      // Disable operator= and copy constructor
      SctgToOsciTlm& operator=(const SctgToOsciTlm& rhs);
      SctgToOsciTlm(const SctgToOsciTlm& rhs);

      void sender(unsigned int agent)
      {
	 
	 // Get pointer to buffer between IP -> NoC
	 sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);

	 // If buffer doesn't point anywhere there is no agent with this ID
	 if(!buffer)
	 { return; }

	 // struct we get from TG
	 tgPacket* packet = 0;

	 // std::ostringstream oss;
	 // oss << "at" << agent << "_sender.txt";
	 // std::ofstream ofs(oss.str().c_str());

	 // TLM structures we're sending to NoC
	 tlm::tlm_generic_payload* trans = 0;
	 tlm::tlm_phase            phase;
	 sc_core::sc_time          delay;

	 tlm::tlm_sync_enum        retval;
	 	 	 

	 while(true)
	 {

	    // Wait for packets to send
	    if(!buffer->txPacketAvailable())
	    {	       
	       sc_core::wait(*(buffer->txGetPacketAvailableEvent()));
	    }

	    // ofs << "Agent " << agent << " got packet  "
	    // 	   << sc_core::sc_time_stamp().value() << std::endl;
	    	    
	    // Read packet from buffer
	    packet = buffer->txGetPacket();
	    	    
	    trans = new tlm::tlm_generic_payload;

	    // Initialize generic payload
	    trans->set_command(tlm::TLM_WRITE_COMMAND);
	    trans->set_data_length(packet->size);
	    trans->set_byte_enable_ptr(0);
	    trans->set_streaming_width(DATA_WIDTH / 8);
	    trans->set_mm(this);
	    trans->set_address(packet->address); 	    
	    trans->set_data_ptr(packet->data); 
	    trans->set_dmi_allowed(false);	    
	    trans->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
	    phase = tlm::BEGIN_REQ;
	    delay = sc_core::SC_ZERO_TIME;

	    // Send packet
	    trans->acquire();
	    retval = initSockets[agent]->nb_transport_fw(*trans, phase, delay);
	    
	    // Check for errors
	    if(trans->is_response_error())
	    {
	       std::ostringstream oss;
	       oss << "Sender " << agent << " got error response: "
		   << trans->get_response_string().c_str();
	       throw std::runtime_error(oss.str().c_str());
	    }

	    if(retval == tlm::TLM_COMPLETED)
	    {
	       //std::cout << "ADAPTER::Sender " << agent << " got TLM_COMPLETED" << std::endl;
	       if(delay != sc_core::SC_ZERO_TIME)
	       {
		  wait(delay);
	       }
	    }
	    else if(retval == tlm::TLM_UPDATED || retval == tlm::TLM_ACCEPTED)
	    {
	       if(phase == tlm::BEGIN_REQ)
	       {
		  //std::cout << "ADAPTER::Sender " << agent << " waiting for END_REQ or BEGIN_RESP" 
		  //<< std::endl;
		  // Wait for END_REQ or BEGIN_RESP
		  wait(_txCompleteEvent[agent]);
	       }
	       else if(phase == tlm::END_REQ)
	       {
		  //std::cout << "ADAPTER::Sender " << agent << " in END_REQ" << std::endl;
		  
		  // BEGIN_RESP will come via backward path
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
		  
		  wait(_txCompleteEvent[agent]);

	       }
	       else if(phase == tlm::BEGIN_RESP)
	       {
		  if(delay != sc_core::SC_ZERO_TIME)
		  {
		     wait(delay);
		  }
		  
		  //std::cout << "ADAPTER::Sender " << agent << " sending END_RESP" << std::endl;

		  // Send END_RESP immediately
		  trans->set_dmi_allowed(false);
		  trans->set_response_status(tlm::TLM_INCOMPLETE_RESPONSE);
		  phase = tlm::END_RESP;
		  delay = sc_core::SC_ZERO_TIME;
		  retval = initSockets[agent]->
		     nb_transport_fw(*trans, phase, delay);

		  // Check for errors
		  if(trans->is_response_error())
		  {
		     std::ostringstream oss;
		     oss << "Sender " << agent << " got error response: "
			 << trans->get_response_string().c_str();
		     throw std::runtime_error(oss.str().c_str());
		  }

		  // Transaction completed
		  
	       }
	       else
	       {
		  std::ostringstream oss;
		  oss << "Sender " << agent << " got invalid TLM_UPDATED";
		  throw std::runtime_error(oss.str().c_str());
	       }

	    }
	    
	    trans->release();

	    // ofs << "Agent " << agent << " sent packet "
	    // 	<< sc_core::sc_time_stamp().value() << std::endl;

	    // Free memory used by the packet
	    packet->data = 0;
	    delete   packet;       packet       = 0;

	 } // End of while(true)

      }

      void receiver(unsigned int agent)
      {	 
	 // Get pointer to buffer between NoC -> IP
	 sctg::BufferInterface* buffer = _confIf->getBufferIf(agent);

	 tlm::tlm_generic_payload* trans;
	 tlm::tlm_phase            phase = tlm::END_REQ;
	 sc_core::sc_time          delay = sc_core::SC_ZERO_TIME;

	 tlm::tlm_sync_enum        retval;

	 tgPacket* packet = 0;

	 // std::ostringstream oss;
	 // oss << "at" << agent << "_receiver.txt";
	 // std::ofstream ofs(oss.str().c_str());
	 
	 // If buffer doesn't point anywhere there is no agent with this ID
	 if(!buffer)
	 { return; }

	 //std::cout << "ADAPTER::Receiver " << agent << " ready" << std::endl;
	 
	 while(true)
	 {
	    if((trans = _rxPeq[agent]->get_next_transaction()) == 0)
	    {	       
	       wait(_rxPeq[agent]->get_event());
	       trans = _rxPeq[agent]->get_next_transaction();
	    }	    

	    // ofs << "Agent " << agent << " getting addr at "
	    // 	<< sc_core::sc_time_stamp().value() << std::endl;

	    packet = new tgPacket;
	    packet->address = trans->get_address();
	    packet->size    = trans->get_data_length();
	    packet->data    = trans->get_data_ptr();

	    /*
	    std::cout << "ADAPTER::Receiver " << agent 
		      << " receiving packet size " << packet->size
		      << " and mem_addr " << std::hex << &packet->data
		      << std::dec << std::endl;
	    */

	    unsigned int flits = 
	       trans->get_data_length() / trans->get_streaming_width();

	    // Wait one clock cycle per flit
	    //wait((flits+2) * _cycleTime);

	    // ofs << "Agent " << agent << " packet received "
	    // 	<< sc_core::sc_time_stamp().value() << std::endl;

	    // Wait until PE has space to receive packet
	    while(buffer->rxSpaceLeft() < packet->size)
	    {
	       wait(*(buffer->rxGetReadEvent()));
	    }
	    
	    buffer->rxPutPacket(packet);
	    packet = 0; // buffer handles freeing memory

	    phase = tlm::END_REQ;
	    delay = sc_core::SC_ZERO_TIME;

	    trans->set_response_status( tlm::TLM_OK_RESPONSE );
	    retval = targetSockets[agent]->nb_transport_bw(*trans, phase, delay);
	    trans->release();

	    // Assuming it returns TLM_COMPLETED as we are not using responces
	    if(retval != tlm::TLM_COMPLETED)
	    {
	       std::ostringstream oss;
	       oss << "SctgToOsciTlm::receiver " << agent
		   << ": was expecting TLM_COMPLETE";
	       throw std::runtime_error(oss.str().c_str());
	    }

	 } // End of while(true)

	 
      }

      tlm::tlm_sync_enum nb_transport_fw(int id,
					 tlm::tlm_generic_payload &trans,
					 tlm::tlm_phase           &phase,
					 sc_core::sc_time         &delay)
      {
	 //std::cout << "adapter::nb_transport_fw " << id << " called" << std::endl;

	 // Only write command is supported
	 if(trans.get_command() != tlm::TLM_WRITE_COMMAND)
	 {
	    std::ostringstream oss;
	    oss << "SctgToOsciTlm::nb_tranport_fw " << id 
		<< ": only write command is supported";
	    throw std::runtime_error(oss.str().c_str());
	 }
	 
	 if(phase == tlm::BEGIN_REQ)
	 {
	    trans.acquire();
	    _rxPeq[id]->notify(trans, delay);
	 }
	 
	 trans.set_response_status( tlm::TLM_OK_RESPONSE );
	 return tlm::TLM_ACCEPTED;
      }
      
      tlm::tlm_sync_enum nb_transport_bw(int id,
					 tlm::tlm_generic_payload &trans,
					 tlm::tlm_phase           &phase,
					 sc_core::sc_time         &delay)
      {

	 //std::cout << "adapter::nb_transport_bw " << id << " called" << std::endl;

	 if(phase == tlm::BEGIN_REQ || phase == tlm::END_RESP)
	 {
	    std::ostringstream oss;
	    oss << "nb_tranport_bw " << id << " got wrong phase";
	    throw std::runtime_error(oss.str().c_str());
	 }
	 
	 _txCompleteEvent[id].notify(delay);
	 
	 trans.set_response_status( tlm::TLM_OK_RESPONSE );
	 return tlm::TLM_COMPLETED;
      }

      // Pointer to configuration interface
      sctg::NocConfIf* _confIf;

      sc_core::sc_event _txCompleteEvent[N_AGENTS];
      
      tlm_utils::peq_with_get<tlm::tlm_generic_payload>* _rxPeq[N_AGENTS];

      sc_core::sc_time _cycleTime;

   };
}

#endif


// Local Variables:
// mode: c++
// c-file-style: "ellemtel"
// c-basic-offset: 3
// End:

