/**
 * File:    ProcessingElement.java
 * Author:  Tomi Jantti <tomi.jantti@tut.fi>
 * Created: 23.2.2007
 *
 *
 *
 * Copyright 2009 Tampere University of Technology
 * 
 *  This file is part of Execution Monitor.
 *
 *  Execution Monitor is free software: you can redistribute it and/or modify
 *  it under the terms of the Lesser GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Execution Monitor is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  Lesser GNU General Public License for more details.
 *
 *  You should have received a copy of the Lesser GNU General Public License
 *  along with Execution Monitor.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 */
package fi.cpu.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import fi.cpu.event.ModelNodeEvent;


/**
 * Class for processing element information.
 */
public class ProcessingElement extends ModelNode implements Comparable<ProcessingElement> {
	public static final int CHART_ADDED_EVENT = 1;
	public static final int CHART_REMOVED_EVENT = 2;

	private String id;
	private String name;
	private ProcessingElementType type;
	private boolean definesCharts;
	private String visibleChart = null;
	private Map<String, Chart> charts = new HashMap<String, Chart>();
	
	
	/**
	 * Creates a new ProcessingElement.
	 */
	public ProcessingElement(String id, String name, ProcessingElementType type) {
		super(name);
		this.id = id;
		this.name = name;
		this.type = type;
		
		definesCharts = false;
	}


	/**
	 * Returns the name.
	 */
	public String getName() {
		return name;
	}


	/**
	 * Sets the name.
	 */
	public void setName(String name) {
		this.name = name;
	}


	/**
	 * Returns the id.
	 */
	public String getId() {
		return id;
	}


	/**
	 * Returns the type.
	 */
	public ProcessingElementType getType() {
		return type;
	}

	
	/**
	 * @return Returns the definesCharts.
	 */
	public boolean definesCharts() {
		return definesCharts;
	}


	/**
	 * Sets the definesCharts.
	 */
	public void setDefinesCharts(boolean definesCharts) {
		this.definesCharts = definesCharts;
	}


	/**
	 * Returns the id of the visible chart.
	 */
	public String getVisibleChart() {
		return visibleChart;
	}


	/**
	 * Sets the visible chart.
	 */
	public void setVisibleChart(String visibleChart) {
		this.visibleChart = visibleChart;
	}


	/**
	 * Returns ids of charts.
	 */
	public Set<String> getChartIds() {
		return charts.keySet();
	}
	
	
	/**
	 * Returns charts.
	 */
	public List<Chart> getCharts() {
		return new ArrayList<Chart>(charts.values());
	}
	
	/**
	 * Sets charts
	 */
	public void setCharts(List<Chart> charts) {
		for (Chart chart : charts) {
			this.charts.put(chart.getId(), chart);
		}
	}
	
	/**
	 * Returns the chart with given id.
	 */
	public Chart getChart(String cid) {
		return charts.get(cid);
	}

	
	/**
	 * Sets the chart with given id.
	 */
	public void setChart(String cid, Chart chart) {
		removeChart(cid);

		if (chart == null) {
			return;
		}
		
		charts.put(cid, chart);
		definesCharts = true;
		fireModelNodeChanged(new ModelNodeEvent(this, CHART_ADDED_EVENT, cid));
	}
	
	
	/**
	 * Removes the chart with given id.
	 */
	public void removeChart(String cid) {
		Chart c = getChart(cid);
		charts.remove(cid);

		if (c != null) {
			fireModelNodeChanged(new ModelNodeEvent(this, CHART_REMOVED_EVENT, cid));
		}
	}

	
	/**
	 * Implements the Compararable-interface
	 */
	public int compareTo(ProcessingElement o) {
		if (type == o.type) {
			return name.compareTo(o.name);
		} else {
			if (type == ProcessingElementType.CPU) {
				return -1;
			} else {
				return 1;
			}
		}
	}
}
