///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2009-2010
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Herve Alexanian - Sonics, inc.
//
//          $Id:
//
//  Description :  This file defines a layer adapter module for TL1 sockets.
//                 Designed to play incoming OCP TLM transactions at the TL1
//                 level with certain user controls.
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

template<unsigned int BUSWIDTH> OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::ocp_tl1_tl3_slave_adapter (
    sc_core::sc_module_name name_, OCPIP_VERSION::ocp_layer_ids layer ) :
  sc_core::sc_module (name_)
  , m_timing_guard( "timing_guard" )
  , clk           ( "clk")
  , slave_socket  ( "slave", layer )
  , tl1_socket    ( "master_tl1", this, &ocp_tl1_tl3_slave_adapter::set_tl1_slave_timing )
  , m_invariant_ext_pool       (10)
  , m_position_ext_pool        (10)
  , m_resp_accept              ( "resp_accept" )
  , m_p_req_delay_calc         ( 0 )
  , m_p_data_delay_calc        ( 0 )
  , m_p_resp_acc_delay_calc    ( 0 )
  , m_phase_delay_peq          ( "phase_delay_peq" )
  , m_cur_sthreadbusy          ( 0 )
  , m_cur_sdatathreadbusy      ( 0 )
  , m_pipelined_sthreadbusy    ( 0 )
  , m_pipelined_sdatathreadbusy( 0 )
{
    slave_socket.register_nb_transport_fw(this, &ocp_tl1_tl3_slave_adapter::nb_transport);
    tl1_socket.register_nb_transport_bw(this, &ocp_tl1_tl3_slave_adapter::nb_transport_tl1);
    tl1_socket.activate_synchronization_protection();
    tl1_socket.make_generic();

    SC_THREAD( request_arb_th );
    SC_THREAD( data_arb_th );

    SC_METHOD( pipeline_threadbusy_signals );
    SC_METHOD( phase_delay_peq_out );
    sensitive << m_phase_delay_peq.get_event();
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::reset()
{
    m_req_issued  = NULL;
    m_data_issued = NULL;
    m_phase_delay_peq.reset();
    for ( typename std::vector<phase_queue*>::iterator it = m_req_txn.begin();
          it != m_req_txn.end(); ++it ) {
        (*it)->clear();
    }
    for ( typename std::vector<phase_queue*>::iterator it = m_data_waiting_txn.begin();
          it != m_data_waiting_txn.end(); ++it ) {
        (*it)->clear();
    }
    for ( typename std::vector<phase_queue*>::iterator it = m_data_txn.begin();
          it != m_data_txn.end(); ++it ) {
        (*it)->clear();
    }
}

template<unsigned int BUSWIDTH>
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::~ocp_tl1_tl3_slave_adapter()
{
    for ( typename std::vector<phase_queue*>::iterator it = m_req_txn.begin();
          it != m_req_txn.end(); ++it ) {
        delete *it;
    }
    for ( typename std::vector<phase_queue*>::iterator it = m_data_waiting_txn.begin();
          it != m_data_waiting_txn.end(); ++it ) {
        delete *it;
    }
    for ( typename std::vector<phase_queue*>::iterator it = m_data_txn.begin();
          it != m_data_txn.end(); ++it ) {
        delete *it;
    }
    delete m_p_req_delay_calc;
    delete m_p_data_delay_calc;
}


template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::before_end_of_elaboration()
{
    ocp_params = tl1_socket.get_ocp_config();
    // per-thread structures
    for ( int thread = 0; thread < ocp_params.threads; ++thread ) {
	m_req_txn.push_back( new phase_queue(0) );  // unlimited queueing
	if ( ocp_params.datahandshake ) {
	    m_data_waiting_txn.push_back( new phase_queue(0) );
	    m_data_txn        .push_back( new phase_queue(0) );
	}
    }

    // Arbitration filters
    m_req_arb.set_range ( 0, ocp_params.threads );
    m_data_arb.set_range( 0, ocp_params.threads );
    if ( ocp_params.sthreadbusy ) {
	 uint32_t& tb_word = ( ocp_params.sthreadbusy_pipelined ) ? m_pipelined_sthreadbusy : m_cur_sthreadbusy;
	 m_sthreadbusy_filt.set_busy_word( tb_word );
	 m_req_arb.add_filter( m_sthreadbusy_filt );
    }
    if ( ocp_params.sdatathreadbusy ) {
	 uint32_t& tb_word = ( ocp_params.sdatathreadbusy_pipelined ) ? m_pipelined_sdatathreadbusy : m_cur_sdatathreadbusy;
	 m_sdatathreadbusy_filt.set_busy_word( tb_word );
	 m_data_arb.add_filter( m_sdatathreadbusy_filt );
    }

    m_force_data_together_thread = s_no_winner;
    uint32_t effective_tags = ocp_params.tags + ocp_params.taginorder;
    m_burst_tracker.resize( ocp_params.threads, std::vector<ocp_txn_track>(
				effective_tags, ocp_txn_track( &this->ocp_params, true ) ) );

    // phase accept
    if ( ocp_params.respaccept ) {
	m_resp_accept.register_callback ( this, &ocp_tl1_tl3_slave_adapter<BUSWIDTH>::accept_resp );
    }

    m_timing_guard   .m_clk( clk );
    m_phase_delay_peq.m_clk( clk );
    m_resp_accept    .m_clk( clk );

    update_timing();
    reset();
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::update_timing()
{
    ocp_tl1_master_timing my_tl1_timing;
    my_tl1_timing.RequestGrpStartTime =
	sc_core::sc_get_time_resolution() + m_timing_guard.m_slave_timing.SThreadBusyStartTime;
    my_tl1_timing.DataHSGrpStartTime =
	sc_core::sc_get_time_resolution() + m_timing_guard.m_slave_timing.SDataThreadBusyStartTime;

    my_tl1_timing.MThreadBusyStartTime = sc_core::SC_ZERO_TIME;
    my_tl1_timing.MRespAcceptStartTime = sc_core::SC_ZERO_TIME;

    if ( my_tl1_timing != m_my_tl1_timing ) {
	m_my_tl1_timing = my_tl1_timing;
	tl1_socket.set_master_timing( m_my_tl1_timing );
    }
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::set_tl1_slave_timing(
    OCPIP_VERSION::ocp_tl1_slave_timing sl_timing )
{
    m_timing_guard.receive_slave_timing( sl_timing );
    update_timing();
}

template<unsigned int BUSWIDTH>
tlm::tlm_sync_enum OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::nb_transport(
    tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim)
{
    if ( ph == tlm::BEGIN_REQ ) {
	ocp_txn_burst_invariant* p_inv = m_invariant_ext_pool.create();
    	*p_inv = ocp_txn_burst_invariant::init_from( txn, ocp_params, tl1_socket );
	acc(txn).set_extension( p_inv );
	ocp_txn_position* p_pos = m_position_ext_pool.create();
	p_pos->clear();
	acc(txn).set_extension( p_pos );
	const uint32_t thread = p_inv->threadid;
	const uint32_t tag     = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;
	assert( thread < m_burst_tracker.size() );
	assert( tag    < m_burst_tracker[thread].size() );

	// expand to TL1 request and data phases
	ocp_tl1_phase_pos tl1_phase;
	tl1_phase.txn = &txn;
	do {
	    tl1_phase.position = m_burst_tracker[thread][tag].track_phase( txn, tlm::BEGIN_REQ );
	    tl1_phase.delay = 0;
	    if ( m_p_req_delay_calc ) { // optional delay
		tl1_phase.delay = m_p_req_delay_calc->calc( tl1_phase.position, tl1_phase.txn );
	    }
	    bool inserted = m_req_txn[thread]->nb_put( tl1_phase );
	    assert( inserted );
	    // kick off the first one (the only one if SRMD)
	    if ( tl1_phase.position.count == 1 && tl1_phase.delay > 0 ) {
		m_phase_delay_peq.notify( m_req_txn[thread]->back().delay, tl1_phase.delay );
	    }
	} while ( tl1_phase.position.remain > 0 );
	if ( txn.is_write() && ocp_params.datahandshake ) {
	    // data phases get queued into m_data_waiting. They will get promoted to m_data_txn
	    // when the corresponding request(s) gets sent
	    do {
		tl1_phase.position = m_burst_tracker[thread][tag].track_phase( txn, BEGIN_DATA );
		bool inserted = m_data_waiting_txn[thread]->nb_put( tl1_phase );
		assert( inserted );
	    } while ( tl1_phase.position.remain > 0 );
	}
	m_new_req_input_ev.notify( sc_core::SC_ZERO_TIME );
    } else {
	std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_fw"<<std::endl;
	std::cerr<<"Only BEGIN_REQ is supported."<<std::endl;
	exit(1);
    }
    return tlm::TLM_ACCEPTED;
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::request_arb_th()
{
    const thread_type threads = m_req_txn.size();
    while ( true ) {
	m_force_data_together_thread = s_no_winner;
	wait ( sc_core::SC_ZERO_TIME );
	bool all_empty = true;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    if ( m_req_txn[thread]->nb_can_peek() ) {
		all_empty = false; break;
	    }
	}
	if ( all_empty ) {
	    wait( m_new_req_input_ev );
	    continue; // something just came in, it can go in this cycle
	}
	if ( m_req_issued != NULL ) {
	    wait( m_cmd_accept_ev );
	    wait( clk->posedge_event() );
	    continue;
	}

	wait( sc_core::SC_ZERO_TIME ); // to let immediate notify complete for a 0 phase delay
	if ( ocp_params.sthreadbusy && ocp_params.sthreadbusy_exact )
	    wait_sthreadbusy_stable();

	// gather ready candidates
	std::set<thread_type> candidates;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    ocp_tl1_phase_pos req_phase;
	    if ( !m_req_txn[thread]->nb_peek( req_phase ) ) {
		continue;
	    }
	    // 1) not ready: latency not elapsed
	    if ( req_phase.delay != 0 ) {
		continue;
	    }
	    if ( ocp_params.reqdata_together && ocp_params.datahandshake ) {
		if ( req_phase.txn->is_write() ) {
		    thread_type eff_data_thread = ocp_params.sdatathreadbusy ? thread : 0;
		    // 2) not ready: data queue must be empty, as data for the current request
		    // phase will be queued upon arb win. Otherwise, data queue needs to drain
		    if ( m_data_txn[eff_data_thread]->nb_can_peek() ) {
			continue;
		    }
		    // 3) not ready: data thread must not be busy
		    if ( ocp_params.sdatathreadbusy && m_sdatathreadbusy_filt.is_busy( thread ) )
			continue;
		}
	    }
	    candidates.insert( thread );
	}

	// arbitrate
	thread_type winner;
	if ( m_req_arb( candidates, winner ) ) { // yielded a winner, send it!
	    bool success = send_request( winner );
	    assert( success );
	    m_req_arb.winner( winner );
	}
	wait ( clk->posedge_event() );
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::data_arb_th()
{
    const thread_type threads = m_data_txn.size();
    while ( true ) {
	wait ( sc_core::SC_ZERO_TIME );
	bool all_empty = true;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    if ( m_data_txn[thread]->nb_can_peek() ) {
		all_empty = false; break;
	    }
	}
	if ( all_empty ) {
	    wait( m_new_data_input_ev );
	    continue; // something just came in, it can go in this cycle
	}
	if ( m_data_issued != NULL ) {
	    wait( m_data_accept_ev );
	    wait( clk->posedge_event() );
	    continue;
	}

	wait( sc_core::SC_ZERO_TIME ); // to let immediate notify complete for a 0 delay
	if ( ocp_params.sdatathreadbusy && ocp_params.sdatathreadbusy_exact )
	    wait_sdatathreadbusy_stable();
	wait( sc_core::SC_ZERO_TIME ); // let request go first

	// gather ready candidates
	std::set<thread_type> candidates;
	for ( thread_type thread=0; thread < threads; ++thread ) {
	    ocp_tl1_phase_pos data_phase;
	    if ( !m_data_txn[thread]->nb_peek( data_phase ) ) {
		continue;
	    }
	    // not ready: latency not elapsed
	    if ( data_phase.delay != 0 ) {
		continue;
	    }
	    candidates.insert( thread );
	}

	// arbitrate
	thread_type winner = s_no_winner;
	if ( m_force_data_together_thread != s_no_winner ) {
	    assert( candidates.find( m_force_data_together_thread ) != candidates.end() );
	    winner = m_force_data_together_thread;
	} else {
	    m_data_arb( candidates, winner );
	}
	if ( winner != s_no_winner ) {
	    bool success = send_data( winner );
	    assert( success );
	    m_data_arb.winner( winner );
	}
	wait ( clk->posedge_event() );
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::phase_delay_peq_out()
{
    int* pit;
    while ( ( pit = m_phase_delay_peq.get_next_transaction() ) != NULL ) {
	(*pit) = 0; // points to a delay
    }
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::pipeline_threadbusy_signals() {
    if ( !ocp_params.sthreadbusy_exact && !ocp_params.sdatathreadbusy_exact )
	return;
    next_trigger( clk->posedge_event() );
    m_pipelined_sthreadbusy     = m_cur_sthreadbusy;
    m_pipelined_sdatathreadbusy = m_cur_sdatathreadbusy;
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::wait_sthreadbusy_stable()
{
    if ( ocp_params.sthreadbusy && ocp_params.sthreadbusy_exact ) {
	sc_core::sc_time wait_time = sc_core::sc_get_time_resolution(); // because of peq protection
	if ( !m_timing_guard.is_sthreadbusy_stable() ) {
	    sc_core::sc_time time_to_stable = m_timing_guard.time_to_sthreadbusy_stable();
	    if ( time_to_stable > wait_time )
		wait_time = time_to_stable;
	}
	wait( wait_time + sc_core::sc_get_time_resolution() ); // exceed stable time by 1 res to be safe
    }
}

template<unsigned int BUSWIDTH>
bool OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::send_request( thread_type thread )
{
    phase_queue::iterator reqIt = m_req_txn[thread]->begin();
    assert( reqIt != m_req_txn[thread]->end() );
    ocp_tl1_phase_pos req_phase = *reqIt;
    assert( req_phase.txn != NULL );
    tlm::tlm_generic_payload& txn = *req_phase.txn;
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( txn, acc );

    time=sc_core::SC_ZERO_TIME;
    phase=tlm::BEGIN_REQ;
    tlm::tlm_sync_enum retVal = tl1_socket->nb_transport_fw(txn, phase, time);

    p_pos->req_position = req_phase.position;
    if ( p_pos->req_position.count % s_interleave_size == 0 || p_pos->req_position.remain == 0 ) {
	// where to handle interleaving ??
    }

    // kick off the next req phase delay if more reqs in burst (only mrmd would)
    if ( m_p_req_delay_calc && p_pos->req_position.remain > 0 ) {
	reqIt++;
	assert( reqIt != m_req_txn[thread]->end() );
	m_phase_delay_peq.notify( reqIt->delay, reqIt->delay );
    }

    // promote waiting data phases to data arbitration candidates
    if ( txn.is_write() && ocp_params.datahandshake ) {
	// we only have arbitration for data when non-blocking flow control is present.
	// otherwise, OCP restricts the data issuance order to be the same as request. Emulate
	// that behavior by shoving all data phases under thread 0 in that case
	uint32_t data_thread = ocp_params.sdatathreadbusy ? thread : 0;

	// for mrmd, each beat passes through here for request, 1-for-1 with data
	uint32_t num_data_phases = ( p_inv->srmd ) ? get_num_phases( txn, BEGIN_DATA, acc ) : 1;
	for ( uint32_t i=0; i<num_data_phases; ++i ) {
	    ocp_tl1_phase_pos promote_data_phase;
	    m_data_waiting_txn[thread]->nb_get( promote_data_phase );
	    assert( promote_data_phase.txn == req_phase.txn );
	    bool inserted = m_data_txn[data_thread]->nb_put( promote_data_phase );
	    assert( inserted );
	    // now work on the phase just inserted in the active data queue
	    // calculate a delay (req2data)
	    ocp_tl1_phase_pos& data_phase = m_data_txn[data_thread]->back();
	    data_phase.delay = 0;
	    if ( m_p_data_delay_calc ) {
		data_phase.delay = m_p_data_delay_calc->calc( data_phase.position, data_phase.txn );
		if ( ocp_params.reqdata_together && ( i==0 || !p_inv->srmd ) ) {
		    // reqdata together: for first data or any MRMD phase, there
		    // can be no delay as arb will be forced by request
		    data_phase.delay = 0;
		}
		// kick it off if mrmd or kick off the first only if srmd
		if ( i==0 && data_phase.delay > 0 ) { // kick off data phase delay
		    m_phase_delay_peq.notify( data_phase.delay, data_phase.delay );
		}
	    }
	}
	m_new_data_input_ev.notify( sc_core::SC_ZERO_TIME );

	if ( ocp_params.reqdata_together ) {
	    // rd_together, seed the data arbitration if a write is being sent
	    m_force_data_together_thread = thread;
	}
    }

    switch(retVal){
    case tlm::TLM_ACCEPTED:
	assert( ocp_params.cmdaccept );
	m_req_issued = &txn;
	break;
    case tlm::TLM_UPDATED:
	if ( phase != tlm::END_REQ ) {
	    std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_REQ."<<std::endl;
	    exit(1);
	}
	handle_end_phase( txn, phase );
	break;
    case tlm::TLM_COMPLETED:;
    }
    return true;
}

template<unsigned int BUSWIDTH>
void OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::wait_sdatathreadbusy_stable()
{
    if ( ocp_params.sdatathreadbusy && ocp_params.sdatathreadbusy_exact ) {
	sc_core::sc_time wait_time = sc_core::sc_get_time_resolution(); // because of peq protection
	if ( !m_timing_guard.is_sdatathreadbusy_stable() ) {
	    sc_core::sc_time time_to_stable = m_timing_guard.time_to_sdatathreadbusy_stable();
	    if ( time_to_stable > wait_time )
		wait_time = time_to_stable;
	}
	wait( wait_time + sc_core::sc_get_time_resolution() ); // exceed stable time by 1 res to be safe
    }
}

template<unsigned int BUSWIDTH>
bool OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::send_data( thread_type thread )
{
    phase_queue::iterator dataIt = m_data_txn[thread]->begin();
    assert( dataIt != m_data_txn[thread]->end() );
    ocp_tl1_phase_pos data_phase = *dataIt;
    assert( data_phase.txn != NULL );
    tlm::tlm_generic_payload& txn = *data_phase.txn;
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( txn, acc );

    time=sc_core::SC_ZERO_TIME;
    phase=BEGIN_DATA;
    tlm::tlm_sync_enum retVal = tl1_socket->nb_transport_fw(txn, phase, time);

    p_pos->dh_position = data_phase.position;
    if ( p_pos->dh_position.count % s_interleave_size == 0 || p_pos->dh_position.remain == 0 ) {
	// where to handle interleaving??
    }

    // kick off the next data phase delay if more data in burst
    if ( m_p_data_delay_calc && p_inv->srmd && p_pos->dh_position.remain > 0 ) {
	dataIt++;
	assert( dataIt != m_data_txn[thread]->end() );
	m_phase_delay_peq.notify( dataIt->delay, dataIt->delay );
    }

    switch(retVal){
    case tlm::TLM_ACCEPTED:
	assert( ocp_params.dataaccept );
	m_data_issued = &txn;
	break;
    case tlm::TLM_UPDATED:
	if ( phase != END_DATA ) {
	    std::cerr<<"Error in "<<name()<<" : got unexpected phase update to "<<phase<<" in response to BEGIN_DATA."<<std::endl;
	    exit(1);
	}
	handle_end_phase( txn, phase );
	break;
    case tlm::TLM_COMPLETED:;
    }

    return true;
}

template<unsigned int BUSWIDTH>
tlm::tlm_sync_enum OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::nb_transport_tl1(
    tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim) {

    if (ph==CMD_THREAD_BUSY_CHANGE){
	cmd_thread_busy* p_tb = tl1_socket.template get_extension<cmd_thread_busy>( txn );
        m_cur_sthreadbusy = p_tb->value;
    } else if (ph==DATA_THREAD_BUSY_CHANGE){
	data_thread_busy* p_tb = tl1_socket.template get_extension<data_thread_busy>( txn );
        m_cur_sdatathreadbusy = p_tb->value;
    } else if ( ph==BEGIN_RESET || ph==END_RESET ) {
	reset();
    } else { // dataflow
	ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
	const uint32_t thread = p_inv->threadid;
	const uint32_t tag     = p_inv->taginorder ? ocp_params.tags : p_inv->tagid;
	assert( thread < m_burst_tracker.size() );
	assert( tag    < m_burst_tracker[thread].size() );

	switch(ph) {
	case tlm::END_REQ:
	    handle_end_phase( txn, ph ); break;
	case tlm::BEGIN_RESP: {
	    m_resp_pending_acc.txn = &txn;
	    m_resp_pending_acc.position = m_burst_tracker[thread][tag].track_phase( txn, ph );
	    if ( !ocp_params.respaccept ) { // must end resp phase on return
		accept_resp( m_resp_pending_acc );
		m_resp_pending_acc.txn = NULL;
		ph = tlm::END_RESP;
		return tlm::TLM_UPDATED;
	    } else {
		// schedule acceptance, will result in callback to accept_resp()
		uint32_t delay = 0;
		if ( m_p_resp_acc_delay_calc )
		    delay = m_p_resp_acc_delay_calc->calc( m_resp_pending_acc.position, &txn );
		m_resp_accept.accept( m_resp_pending_acc, delay );
	    }
	    break;
	}
	default:
	    if ( ph == END_DATA ) {
		handle_end_phase( txn, ph );
	    } else {
		std::cerr<<"Error in "<<name()<<" : got unexpected phase "<<ph<<" in nb_transport_bw"<<std::endl;
		exit(1);
	    }
	}
    }
    return tlm::TLM_ACCEPTED;
}

template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::accept_resp( OCPIP_VERSION::ocp_tl1_phase_pos& resp_phase )
{
    tlm::tlm_generic_payload& txn = *resp_phase.txn;
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( txn, acc );

    p_pos->resp_position = resp_phase.position;
    if ( ocp_params.respaccept ) {
// 	if ( !can_accept_resp( resp_phase ) ) {
// 	    return; // accept will have to be triggered by exert_resp_flow_control()
// 	}
	tlm::tlm_phase ph = tlm::END_RESP;
	sc_core::sc_time zero;
	tl1_socket->nb_transport_fw( txn, ph, zero );
    }

    // last response, send resp to (tl3) slave socket
    if ( p_pos->resp_position.remain == 0 ) {
	phase = tlm::BEGIN_RESP;
	slave_socket->nb_transport_bw( txn, phase, time);
	acc(txn).clear_extension( p_inv );
	m_invariant_ext_pool.recycle( p_inv );
	acc(txn).clear_extension( p_pos );
	m_position_ext_pool.recycle( p_pos );
    }
    m_resp_pending_acc.txn = NULL;
}


template<unsigned int BUSWIDTH>
void
OCPIP_VERSION::ocp_tl1_tl3_slave_adapter<BUSWIDTH>::handle_end_phase(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph)
{
    ocp_txn_burst_invariant* p_inv = require_ispec_extension<ocp_txn_burst_invariant>( txn, acc );
    ocp_txn_position*        p_pos = require_ispec_extension<ocp_txn_position>       ( txn, acc );

    if ( ph == tlm::END_REQ ) {
	if ( ocp_params.cmdaccept ) {
	    m_req_issued=NULL;
	    m_cmd_accept_ev.notify();
	}
	ocp_tl1_phase_pos req_phase;
	bool success = m_req_txn[p_inv->threadid]->nb_get( req_phase );
	assert( success );
	assert( req_phase.txn == &txn );
	if ( p_pos->req_position.remain == 0 ) {
	    // last request of burst ended
	    if ( p_inv->cmd == ocpip_legacy::OCP_MCMD_WR &&
		 !ocp_params.datahandshake && !ocp_params.writeresp_enable) {
		// the write is over
		phase = tlm::BEGIN_RESP;
		slave_socket->nb_transport_bw( txn, phase, time);
		acc(txn).clear_extension( p_inv );
		m_invariant_ext_pool.recycle( p_inv );
		acc(txn).clear_extension( p_pos );
		m_position_ext_pool.recycle( p_pos );
	    }
	}
    }
    if ( ph == END_DATA ) {
	if ( ocp_params.dataaccept ) {
	    m_data_issued=NULL;
	    m_data_accept_ev.notify();
	}
	ocp_tl1_phase_pos data_phase;
	// for data_thread, see comment in send_request about non-blocking flow control
	uint32_t data_thread = ocp_params.sdatathreadbusy ? p_inv->threadid : 0;
	bool success = m_data_txn[data_thread]->nb_get( data_phase );
	assert( success );
	assert( data_phase.txn == &txn );
	if ( p_pos->dh_position.remain == 0 &&
	     ( p_inv->cmd == ocpip_legacy::OCP_MCMD_WR && !ocp_params.writeresp_enable) ) {
	    // last DH of write burst with no response => burst over
	    phase = tlm::BEGIN_RESP;
	    slave_socket->nb_transport_bw( txn, phase, time);
	    acc(txn).clear_extension( p_inv );
	    m_invariant_ext_pool.recycle( p_inv );
	    acc(txn).clear_extension( p_pos );
	    m_position_ext_pool.recycle( p_pos );
	}
    }
}
