// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level 
//      Authors : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com 
//         Date : 03/31/2003
//
//  Description : Transaction Level - Layer-1 data class for
//                the Open Core Protocol (OCP) 1.0
//                The data class can be used in conjunction with
//                OCP transaction channel (tl_channel.h) to transport
//                data between Initiator and Target modules.
//                Contains member variables and access methods
//                to set or read the member variables.
//                The class contains members for transporting all OCP signals
//
//                The members are split in to three groups:
//                - Request-Channel members (set by Master read by Slave)
//                - Data Request-Channel members (set by Master read by Slave)
//                - Response-Channel members (set by Slave, read by Master)
//
//                See definition of these groups in OCP 1.0 specification
//
// ============================================================================

// This file contains the definition of the classes:
// template <class Td, class Ta> class OCP_TL1_DataCl
// 
// Template parameters Td: OCP data field type, Ta: OCP address field type 

#ifndef _OCP_TL1_DATA_CL
#define _OCP_TL1_DATA_CL

// -------------------------------------------------
// MCmd encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMCmdType {
  OCP_MCMD_IDLE = 0,
  OCP_MCMD_WR,
  OCP_MCMD_RD,
  OCP_MCMD_RDEX,
  OCP_MCMD_RESERVED4,
  OCP_MCMD_WRNP,
  OCP_MCMD_RESERVED6,
  OCP_MCMD_BCST
};

// -------------------------------------------------
// SResp encoding (OCP 1.0)
// -------------------------------------------------
enum OCPSRespType {
  OCP_SRESP_NULL = 0,
  OCP_SRESP_DVA,
  OCP_SRESP_RESERVED2,
  OCP_SRESP_ERR
};

// -------------------------------------------------
// MBurst encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMBurstType {
  OCP_MBURST_LAST = 0,
  OCP_MBURST_DFLT1,
  OCP_MBURST_TWO,
  OCP_MBURST_DFLT2,
  OCP_MBURST_FOUR,
  OCP_MBURST_STRM,
  OCP_MBURST_EIGHT,
  OCP_MBURST_CONT
};

template <class Td, class Ta> class OCP_TL1_DataCl : public sc_prim_channel
{
public:

  typedef Td DataType;
  typedef Ta AddrType;

  // Constructor
  OCP_TL1_DataCl()
  {
    // initialize public members
    Reset();
  }

  // Destructor
  ~OCP_TL1_DataCl()
  {
  }

  void Reset()
  {
    // Request Signal Group
    MAddr[0] = MAddr[1]           = 0;
    MAddrSpace[0] = MAddrSpace[1] = 0;
    MBurst[0] = MBurst[1]         = OCP_MBURST_LAST;
    MByteEn[0] = MByteEn[1]       = 0;
    MCmd[0] = MCmd[1]             = OCP_MCMD_IDLE;
    MConnID[0] = MConnID[1]       = 0;
    MData[0] = MData[1]           = 0;
    MThreadID[0] = MThreadID[1]   = 0;

    // Response Signal Group
    SData[0] = SData[1] = 0;
    SResp[0] = SResp[1] = OCP_SRESP_NULL;
    SThreadID[0] = SThreadID[1] = 0;

    // Datahandshake Signal Group
    MDataValid[0] = MDataValid[1] = false;
    MDataThreadID[0] = MDataThreadID[1] = 0;

    // initialize internal members
    m_Synchron = true; // hard coded, use TL_SYNCHRON_DATA_CL to configure

    m_Request  = false;
    m_ReqToggle = 0;

    m_Response = false;
    m_ResToggle = 0;
    
    m_DataRequest   = false;
    m_DataReqToggle = 0;

    m_UgRequest  = false;
    m_UgReqToggle = 0;

    m_UgResponse = false;
    m_UgResToggle = 0;
  }

//__________________________________________________________
//
// START of ACCESS METHODS
//__________________________________________________________

// __________________________________________________________
//
// Channel Methods
// These methods are called by the transaction channel, and
// they are mandatory.
// __________________________________________________________

  

  // Test and set access direction (read or write) 
  bool IsWriteRequest()
  {
    return((MCmd[1 - m_ReqToggle] == OCP_MCMD_WR) ? true : false);
  }
  void SetWriteRequest()
  {
    MCmd[m_ReqToggle] = OCP_MCMD_WR;
  }
  void SetReadRequest() 
  {
    MCmd[m_ReqToggle] = OCP_MCMD_RD;
  }


  // Makes new initiator request data visible to the target and
  // implements inertial assignment for request fields.
  // ToggleRequest() must be called for: 
  // SetWriteRequest(), SetReadRequest(), MputMCmd()
  void ToggleRequest()
  {
    m_Request = true;
    if (m_Synchron) request_update();
    else update();
  }

  // Makes new target response data visible to the initiator and
  // implements inertial assignment for response fields.
  // ToggleRequest() must be called for: SputSResp()
  void ToggleResponse()
  {
    m_Response = true;
    if (m_Synchron) request_update();
    else update();
  }

  // Makes new initiator request data (second request channel)
  // visible to the target and
  // implements inertial assignment for data request fields.
  // ToggleDataRequest() must be called for: MDataValid()
  void ToggleDataRequest()
  {
    m_DataRequest = true;
    if (m_Synchron) request_update();
    else update();
  }

// __________________________________________________________
//
// Request Signal Group 
// __________________________________________________________

  // Access methods for MAddr
  void MputMAddr(Ta a)
  {
    MAddr[m_ReqToggle] = a;
  }
  Ta SgetMAddr()
  {
    return MAddr[1 - m_ReqToggle];
  }

  // Access methods for MAddrSpace
  void MputMAddrSpace(int a)
  {
    MAddrSpace[m_ReqToggle] = a;
  }
  int SgetMAddrSpace()
  {
    return MAddrSpace[1 - m_ReqToggle];
  }

  // Access methods for MBurst
  void MputMBurst(OCPMBurstType a)
  {
    MBurst[m_ReqToggle] = a;
  }
  OCPMBurstType SgetMBurst()
  {
    return MBurst[1 - m_ReqToggle];
  }

  // Access methods for MByteEn
  void MputMByteEn(int a)
  {
    MByteEn[m_ReqToggle] = a;
  }
  int SgetMByteEn()
  {
    return MByteEn[1 - m_ReqToggle];
  }

  // Access methods for MCmd
  void MputMCmd(OCPMCmdType a)
  {
    MCmd[m_ReqToggle] = a;
  }
  OCPMCmdType SgetMCmd()
  {
    OCPMCmdType a = MCmd[1 - m_ReqToggle];
    MCmd[1 - m_ReqToggle] = OCP_MCMD_IDLE; // reset MCmd after testing it
    return a;
  }

  // Access methods for MConnID
  void MputMConnID(int a)
  {
    MConnID[m_ReqToggle] = a;
  }
  int SgetMConnID()
  {
    return MConnID[1 - m_ReqToggle];
  }

  // Access methods for MData
  void MputMData(Td d)
  {
    MData[m_ReqToggle] = d;
  }
  void SgetMData(Td &d)
  {
    d = MData[1 - m_ReqToggle];
  }
  Td SgetMData()
  {
    return MData[1 - m_ReqToggle];
  }

  // Access methods for MThreadID
  void MputMThreadID(int a)
  {
    MThreadID[m_ReqToggle] = a;
  }
  int SgetMThreadID()
  {
    return MThreadID[1 - m_ReqToggle];
  }

// __________________________________________________________
//
// Response Signal Group 
// __________________________________________________________

  // Access methods for SData
  void SputSData(Td d)
  {
    SData[m_ResToggle] = d;
  }
  void MgetSData(Td &d)
  {
    d = SData[1 - m_ResToggle];
  }
  Td MgetSData()
  {
    return SData[1 - m_ResToggle];
  }

  // Access methods for SResp
  void SputSResp(OCPSRespType a)
  {
    SResp[m_ResToggle] = a;
  }
  OCPSRespType MgetSResp()
  {
    OCPSRespType a = SResp[1 - m_ResToggle];
    SResp[1 - m_ResToggle] = OCP_SRESP_NULL; // reset SResp after testing it
    return a;
  }

  // Access methods for SThreadID
  void SputSThreadID(int a)
  {
    SThreadID[m_ResToggle] = a;
  }
  int MgetSThreadID()
  {
    return SThreadID[1 - m_ResToggle];
  }

// __________________________________________________________
//
// Datahandshake Signal Group 
// __________________________________________________________

  // Access methods for MData with data handshake
  void MputMDataHS(Td d)
  {
    MData[m_DataReqToggle] = d;
  }
  void SgetMDataHS(Td &d)
  {
    d = MData[1 - m_DataReqToggle];
  }
  Td SgetMDataHS()
  {
    return MData[1 - m_DataReqToggle];
  }

  // Access methods for MDataValid
  void MputMDataValid(bool a)
  {
    MDataValid[m_DataReqToggle] = a;
  }
  bool SgetMDataValid()
  {
    bool a = MDataValid[1 - m_DataReqToggle];
    MDataValid[1 - m_DataReqToggle] = false; // reset
    return a;
  }

  // Access methods for MDataThreadID
  void MputMDataThreadID(int a)
  {
    MDataThreadID[m_DataReqToggle] = a;
  }
  int SgetMDataThreadID()
  {
    return MDataThreadID[1 - m_DataReqToggle];
  }

  
//__________________________________________________________
//
// The following members have not been throughly tested
// for this release, and may change in later releases.
// They are included for completeness.
//__________________________________________________________

// __________________________________________________________
//
// Thread busy signaling. Ungrouped. (Alpha version)
// __________________________________________________________

  // Access methods for MThreadBusy
  void MputMThreadBusy(int a)
  {
    MThreadBusy[m_UgReqToggle] = a;
    m_UgRequest = true;
    if (m_Synchron) request_update();
    else update();
  }
  int SgetMThreadBusy()
  {
    return MThreadBusy[1 - m_UgReqToggle];
  }

  // Access methods for SThreadBusy
  void SputSThreadBusy(int a)
  {
    SThreadBusy[m_UgResToggle] = a;
    m_UgResponse = true;
    if (m_Synchron) request_update();
    else update();
  }
  int MgetSThreadBusy()
  {
    return SThreadBusy[1 - m_UgResToggle];
  }

// __________________________________________________________
//
// Sideband Signal Group. (Alpha version)
// __________________________________________________________

  // Access methods for MFlag
  void MputMFlag(int a)
  {
    MFlag[m_SbReqToggle] = a;
    MFlagEvent.notify();
  }
  int SgetMFlag()
  {
    return MFlag[1 - m_SbReqToggle];
  }

  // Access methods for SFlag
  void SputSFlag(int a)
  {
    SFlag[m_SbResToggle] = a;
    SFlagEvent.notify();
  }
  int MgetSFlag()
  {
    return SFlag[1 - m_SbResToggle];
  }

  // Access methods for SError
  void SputSError(bool a)
  {
    SError[m_SbResToggle] = a;
    SErrorEvent.notify();
  }
  bool MgetSError()
  {
    return SError[1 - m_SbResToggle];
  }

  // Access methods for SInterrupt
  void SputSInterrupt(bool a)
  {
    SInterrupt[m_SbResToggle] = a;
    SInterruptEvent.notify();
  }
  bool MgetSInterrupt()
  {
    return SInterrupt[1 - m_SbResToggle];
  }


  //---------------------------------------------------------------------
  // system/core signal group. (Alpha version)
  //---------------------------------------------------------------------

  // Access methods for Control
  void SysputControl(int a)
  {
    Control[m_CtrlToggle] = a;
  }
  int CgetControl()
  {
    return Control[1 - m_CtrlToggle];
  }

  // Access methods for ControlBusy
  void CputControlBusy(bool a)
  {
    ControlBusy[m_CtrlToggle] = a;
  }
  bool SysgetControlBusy()
  {
    return ControlBusy[1 - m_CtrlToggle];
  }

  // Access methods for ControlWr
  void SysputControlWr(bool a)
  {
    ControlWr[m_CtrlToggle] = a;
    ControlWrEvent.notify();
  }
  bool CgetControlWr()
  {
    return ControlWr[1 - m_CtrlToggle];
  }

  // Access methods for Status
  void CputStatus(int a)
  {
    Status[m_CtrlToggle] = a;
    StatusEvent.notify();
  }
  int SysgetStatus()
  {
    return Status[1 - m_CtrlToggle];
  }

  // Access methods for StatusBusy
  void CputStatusBusy(bool a)
  {
    StatusBusy[m_CtrlToggle] = a;
  }
  bool SysgetStatusBusy()
  {
    return StatusBusy[1 - m_CtrlToggle];
  }

  // Access methods for StatusRd
  void SysputStatusRd(bool a)
  {
    StatusRd[m_CtrlToggle] = a;
    StatusRdEvent.notify();
  }
  bool CgetStatusRd()
  {
    return StatusRd[1 - m_CtrlToggle];
  }

//__________________________________________________________
//
// END of ACCESS METHODS
//__________________________________________________________


//__________________________________________________________
//
// This method updates the data members, making the new 
// initiator data visible to the target. The method is either
// called synchronously (via request_update()) or asynchronously
// (as direct call), depending on the value of m_Synchron.
//__________________________________________________________

  void update()
  {
    if (m_Request)  m_ReqToggle = 1 - m_ReqToggle;
    if (m_Response) m_ResToggle = 1 - m_ResToggle;
    if (m_DataRequest) m_DataReqToggle = 1 - m_DataReqToggle;
    if (m_UgRequest)  m_UgReqToggle = 1 - m_UgReqToggle;
    if (m_UgResponse) m_UgResToggle = 1 - m_UgResToggle;
    m_Request  = false;
    m_Response = false;
    m_DataRequest = false;
    m_UgRequest  = false;
    m_UgResponse = false;
  }


//__________________________________________________________
//
// OCP Data Members
//__________________________________________________________

// --------------------
// Request Signal Group
// --------------------
  Ta MAddr[2];
  int MAddrSpace[2];
  OCPMBurstType MBurst[2];
  int MByteEn[2];
  OCPMCmdType MCmd[2];
  int MConnID[2];
  Td MData[2];
  int MThreadID[2];

// ---------------------
// Response Signal Group
// ---------------------
  Td SData[2];
  OCPSRespType SResp[2];
  int SThreadID[2];

// --------------------------
// Datahandshake Signal Group
// -------------------------
  bool MDataValid[2];
  int MDataThreadID[2];
//  sc_event MDataValidEvent;
//  sc_event SDataAcceptEvent;

// -----------------
// Ungrouped Signals
// -----------------
  bool MThreadBusy[2];
  bool SThreadBusy[2];

// ---------------------
// Sideband Signal Group
// ---------------------
  // reset
  bool Reset_n[2];

  // side band signals
  int  MFlag[2];
  int  SFlag[2];
  bool SError[2];
  bool SInterrupt[2];

  // system/core signals
  int  Control[2];
  bool ControlWr[2];
  bool ControlBusy[2];
  int  Status[2];
  bool StatusRd[2];
  bool StatusBusy[2];

  // events
  sc_event MFlagEvent;
  sc_event SFlagEvent;
  sc_event SErrorEvent;
  sc_event SInterruptEvent;
  sc_event ControlWrEvent;
  sc_event StatusEvent;
  sc_event StatusRdEvent;

//__________________________________________________________
//
// End of OCP Data Members
//__________________________________________________________


// private: 
// We make them public for read access.

  // synchronous/asynchronous update of the data members
  bool m_Synchron;

  // request channel update
  bool m_Request;
  int m_ReqToggle;

  // response channel update
  bool m_Response;
  int m_ResToggle;

  // second request channel (for data handshake) update
  bool m_DataRequest;
  int m_DataReqToggle;

  // ungrouped signals update
  bool m_UgRequest;
  int m_UgReqToggle;
  bool m_UgResponse;
  int m_UgResToggle;
};

#endif // _OCP_TL1_DATA_CL
