// 
//  Copyright 2008 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG
//       Author : Albert Chitemyan, Sonics Inc. 
//                
//         $Id: 
//
//  Description : OCP API - Layer-2 Simple Example slave
//    Features:
//    - Synchronous
//    - Non-blocking methods are used
//    - Not pipelined
// ============================================================================

#include "slave.h"

using namespace std;

/////////////////////////////////////////////////////////////////////////////
// 
// This example demonstrates how to use timing extensions for TL2 modeling
// as well as how to use special transactions to transport timing information 
// from master to slave and visa versa.
//
// In this particular example we demonstrate ReqSndI timing annotation usage,
// other timing annotation should be used in the same way.
// 
/////////////////////////////////////////////////////////////////////////////

// ----------------------------------------------------------------------------
// Constructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::Slave (sc_core::sc_module_name name_)
  : sc_core::sc_module (name_), tp_socket("tp_socket"), req(NULL), ongoing_rsp(false) {

  // initialize common members
  last_request = tlm::TLM_IGNORE_COMMAND;

  SC_METHOD(proc);
      sensitive << clk.pos();
      dont_initialize(); // make sure we execute only at clock edge

  SC_THREAD(process_waiting_for_burst_end);
      sensitive << clk.pos();

  tp_socket.register_nb_transport_fw(this, &Slave::nb_transport);

  // Master should send value of this paramater
  // but just in case, default value
  req_send_interval = 1; 
  burst_is_in_porcess = false;
}


// ----------------------------------------------------------------------------
// Destructor
// ----------------------------------------------------------------------------
template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> Slave<BUSWIDTH, ADDRWIDTH>::~Slave(){}

// ----------------------------------------------------------------------------
//  Method : Slave::proc()
//
//  Synchronous slave process
//
// ----------------------------------------------------------------------------

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH> void Slave<BUSWIDTH, ADDRWIDTH>::proc(){

  scv_smart_ptr< unsigned int > rand;
  rand->next();

  // If a previous read-burst has been served, we check for new burst 
  if ((last_request == tlm::TLM_IGNORE_COMMAND) && !burst_is_in_porcess) 
  {
    if (req){
      last_request = req->get_command();

      if (last_request == tlm::TLM_READ_COMMAND)
        rsp = req;

      time = sc_core::SC_ZERO_TIME;
      phase = tlm::END_REQ;
      tlm::tlm_sync_enum retVal = tp_socket->nb_transport_bw(*req, phase, time);

      switch(retVal){
        case tlm::TLM_ACCEPTED: break;
        case tlm::TLM_UPDATED:
          cerr<<"Error in " << name() << " got unexpected phase update " << phase << " in response to END_REQ" << endl;
          exit(1);
          break;
        case tlm::TLM_COMPLETED:;
      }
      req = NULL;
    }
  }

  // If request was write
  if (last_request == tlm::TLM_WRITE_COMMAND) {
    last_request = tlm::TLM_IGNORE_COMMAND;
  }
  else 
  if (last_request == tlm::TLM_READ_COMMAND) {	
    if (!ongoing_rsp && !burst_is_in_porcess) {
      assert(rsp);

      // generate some data
      unsigned int resp_data = 10000 + ( (*rand)/100) * 100;

      for (unsigned int i = 0; i < burst_len; ++i)
          *(((Td*)(rsp->get_data_ptr())) + i) = resp_data++;

      rsp->set_response_status(tlm::TLM_OK_RESPONSE);

      // Send response
      time = sc_core::SC_ZERO_TIME;
      phase = tlm::BEGIN_RESP;
      tlm::tlm_sync_enum retVal = tp_socket->nb_transport_bw(*rsp,phase, time);

      switch(retVal){
        case tlm::TLM_ACCEPTED: 
          ongoing_rsp = true; 
          break;
        case tlm::TLM_UPDATED:
          if (phase != tlm::END_RESP) {
            std::cerr<<"Error in "<< name() <<" got unexpected phase update "<< phase <<" in response to BEGIN_RESP"<<std::endl;
            exit(1);
          }
          break;
        case tlm::TLM_COMPLETED: break;
      }

#ifdef DEBUG_TL2
      cout << "Slave sent response, status=" << rsp->get_response_status()
	       << "                      time=" << sc_core::sc_time_stamp() << endl;
#endif

      last_request = tlm::TLM_IGNORE_COMMAND;
      rsp = NULL;
    }
  } 
} // end of method

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
tlm::tlm_sync_enum Slave<BUSWIDTH, ADDRWIDTH>::nb_transport(tlm::tlm_generic_payload& txn, tlm::tlm_phase& ph, sc_core::sc_time& tim)
{
  switch(ph){
    case tlm::BEGIN_REQ: 
    
      req = &txn;     // get burst 
  
      assert (!burst_is_in_porcess);
      burst_is_in_porcess = true;

      // get burst length 
      ocpip::burst_length* b_len;
      tp_socket.get_extension<ocpip::burst_length>(b_len, *req);
      burst_len = b_len->value;

      burst_duration = burst_len * req_send_interval;
   
#ifdef DEBUG_TL2
     print_burst();
#endif
 
      break;
    case tlm::END_RESP: 
      ongoing_rsp = false; 
      break;
    default:
      if (ph == ocpip::TL2_TIMING_CHANGE){
         // Now, slave knows how frequently master sends requests
         //  and can calculate burst duration
         req_send_interval = tp_socket.template get_extension<ocpip::tl2_timing>(txn)->value.master_timing.RqSndI;
      }
      else {
         cerr<<"I only expect BEGIN_REQ or END_RESP on the forward path, but got "
              << ph << " In " << name() << " at " << sc_core::sc_time_stamp() << endl;
         exit(1);
      }
  }
  return tlm::TLM_ACCEPTED;

}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Slave<BUSWIDTH, ADDRWIDTH>::process_waiting_for_burst_end()
{
    while(true)
    {
       wait();
       if (burst_is_in_porcess)
       {
          // we already waited 1 cycle
          for (unsigned int i = 0; i < burst_duration - 1; i++)
             wait();

          burst_is_in_porcess = false;
       }
    }
}

template<unsigned int BUSWIDTH, unsigned int ADDRWIDTH>
void Slave<BUSWIDTH, ADDRWIDTH>::print_burst()
{
      if (last_request == tlm::TLM_READ_COMMAND)
      {
        cout << "Slave got READ burst "
             << "                     addr=" << rsp->get_address()
             << "                             time=" << sc_core::sc_time_stamp() << endl;
      }
      else{
        cout << "Slave got WRITE burst "
             << "                    time=" << sc_core::sc_time_stamp() << endl;
      }

      cout <<  "Slave will process this burst: " << burst_duration << " cycle " << endl;
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the Slave
//
// ----------------------------------------------------------------------------
template class Slave< 32,32 >; 

