///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// (c) Copyright OCP-IP 2008
// OCP-IP Confidential and Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : James Aldis, Texas Instruments
//                Robert Guenzel (from TU of Braunschweig) for Greensocs Ltd.
//
//          $Id:
//
//  Description :  Merger module for the TL1 timing example
//
//                                                                           //
///////////////////////////////////////////////////////////////////////////////

#ifndef __TIMING_MERGER_H__
#define __TIMING_MERGER_H__


#include "timing_common.h"

struct pseudo_thread_ID : public tlm_utils::instance_specific_extension<pseudo_thread_ID>{
};

class timing_merger : 
  public sc_core::sc_module
{
  SC_HAS_PROCESS(timing_merger);
  
  tlm_utils::instance_specific_extension_accessor acc;
  public:
  // ports
  sc_core::sc_in<bool> clk;  
  ocpip::ocp_slave_socket_tl1<>  ocpsA;
  ocpip::ocp_slave_socket_tl1<>  ocpsB;
  ocpip::ocp_master_socket_tl1<> ocpm;
  

    timing_merger(sc_core::sc_module_name name, int threads_Ap, int threads_Bp) 
      : sc_core::sc_module(name)
      , ocpsA("ocpsA", this, &timing_merger::setOCPTL1MasterTiming)
      , ocpsB("ocpsB", this, &timing_merger::setOCPTL1MasterTiming)
      , ocpm("ocpm", this, &timing_merger::setOCPTL1SlaveTiming)
      , threads_A(threads_Ap)
      , threads_B(threads_Bp)
      , mtb_A(0)
      , mtb_B(0) {

      ocpsA.register_nb_transport_fw(this, &timing_merger::nb_transport_fwA);
      ocpsB.register_nb_transport_fw(this, &timing_merger::nb_transport_fwB);
      ocpm.register_nb_transport_bw(this, &timing_merger::nb_transport_bw);


      SC_METHOD(clock_rising);
      sensitive<<clk.pos();
      dont_initialize();

      SC_METHOD(request_arb);
      sensitive<<arbiter_event;
      dont_initialize();

      SC_METHOD(mthreadbusy_propagate);
      sensitive<<mthreadbusy_event;
      dont_initialize();

      typedef tlm::tlm_generic_payload* tlm_generic_payload_ptr;
      req_reg = //new arbitration_signal<tlm::tlm_generic_payload*>[threads_A + threads_B];
                new tlm_generic_payload_ptr[threads_A + threads_B];
      for (int i=0; i<threads_A + threads_B; i++) req_reg[i]=NULL;
      reqA_reg = req_reg;
      reqB_reg = &(req_reg[threads_A]);

      time_quantum = sc_core::sc_get_time_resolution();
      sthreadbusy_sample_time = time_quantum;  // initial guess
      request_sample_time = time_quantum;  // initial guess
      mthreadbusy_sample_time = time_quantum;  // initial guess
      
      tb_txn_A=ocpsA.get_tb_transaction();
      tb_txn_B=ocpsB.get_tb_transaction();
      tb_txn_m=ocpm.get_tb_transaction();
      tb_ph=ocpip::THREAD_BUSY_CHANGE;
      
      
      std::cout << "<<<< E-O-E >>>> " << sc_core::sc_module::name() << std::endl;
      ocpip::ocp_tl1_master_timing my_mtiming;
      my_mtiming.RequestGrpStartTime =
                my_max(sthreadbusy_sample_time, request_sample_time);
      my_mtiming.MThreadBusyStartTime = mthreadbusy_sample_time;
      ocpm.set_master_timing(my_mtiming);

      ocpip::ocp_tl1_slave_timing my_stiming;
      my_stiming.ResponseGrpStartTime = response_in_time + time_quantum;
      ocpsA.set_slave_timing(my_stiming);
      ocpsB.set_slave_timing(my_stiming);

      ocpip::map_string_type config_map=get_config_map(threads_Ap,32);
      ocpip::ocp_parameters  config;
      config.set_ocp_configuration(ocpsA.name(), config_map);
      ocpsA.set_ocp_config(config);
      
      config_map=get_config_map(threads_Bp, 32);
      config.set_ocp_configuration(ocpsB.name(), config_map);
      ocpsB.set_ocp_config(config);
      
      config_map=get_config_map(threads_Bp+threads_Ap, 32);
      config.set_ocp_configuration(ocpm.name(), config_map);
      ocpm.set_ocp_config(config);
      
      tb_A=ocpsA.get_extension<ocpip::thread_busy>(*tb_txn_A);
      tb_A->value.type=ocpip::S_THREAD;
      tb_B=ocpsB.get_extension<ocpip::thread_busy>(*tb_txn_B);
      tb_B->value.type=ocpip::S_THREAD;
      tb_m=ocpm.get_extension<ocpip::thread_busy>(*tb_txn_m);
      tb_m->value.type=ocpip::M_THREAD;
      
      
    };

    ~timing_merger() {
      std::cout << "Deleting merger:   " << name() << std::endl;

      for(int i=0; i<threads_A; i++) {
        std::cout << (reqA_reg[i] ? "A " : "- ");
      }
      std::cout << std::endl;
      for(int i=0; i<threads_B; i++) {
        std::cout << (reqB_reg[i] ? "B " : "- ");
      }
      std::cout << std::endl;
      std::cout << "  SThreadBusy sample time: " << sthreadbusy_sample_time << std::endl;
      std::cout << "  MThreadBusy sample time: " << mthreadbusy_sample_time << std::endl;
      std::cout << "  Request sample time:     " << request_sample_time << std::endl;
      std::cout << "  Response input time:     " << response_in_time << std::endl;
    }

    // processes
    void clock_rising() {

      // generate SThreadBusy for both slave ports based on
      // loading of request registers
      int stb_out = 0;
      int mask = 1;
      for(int i=0; i<(threads_A+threads_B); i++) {
        if(req_reg[i]) {
          stb_out |= mask;
        }
        mask <<= 1;
      }
      tb_A->value.mask=(stb_out & ((1 << threads_A)-1));
      tb_B->value.mask=(stb_out >> threads_A);
      ocpsA->nb_transport_bw(*tb_txn_A, tb_ph, tb_time);
      ocpsB->nb_transport_bw(*tb_txn_B, tb_ph, tb_time);

      arbiter_event.notify(my_max(request_sample_time, sthreadbusy_sample_time));

      // response side:  get thread busy from both masters and give to
      // slave.  responses are passed to the appropriate master (based on thread
      // ID), as they arrive.
      // ** this is not needed once we have MThreadBusy value change events **
      // but we will still need to tell other modules our timing, which depends
      // on input timing.
      mthreadbusy_event.notify(mthreadbusy_sample_time);
    }

    // the response path is really simple - thread-busy-signals are
    // concatencated and passed to slave.  slave returns a response which
    // is forwarded to the right slave port
    void mthreadbusy_propagate() {
      tb_m->value.mask=(mtb_A | (mtb_B << threads_A));
      ocpm->nb_transport_fw(*tb_txn_m, tb_ph, tb_time);
    }

    void request_arb() {
      // now arbitrate: priority is smallest thread ID, A then B
      int mask = 1;
      for(int i=0; i<(threads_A+threads_B); i++) {
        if((req_reg[i]) && !(mask & stb)) {
          // grant this one
          txn_ph=tlm::BEGIN_REQ;
          txn_time=sc_core::SC_ZERO_TIME;
          tlm::tlm_sync_enum retVal=ocpm->nb_transport_fw(*req_reg[i], txn_ph, txn_time);
          switch (retVal){
            case tlm::TLM_UPDATED:
              if (txn_ph==tlm::END_REQ); //great! just an immediate accept!
              else{
                std::cerr<<"Unexpected phase "<<txn_ph<<" on return path. In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
                exit(1);
              }
              break;
            case tlm::TLM_COMPLETED:
              break;              
            default:
              std::cerr<<"When using thread busy exact, I expect the use of the return path. In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
              exit(1);
          }
          req_reg[i]=NULL;
          break;
        }
        mask <<= 1;
      }
    }

    // when informed of master port timing, merger must re-inform the OCP
    // channels if anything changed
    void setOCPTL1SlaveTiming(ocpip::ocp_tl1_slave_timing slave_timing) {
      std::cout << "  << S-S-T >>   " << name() << std::endl;

      // if sthreadbusy start changed, may need to change request timing.

      // calculate the sample time for sthreadbusy based on the new timing
      // information from the channel
      sc_core::sc_time stb_sample = slave_timing.SThreadBusyStartTime + time_quantum;
      // if larger than before, update the local sample time configuration
      // and, only if the request start time increased, inform the OCP port
      // (the request start time also depends on the request sample time)
      if(stb_sample > sthreadbusy_sample_time) {
        sc_core::sc_time old_req_start =
                  my_max(sthreadbusy_sample_time, request_sample_time);
        sthreadbusy_sample_time = stb_sample;
        sc_core::sc_time new_req_start =
                  my_max(sthreadbusy_sample_time, request_sample_time);
        if(new_req_start > old_req_start) {
          ocpip::ocp_tl1_master_timing my_mtiming;
          my_mtiming.RequestGrpStartTime = new_req_start;
          my_mtiming.MThreadBusyStartTime = mthreadbusy_sample_time;
          ocpm.set_master_timing(my_mtiming);
        }
      }

      // if response start increased, the OCP slave ports need to be informed
      // that the response timing increased
      if(slave_timing.ResponseGrpStartTime > response_in_time) {
        response_in_time = slave_timing.ResponseGrpStartTime;
        ocpip::ocp_tl1_slave_timing my_stiming;
        my_stiming.ResponseGrpStartTime = response_in_time + time_quantum;
        ocpsA.set_slave_timing(my_stiming);
        ocpsB.set_slave_timing(my_stiming);
      }
    }

    // when informed of master timing, merger must re-inform the OCP
    // channels if anything changed
    void setOCPTL1MasterTiming(ocpip::ocp_tl1_master_timing master_timing) {
      std::cout << "  << S-M-T >>   " << name() << std::endl;

      // start by calculating current (master) timing
      ocpip::ocp_tl1_master_timing my_mtiming;
      my_mtiming.RequestGrpStartTime =
                my_max(sthreadbusy_sample_time, request_sample_time);
      my_mtiming.MThreadBusyStartTime = mthreadbusy_sample_time;
      bool changed = false;

      // if request sample time increased this must be recorded
      sc_core::sc_time new_req_sample = master_timing.RequestGrpStartTime + time_quantum;
      if(new_req_sample > request_sample_time) {
        request_sample_time = new_req_sample;

        // if request output time increased (also depends on sthreadbusy
        // sample time) the channel must be informed
        sc_core::sc_time new_req_start = my_max(sthreadbusy_sample_time,
                  master_timing.RequestGrpStartTime +time_quantum);
        if(new_req_start > my_mtiming.RequestGrpStartTime) {
          changed = true;
          my_mtiming.RequestGrpStartTime = new_req_start;
        }
      }

      // if mthreadbusy sample time increased, mthreadbusy-output time
      // will increase and the channel must be informed
      sc_core::sc_time new_mtb_sample = master_timing.MThreadBusyStartTime +
                time_quantum;
      if(new_mtb_sample > mthreadbusy_sample_time) {
        changed = true;
        mthreadbusy_sample_time = new_mtb_sample;
        my_mtiming.MThreadBusyStartTime = new_mtb_sample;
      }

      if(changed) {
        ocpm.set_master_timing(my_mtiming);
      }
    }
    
    tlm::tlm_sync_enum nb_transport_fwA(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& time){
      if (ph==ocpip::THREAD_BUSY_CHANGE){
        assert(ocpsA.get_extension<ocpip::thread_busy>(gp)->value.type==ocpip::M_THREAD);
        mtb_A=ocpsA.get_extension<ocpip::thread_busy>(gp)->value.mask;
        return tlm::TLM_ACCEPTED;
      }
      if (ph!=tlm::BEGIN_REQ){
        std::cerr<<"I only expect BEGIN_REQ on the forward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      } 
      ocpip::thread_id* tmp;
      bool thread_id_available=ocpm.get_extension<ocpip::thread_id>(tmp, gp);
      if (!thread_id_available){
        assert(threads_A==1); //if there were more than 1 thread, there should have been a thread ID
        ocpm.validate_extension<ocpip::thread_id>(gp); 
        tmp->value=0; //we need a thread ID so we add it and activate it
      }

      assert(!(((tb_A->value.mask)>>(tmp->value)) & 1)); //make sure our thread is not busy
      
      //assert that we have not seen this thing before
      pseudo_thread_ID* check;
      acc(gp).get_extension(check);
      assert(!check);

      acc(gp).set_extension(&threadIDA);
      reqA_reg[tmp->value]=&gp; //at this time the thread id is correct
      ph=tlm::END_REQ;
      return tlm::TLM_UPDATED;
    }

    tlm::tlm_sync_enum nb_transport_fwB(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& tim){
      if (ph==ocpip::THREAD_BUSY_CHANGE){
        assert(ocpsB.get_extension<ocpip::thread_busy>(gp)->value.type==ocpip::M_THREAD);
        mtb_B=ocpsB.get_extension<ocpip::thread_busy>(gp)->value.mask;
        return tlm::TLM_ACCEPTED;
      }
      if (ph!=tlm::BEGIN_REQ){
        std::cerr<<"I only expect BEGIN_REQ on the forward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      ocpip::thread_id* tmp;
      bool thread_id_available=ocpm.get_extension<ocpip::thread_id>(tmp, gp);
      if (!thread_id_available){
        assert(threads_B==1); //if there were more than 1 thread, there should have been a thread ID
        ocpm.validate_extension<ocpip::thread_id>(gp); //NOTE: we use the multi threaded socket to set the extension, because the single threaded sockets 
                                                // do not know the extension!
        tmp->value=0; //we need a thread ID so we add it and activate it
      }

      assert(!(((tb_B->value.mask)>>(tmp->value)) & 1)); //make sure our thread is not busy
    
      
      //assert that we have not seen this thing before
      pseudo_thread_ID* check;
      acc(gp).get_extension(check);
      assert(!check);
      
      acc(gp).set_extension(&threadIDB);
      
      reqB_reg[tmp->value]=&gp; //at this time the thread id is correct

      tmp->value+=threads_A; //change threadID   
      ph=tlm::END_REQ;   
      return tlm::TLM_UPDATED;
    }

    tlm::tlm_sync_enum nb_transport_bw(tlm::tlm_generic_payload& gp, tlm::tlm_phase& ph, sc_core::sc_time& t){
      if (ph==ocpip::THREAD_BUSY_CHANGE){
        assert(ocpm.get_extension<ocpip::thread_busy>(gp)->value.type==ocpip::S_THREAD);
        stb=ocpm.get_extension<ocpip::thread_busy>(gp)->value.mask;
        return tlm::TLM_ACCEPTED;
      }
    
      if (ph!=tlm::BEGIN_RESP){
        std::cerr<<"I only expect BEGIN_RESP on the backward path, but got "<<ph<<" In "<<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      pseudo_thread_ID* threadID;
      acc(gp).get_extension(threadID);
      acc(gp).clear_extension(threadID); //remove the thing from the txn
      if (threadID==&threadIDA)
        return ocpsA->nb_transport_bw(gp, ph, t);
      else
      if (threadID==&threadIDB)
        return ocpsB->nb_transport_bw(gp, ph, t);
      else{
        std::cerr<<"There is a response which was not tagged with a local thread ID. That's strange... In "
                 <<name()<<" at "<<sc_core::sc_time_stamp()<<std::endl;
        exit(1);
      }
      return tlm::TLM_ACCEPTED;
    }

  private:
    sc_core::sc_time time_quantum;
    sc_core::sc_time sthreadbusy_sample_time;
    sc_core::sc_time request_sample_time;
    sc_core::sc_time mthreadbusy_sample_time;
    sc_core::sc_time response_in_time;

    sc_core::sc_event arbiter_event, mthreadbusy_event;
    int threads_A, threads_B, mtb_A, mtb_B, stb;
    pseudo_thread_ID threadIDA, threadIDB;
    tlm::tlm_generic_payload *tb_txn_A, *tb_txn_B, *tb_txn_m, **req_reg, **reqA_reg, **reqB_reg;
    tlm::tlm_phase tb_ph, txn_ph;
    ocpip::thread_busy *tb_A, *tb_B, *tb_m;
    sc_core::sc_time tb_time, txn_time;
};


// end of multiple inclusion protection
#endif

