///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2004 OCP-IP
// OCP-IP Confidential & Proprietary
//
//
//============================================================================
//      Project : OCP SLD WG
//       Author : Alan Kamas for Sonics, Inc.
//              : This channel is modeled on the original OCP TL2 channel by 
//              : Yann Bajot, Prosilog
//              : Joe Chou, Sonics, Inc.
//              : Anssi Haverinen, Nokia
//              : Norman Weyrich, Synopsys
//          $Id :
//
//  Description :  OCP TL 2 SystemC Channel Model
//        This performance version of the channel is event driven with new
//        timing variables for increased accuracy.
//        This is channel is backward compatible with the original OCP TL2 API.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _OCP_TL2_CHANNEL_H
#define _OCP_TL2_CHANNEL_H

#include <string>
#include <iomanip>
#include <map>
#include <deque>
#include <bitset>

#include "systemc.h"

#include "ocp_param.h"

#include "ocp_globals.h"
#include "ocp_tl2_master_if.h"
#include "ocp_tl2_slave_if.h"
#include "ocp_tl2_tmon.h"

// -----------------------------------------------------------------

#define MAX_THREADS 32

template <class Tdata, class Taddr>
class OCP_TL2_Channel :
    public sc_module,
    public OCP_TL2_MasterIF<Tdata,Taddr>,
    public OCP_TL2_SlaveIF<Tdata,Taddr>
{

  public:

    friend class OCP_TL2_TMon<Tdata,Taddr>;
    
    // SystemC macro to support the Req & Resp Accept methods
    SC_HAS_PROCESS(OCP_TL2_Channel);

    //---------------------------------------------------------------
    // constructor
    //---------------------------------------------------------------
    OCP_TL2_Channel(sc_module_name name, ostream *traceStreamPtr=NULL) :
        sc_module(name),
        m_requestInProgress(false),
        m_requestUnread(false),
        m_responseInProgress(false),
        m_responseUnread(false),
        m_mThreadBusy(0),
        m_sThreadBusy(0),
        m_clkPeriod(1,SC_NS),
        m_tMonPtr(NULL)
    { 
        // setup the SystemC methods 
        // for delayed request and response accept
        SC_METHOD(release_request_method);
        sensitive << m_ReleaseRequestEvent;
        dont_initialize();

        SC_METHOD(release_response_method);
        sensitive << m_ReleaseResponseEvent;
        dont_initialize();

        // Set up the monitor (if any)
        if (traceStreamPtr) {
            m_tMonPtr = new OCP_TL2_TMon<Tdata, Taddr> (traceStreamPtr, this);
        }
    }

    //---------------------------------------------------------------
    // destructor
    //---------------------------------------------------------------
    virtual ~OCP_TL2_Channel()
    { 
        // Clean up the transaction monitor (if any)
        if (m_tMonPtr) {
            delete m_tMonPtr;
        }
    }

  public:

    //---------------------------------------------------------------
    // Request Commands
    //---------------------------------------------------------------

    bool sendOCPRequest(const OCPTL2RequestGrp<Tdata,Taddr>& req) 
    {
        // NOTE: for performance reasons, no check for threadbusy here

        // Check to see if there is already a request in progress
        if (! m_requestInProgress)
        {
            // Put the request on the channel
            m_requestInProgress = true;
            m_requestUnread = true;
            m_currentRequest = req;
            // Trigger the event
#ifndef NDEBUG
            if (m_tMonPtr) {
                m_tMonPtr -> tLogNewReq();
            }
#endif
            m_RequestStartEvent.notify();
            return true;
        }
        // Already a request in progress. Failed.
        return false;
    }

    bool sendOCPRequestBlocking(const OCPTL2RequestGrp<Tdata,Taddr>& req) 
    {
        if (!sendOCPRequest(req)) {
            wait(m_RequestEndEvent);
            if (!sendOCPRequest(req))
            {
                return false;
            }
        }
        wait(m_RequestEndEvent);
        return true;
    }

    bool getOCPRequest(OCPTL2RequestGrp<Tdata,Taddr>& req) 
    {
        // Check to see if there is already a request in progress
        // that we haven't read yet
        if (m_requestInProgress && m_requestUnread)
        {
            // There is a request to get
            req = m_currentRequest;
            // Mark it as read
            m_requestUnread = false;
            return true;
        }
        return false;
    }

    bool getOCPRequestBlocking(OCPTL2RequestGrp<Tdata,Taddr>& req) 
    {
        if (!getOCPRequest(req)) {
            wait(m_RequestStartEvent);
            return getOCPRequest(req);
        }
        return true;
    }

    bool acceptRequest(void)
    {
        if (m_requestInProgress) {
            m_requestInProgress = false;
#ifndef NDEBUG
            if (m_requestUnread) {
                cerr << "WARNING: TL2 Channel \"" << name() 
                    << "\" - request was accepted but never read at time: " 
                    << sc_time_stamp() << endl;
            }
#endif
            m_requestUnread = false;
            m_RequestEndEvent.notify();
            return true;
        }
        return false;
    }
    
    bool acceptRequest(const sc_time& accept_time)
    {
        if (m_requestInProgress) {
            m_ReleaseRequestEvent.notify(accept_time);
            return true;
        }
        return false;
    }

    bool acceptRequest(int cycles)
    {
        if (m_requestInProgress) {
            if (cycles < 0) {
                // Do automatic acceptance based on the
                // estimated number of cycles this request
                // should take
                int estimatedCycles = getTL2ReqDuration();
                m_ReleaseRequestEvent.notify(
                        estimatedCycles * m_clkPeriod);
                return true;
            } else if (cycles == 0) {
                // Accept Right now
                return acceptRequest();
            } else {
                // wait for "cycles" clock periods and accept then
                m_ReleaseRequestEvent.notify(cycles * m_clkPeriod);
                return true;
            }
        }
        return false;
    }

    bool requestInProgress(void)
    {
        return m_requestInProgress;
    }

    //---------------------------------------------------------------
    // SystemC Methods to support delayed accepts
    //---------------------------------------------------------------
    void release_request_method(void)
    {
        acceptRequest();
    }


    //---------------------------------------------------------------
    // Old API Request Commands
    // For backward compatibility
    //---------------------------------------------------------------
    bool sendOCPRequest(const OCPRequestGrp<Tdata,Taddr>& req, 
            unsigned int ReqChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 request group
        temp_Request.copyFrom(req,ReqChunkLen,last_chunk_of_a_burst);
        
        return sendOCPRequest(temp_Request);
    }

    bool startOCPRequest(const OCPRequestGrp<Tdata,Taddr>& req, 
            unsigned int ReqChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 request group
        temp_Request.copyFrom(req,ReqChunkLen,last_chunk_of_a_burst);
        
        return sendOCPRequest(temp_Request);
    }

    bool sendOCPRequestBlocking(const OCPRequestGrp<Tdata,Taddr>& req, 
            unsigned int ReqChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 request group
        temp_sendBlkRequest.copyFrom(req,ReqChunkLen,last_chunk_of_a_burst);
        
        return sendOCPRequestBlocking(temp_sendBlkRequest);
    }

    bool startOCPRequestBlocking(const OCPRequestGrp<Tdata,Taddr>& req, 
            unsigned int ReqChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 request group
        temp_sendBlkRequest.copyFrom(req,ReqChunkLen,last_chunk_of_a_burst);
        
        if (!sendOCPRequest(temp_sendBlkRequest)) {
            wait(m_RequestEndEvent);
            if (!sendOCPRequest(temp_sendBlkRequest))
            {
                return false;
            }
        }
        return true;
    }
    
    bool getOCPRequest(OCPRequestGrp<Tdata,Taddr>& req, 
            bool accept, 
            unsigned int& ReqChunkLen, 
            bool& last_chunk_of_a_burst)
    {
        if (!getOCPRequest(temp_Request))
        {
            return false;
        } 

        // convert result to original style Request Group
        req.copyFrom(temp_Request);
        ReqChunkLen = temp_Request.DataLength;
        last_chunk_of_a_burst = temp_Request.LastOfBurst;

        // Auto accept if requested.
        if (accept) {
            return acceptRequest();
        }
        return true;
    }

    bool getOCPRequestBlocking(OCPRequestGrp<Tdata,Taddr>& req, 
            bool accept, 
            unsigned int& ReqChunkLen, 
            bool& last_chunk_of_a_burst)
    {
        if (!getOCPRequestBlocking(temp_getBlkRequest))
        {
            return false;
        } 

        // convert result to original style Request Group
        req.copyFrom(temp_getBlkRequest);
        ReqChunkLen = temp_getBlkRequest.DataLength;
        last_chunk_of_a_burst = temp_getBlkRequest.LastOfBurst;

        // Auto accept if requested.
        if (accept) {
            return acceptRequest();
        }
        return true;
    }

    bool getSBusy(void) const
    {
        return m_requestInProgress;
    }
    
    bool putSCmdAccept(void)
    {
        return acceptRequest();
    }

    // NOTE: different than original TL1/TL2 implementation
    bool putSCmdAccept(const sc_time& after)
    {
        return acceptRequest( after );
    }

    bool getSCmdAccept(void) const
    {
        return !m_requestInProgress;
    }

    void waitSCmdAccept(void)
    {
        if (!m_requestInProgress) {
            return;
        } else {
            wait(m_RequestEndEvent);
        }
    }

    //---------------------------------------------------------------
    // Response Commands
    //---------------------------------------------------------------

    bool sendOCPResponse(const OCPTL2ResponseGrp<Tdata>& resp) 
    {
        // NOTE: for performance reasons, no check for threadbusy here

        // Check to see if there is already a response in progress
        if (! m_responseInProgress)
        {
            // Put the response on the channel
            m_responseInProgress = true;
            m_responseUnread = true;
            m_currentResponse = resp;
            // Trigger the event
#ifndef NDEBUG
            if (m_tMonPtr) {
                m_tMonPtr->tLogNewResp();
            }
#endif
            m_ResponseStartEvent.notify();
            return true;
        }
        // Already a resposne in progress. Failed.
        return false;
    }

    bool sendOCPResponseBlocking(const OCPTL2ResponseGrp<Tdata>& resp) 
    {
        if(!sendOCPResponse(resp)){
            wait(m_ResponseEndEvent);
            if(!sendOCPResponse(resp)){
                return false;
            }
        }
        wait(m_ResponseEndEvent);
        return true;
    }
    
    bool getOCPResponse(OCPTL2ResponseGrp<Tdata>& resp) 
    {
        // Check to see if there is already a response in progress
        // that we haven't read yet
        if (m_responseInProgress && m_responseUnread)
        {
            // There is a response to get
            resp = m_currentResponse;
            // Mark it as read
            m_responseUnread = false;
            return true;
        }
        return false;
    }

    bool getOCPResponseBlocking(OCPTL2ResponseGrp<Tdata>& resp)
    {
        if (!getOCPResponse(resp)) {
            wait(m_ResponseStartEvent);
            return getOCPResponse(resp);
        }
        return true;
    }

    bool acceptResponse(void)
    {
        if (m_responseInProgress) {
            m_responseInProgress = false;
#ifndef NDEBUG
            if (m_responseUnread) {
                cerr << "WARNING: TL2 Channel \"" << name() 
                    << "\" - response was accepted but never read at time: " 
                    << sc_time_stamp() << endl;
            }
#endif
            m_responseUnread = false;
            m_ResponseEndEvent.notify();
            return true;
        }
        return false;
    }

    bool acceptResponse(const sc_time& accept_time)
    {
        if (m_responseInProgress) {
            m_ReleaseResponseEvent.notify(accept_time);
            return true;
        }
        return false;
    }

    bool acceptResponse(int cycles)
    {
        if (m_responseInProgress) {
            if (cycles < 0) {
                // Do automatic acceptance based on the
                // estimated number of cycles this response
                // should take
                int estimatedCycles = getTL2RespDuration();
                m_ReleaseResponseEvent.notify(
                        estimatedCycles * m_clkPeriod);
                return true;
            } else if (cycles == 0) {
                // Accept Right now
                return acceptResponse();
            } else {
                // wait for "cycles" clock periods and accept then
                m_ReleaseResponseEvent.notify(cycles * m_clkPeriod);
                return true;
            }
        }
        return false;
    }

    bool responseInProgress(void)
    {
        return m_responseInProgress;
    }
    
    //---------------------------------------------------------------
    // SystemC Methods to support delayed accepts
    //---------------------------------------------------------------
    void release_response_method(void)
    {
        acceptResponse();
    }

    //---------------------------------------------------------------
    // Old API Response Commands
    // For backward compatibility
    //---------------------------------------------------------------

    bool sendOCPResponse(const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 response group
        temp_Response.copyFrom(resp,RespChunkLen,last_chunk_of_a_burst);
        
        return sendOCPResponse(temp_Response);
    }

    bool startOCPResponse(const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 response group
        temp_Response.copyFrom(resp,RespChunkLen,last_chunk_of_a_burst);
        
        return sendOCPResponse(temp_Response);
    }

    bool sendOCPResponseBlocking(const OCPResponseGrp<Tdata>& resp, 
            unsigned int ReqChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 request group
        temp_sendBlkResponse.copyFrom(resp,ReqChunkLen,last_chunk_of_a_burst);
        
        return sendOCPResponseBlocking(temp_sendBlkResponse);
    }

    bool startOCPResponseBlocking(const OCPResponseGrp<Tdata>& resp, 
            unsigned int RespChunkLen = 1, 
            bool last_chunk_of_a_burst = true) 
    {
        // Convert to TL2 request group
        temp_sendBlkResponse.copyFrom(resp,RespChunkLen,last_chunk_of_a_burst);
        
        if (!sendOCPResponse(temp_sendBlkResponse)) {
            wait(m_ResponseEndEvent);
            if (!sendOCPResponse(temp_sendBlkResponse))
            {
                return false;
            }
        }
        return true;
    }

    bool getOCPResponse(OCPResponseGrp<Tdata>& resp, 
            bool accept, 
            unsigned int& RespChunkLen, 
            bool& last_chunk_of_a_burst)
    {
        if (!getOCPResponse(temp_Response))
        {
            return false;
        } 

        // convert result to original style Request Group
        resp.copyFrom(temp_Response);
        RespChunkLen = temp_Response.DataLength;
        last_chunk_of_a_burst = temp_Response.LastOfBurst;

        // Auto accept if requested.
        if (accept) {
            return acceptResponse();
        }
        return true;
    }

    bool getOCPResponseBlocking(OCPResponseGrp<Tdata>& resp, 
            bool accept, 
            unsigned int& RespChunkLen, 
            bool& last_chunk_of_a_burst)
    {
        if (!getOCPResponseBlocking(temp_getBlkResponse))
        {
            return false;
        } 

        // convert result to original style Response Group
        resp.copyFrom(temp_getBlkResponse);
        RespChunkLen = temp_getBlkResponse.DataLength;
        last_chunk_of_a_burst = temp_getBlkResponse.LastOfBurst;

        // Auto accept if requested.
        if (accept) {
            return acceptResponse();
        }
        return true;
    }

    bool getMBusy(void) const
    {
        return m_responseInProgress;
    }

    bool putMRespAccept(void)
    {
        return acceptResponse();
    }

    bool putMRespAccept(const sc_time& after)
    {
        return acceptResponse(after);
    }

    bool getMRespAccept(void) const
    {
        return !m_responseInProgress;
    }

    void waitMRespAccept(void)
    {
        if (!m_responseInProgress) {
            return;
        } else {
            wait(m_ResponseEndEvent);
        }
    }

    //---------------------------------------------------------------
    // Old API Serialized Commands
    // For backward compatibility
    //---------------------------------------------------------------
    bool OCPReadTransfer(const OCPRequestGrp<Tdata,Taddr>& req,
            OCPResponseGrp<Tdata>& resp,
            unsigned int TransferLen =1)
    {
        if( req.MCmd != OCP_MCMD_RD ) {
            return false;
        }
        if (getSThreadBusyBit(req.MThreadID)) {
            return false;
        }
        if (!sendOCPRequestBlocking(req,TransferLen,true)) {
            return false; 
        }
        bool lastChunk;
        return getOCPResponseBlocking(resp, true, TransferLen, lastChunk);
    }

    bool OCPWriteTransfer(const OCPRequestGrp<Tdata,Taddr>& req,
            unsigned int TransferLen =1)
    {
        if( req.MCmd != OCP_MCMD_WR ) {
            return false;
        }
        if (getSThreadBusyBit(req.MThreadID)) {
            return false;
        }
        return sendOCPRequestBlocking(req,TransferLen,true);
    } 
        
    //---------------------------------------------------------------
    // ThreadBusy Commands
    //---------------------------------------------------------------
    
    bool getSThreadBusyBit(unsigned int myThreadID = 0) const 
    { 
        // NOTE: could check to see if threadbusy is part of the channel here.
        //       (if so, then do inside #ifndef NDEBUG )
        // NOTE: could check that myThreadID is in range.
        //       (if so, then do inside #ifndef NDEBUG )

        return m_sThreadBusy[myThreadID];
    }

    void putSThreadBusyBit(bool nextBitValue, unsigned int myThreadID = 0) 
    {
        // NOTE: could check that myThreadID is in range.
        //       (if so, then do inside #ifndef NDEBUG )
        // NOTE: could check to see if threadbusy is part of the channel here.
        //       (if so, then do inside #ifndef NDEBUG )
        m_sThreadBusy[myThreadID] = nextBitValue;
        m_SThreadBusyEvent.notify();
    }

    bool getMThreadBusyBit(unsigned int myThreadID = 0) const 
    { 
        // NOTE: could check to see if threadbusy is part of the channel here.
        //       (if so, then do inside #ifndef NDEBUG )
        // NOTE: could check that myThreadID is in range.
        //       (if so, then do inside #ifndef NDEBUG )

        return m_mThreadBusy[myThreadID];
    }

    void putMThreadBusyBit(bool nextBitValue, unsigned int myThreadID = 0) 
    {
        // NOTE: could check that myThreadID is in range.
        //       (if so, then do inside #ifndef NDEBUG )
        // NOTE: could check to see if threadbusy is part of the channel here.
        //       (if so, then do inside #ifndef NDEBUG )
        m_mThreadBusy[myThreadID] = nextBitValue;
        m_MThreadBusyEvent.notify();
    }

    //---------------------------------------------------------------
    // Old API ThreadBusy Commands
    // For backward compatibility
    //---------------------------------------------------------------
    void putMThreadBusy(unsigned int nextMThreadBusy)
    {
        m_mThreadBusy = bitset<MAX_THREADS>(nextMThreadBusy);
        m_MThreadBusyEvent.notify();
    }

    void putSThreadBusy(unsigned int nextSThreadBusy)
    {
        m_sThreadBusy = bitset<MAX_THREADS>(nextSThreadBusy);
        m_SThreadBusyEvent.notify();
    }

    unsigned int getMThreadBusy(void) const
    {
        return m_mThreadBusy.to_ulong();
    }

    unsigned int getSThreadBusy(void) const
    {
        return m_sThreadBusy.to_ulong();
    }

    //---------------------------------------------------------------
    // Timing Value Functions
    //---------------------------------------------------------------

    void putMasterTiming(const MTimingGrp& mTimes)
    {
        m_mTiming = mTimes;
        m_MasterTimingEvent.notify();
    }

    void getMasterTiming(MTimingGrp& mTimes) const
    {
        mTimes = m_mTiming;
    }

    void putSlaveTiming(const STimingGrp& sTimes)
    {
        m_sTiming = sTimes;
        m_SlaveTimingEvent.notify();
    }

    void getSlaveTiming(STimingGrp& sTimes) const
    {
        sTimes = m_sTiming;
    }

    void setPeriod( const sc_time& newPeriod )
    {
        m_clkPeriod = newPeriod;
    }

    const sc_time& getPeriod(void) const 
    {
        return m_clkPeriod;
    }

    //---------------------------------------------------------------
    // Timing Helper Functions
    //---------------------------------------------------------------

    // Write Data Interval
    int getWDI(void) const   
    {
        return max(m_mTiming.DSndI, m_sTiming.DAL);
    }
    
    // Read Request Interval
    int getRqI(void) const   
    {
        return max(m_mTiming.RqSndI, m_sTiming.RqAL);
    }

    int getTL2ReqDuration(void) const    
    {
        // Add Request Data Latency only if this is a write request of some sort
        if ( (m_currentRequest.MCmd==OCP_MCMD_WR) ||
                (m_currentRequest.MCmd==OCP_MCMD_WRNP) ||
                (m_currentRequest.MCmd==OCP_MCMD_WRC) ||
                (m_currentRequest.MCmd==OCP_MCMD_BCST) ) {
            return(m_mTiming.RqDL + (getWDI())*(m_currentRequest.DataLength));
        }
        // A read request
        int numReqs = m_currentRequest.DataLength;
        if ( ocpParams.burstsinglereq ) {
            numReqs = 1;
        }
        // Duration is that time for one request times number of requests
        return((getRqI())*(numReqs) );
    }

    // Response Data Interval
    int getRDI(void) const
    {
        return max(m_sTiming.RpSndI, m_mTiming.RpAL);
    }
        
    int getTL2RespDuration(void) const
    {
        // Check for SRMD write response
        int numResp = m_currentResponse.DataLength;
        if ( ocpParams.burstsinglereq && (m_currentResponse.SDataPtr==NULL)){
            // SRMR write response
            numResp = 1;
        }
        return( (getRDI())*(numResp) );
    }

    // -------------------------------------------------------------
    // RESET
    // Reset functions for backward compatibility
    // -------------------------------------------------------------
    void MResetAssert(void)
    {
        // Set the reset signal
        m_sideband.MReset_n = false;
        // Call the reset event
        m_ResetStartEvent.notify();
    }
    void MResetDeassert(void)
    {
        // Now pull out of reset mode
        m_sideband.MReset_n = true;
        // Call the end of reset event
        m_ResetEndEvent.notify();
    }
    void SResetAssert(void)
    {
        // Set the reset signal
        m_sideband.SReset_n = false;
        // Call the reset event
        m_ResetStartEvent.notify();
    }
    void SResetDeassert(void)
    {
        // Now pull out of reset mode
        m_sideband.SReset_n = true;
        // Call the end of reset event
        m_ResetEndEvent.notify();
    }
    bool getReset(void) 
    { 
        return ((m_sideband.MReset_n==false)||(m_sideband.SReset_n==false));
    }
    // -------------------------------------------------------------
    // RESET
    // Old TL2 API & Generic Methods for Reset
    // -------------------------------------------------------------
    void reset(void)
    {
        // Set both reset signals
        m_sideband.MReset_n = false;
        m_sideband.SReset_n = false;
        // Call the reset event
        m_ResetStartEvent.notify();
    }
    void remove_reset(void)
    {
        // Clear both resets
        // Set both reset signals
        m_sideband.MReset_n = true;
        m_sideband.SReset_n = true;
        // Call the reset end event
        m_ResetEndEvent.notify();
    }
    bool get_reset(void)
    {
        return getReset();
    }
    // NOTE: Slightly different implementation then original TL2 channel.
    // NOTE: Can only be called from an SC_THREAD due to the wait.
    void Reset(void) {
        reset();
        wait(SC_ZERO_TIME);
        remove_reset();
    }
    // Sideband reset replaced with reset assert/deassert calls
    void SputSReset_n(bool nextValue)
    {
        noSupport("SputSReset_n","SResetAssert() and SResetDeassert()");
    }
    bool SgetMReset_n(void) const
    {
        noSupport("SgetMReset_n","getReset");
        return false;
    }
    // Sideband reset replaced with reset assert/deassert calls
    void MputMReset_n(bool nextValue)
    {
        noSupport("SputMReset_n","MResetAssert() and MResetDeassert()");
    }
    bool MgetSReset_n(void) const
    {
        noSupport("MgetSReset_n","getReset");
        return false;
    }
    
    // TODO:(Alan Kamas) Document sideband commands.

    //---------------------------------------------------------------------
    // SIDEBAND methods
    //---------------------------------------------------------------------

    // Master SideBand Group Access methods
    void MputMError(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.MError!=nextValue) {
            m_sideband.MError=nextValue;
            m_SidebandMasterEvent.notify();
        }
    }
    void MputMFlag(unsigned long long int nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.MFlag!=nextValue) {
            m_sideband.MFlag=nextValue;
            m_SidebandMasterEvent.notify();
        }
    }
    bool MgetSError(void) const
    {
        return m_sideband.SError;
    }
    unsigned long long int MgetSFlag(void) const
    {
        return m_sideband.SFlag;
    }
    bool MgetSInterrupt(void) const
    {
        return m_sideband.SInterrupt;
    }
    // Slave SideBand Group Access methods
    bool SgetMError(void) const
    {
        return m_sideband.MError;
    }
    unsigned long long int SgetMFlag(void) const
    {
        return m_sideband.MFlag;
    }
    void SputSError(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.SError!=nextValue) {
            m_sideband.SError=nextValue;
            m_SidebandSlaveEvent.notify();
        }
    }
    void SputSFlag(unsigned long long int nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.SFlag!=nextValue) {
            m_sideband.SFlag=nextValue;
            m_SidebandSlaveEvent.notify();
        }
    }
    void SputSInterrupt(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.SInterrupt!=nextValue) {
            m_sideband.SInterrupt=nextValue;
            m_SidebandSlaveEvent.notify();
        }
    }
    //---------------------------------------------------------------------
    // System/Core Sideband signals 
    //---------------------------------------------------------------------
    void SysputControl(unsigned int nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.Control!=nextValue) {
            m_sideband.Control=nextValue;
            m_SidebandSystemEvent.notify();
        }
    }
    bool SysgetControlBusy(void) const
    {
        return m_sideband.ControlBusy;
    }
    void SysputControlWr(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.ControlWr!=nextValue) {
            m_sideband.ControlWr=nextValue;
            m_SidebandSystemEvent.notify();
        }
    }
    unsigned int SysgetStatus(void) const
    {
        return m_sideband.Status;
    }
    bool SysgetStatusBusy(void) const
    {
        return m_sideband.StatusBusy;
    }
    void SysputStatusRd(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.StatusRd!=nextValue) {
            m_sideband.StatusRd=nextValue;
            m_SidebandSystemEvent.notify();
        }

    }
    unsigned int CgetControl(void) const
    {
        return m_sideband.Control;
    }
    void CputControlBusy(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.ControlBusy!=nextValue) {
            m_sideband.ControlBusy=nextValue;
            m_SidebandCoreEvent.notify();
        }
    }
    bool CgetControlWr(void) const
    {
        return m_sideband.ControlWr;
    }
    void CputStatus(unsigned int nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.Status!=nextValue) {
            m_sideband.Status=nextValue;
            m_SidebandCoreEvent.notify();
        }
    }
    void CputStatusBusy(bool nextValue)
    {
        // only trigger event if signal has actually changed
        if (m_sideband.StatusBusy!=nextValue) {
            m_sideband.StatusBusy=nextValue;
            m_SidebandCoreEvent.notify();
        }
    }
    bool CgetStatusRd(void) const
    {
        return m_sideband.StatusRd;
    }

    //---------------------------------------------------------------------
    // Configuration Functions
    //---------------------------------------------------------------------

    void setConfiguration( MapStringType& passedMap )
    {
        // Use the passed data base to set up the configuration class
        ocpParams.setOCPConfiguration( name(), passedMap );

#ifndef NDEBUG
        if (m_tMonPtr) {
            m_tMonPtr->tLogSetUp();
            m_tMonPtr->tLogHeader();
        }
#endif

    } 

    //---------------------------------------------------------------
    // Event Access
    //---------------------------------------------------------------

    const sc_event& RequestStartEvent(void) const
    {
        return m_RequestStartEvent;
    }
    const sc_event& RequestEndEvent(void) const
    {
        return m_RequestEndEvent;
    }
    const sc_event& ResponseStartEvent(void) const
    {
        return m_ResponseStartEvent;
    }
    const sc_event& ResponseEndEvent(void) const
    {
        return m_ResponseEndEvent;
    }
    const sc_event& SThreadBusyEvent(void) const
    {
        return m_SThreadBusyEvent;
    }
    const sc_event& MThreadBusyEvent(void) const
    {
        return m_MThreadBusyEvent;
    }
    const sc_event& MasterTimingEvent(void) const
    {
        return m_MasterTimingEvent;
    }
    const sc_event& SlaveTimingEvent(void) const
    {
        return m_SlaveTimingEvent;
    }
    const sc_event& ResetStartEvent(void) const
    {
        return m_ResetStartEvent;
    }
    const sc_event& ResetEndEvent(void) const
    {
        return m_ResetEndEvent;
    }
    const sc_event& SidebandMasterEvent(void) const
    {
        return m_SidebandMasterEvent;
    }
    const sc_event& SidebandSlaveEvent(void) const{
        return m_SidebandSlaveEvent;
    }
    const sc_event& SidebandSystemEvent(void) const
    {
        return m_SidebandSystemEvent;
    }
    const sc_event& SidebandCoreEvent(void) const
    {
        return m_SidebandCoreEvent;
    }
    const sc_event& SidebandControlEvent(void) const
    {
        noSupport("SidebandControlEvent()",
                "SidebandSystemEvent() and SidebandCoreEvent()");
        return m_SidebandCoreEvent;
    }
    const sc_event& SidebandStatusEvent(void) const
    {
        noSupport("SidebandStatusEvent()",
                "SidebandSystemEvent() and SidebandCoreEvent()");
        return m_SidebandSystemEvent;
    }

    //---------------------------------------------------------------
    // Old API Data Class Commands
    // For backward compatibility
    //---------------------------------------------------------------
    
    // "No Support" error messages
    void noSupport(string oldCommandName, string newCommandName = "noNewCommand") const
    {
        cerr << "ERROR: TL2 Channel \"" << name() 
            << "\" no longer supports the command: \"" << oldCommandName 
            << "\"." << endl;
        if (newCommandName != "noNewCommand") {
            cerr << "       Please use new command \"" 
                << newCommandName << "\" instead." << endl;
        }
        assert(0);
    }
    void noSignal(string oldSignalName, string newSignalName = "noNewSignal") const
    {
        cerr << "WARNING: The TL2 Channel \"" << name() 
            << "\" no longer supports the TL1 OCP signal: \"" << oldSignalName 
            << "\"." << endl;
        if (newSignalName != "noNewSignal") {
            cerr << "         Please use new signal \"" 
                << newSignalName << "\" instead." << endl;
        }
    }

    // Old Master Data Class Request Methods
    void MputMAddr(Taddr a) 
    {
        noSupport("MputMAddr","sendOCPRequest");
    }
    void MputMAddrSpace(unsigned int a)
    {
        noSupport("MputMAddrSpace","sendOCPRequest");
    }
    void MputMAtomicLength(unsigned int a)
    {
        noSignal("MAtomicLength");
        noSupport("MputMAtomicLength");
    }
    void MputMBurstSeq(OCPMBurstSeqType a)
    {
        noSupport("MputMBurstSeq","sendOCPRequest");
    }
    void MputMByteEn(unsigned int a)
    {
        noSupport("MputMByteEn","sendOCPRequest");
    }
    void MputMCmd(OCPMCmdType a)
    {
        noSupport("MputMCmd","sendOCPRequest");
    }
    void MputMConnID(unsigned int a)
    {
        noSupport("MputMConnID","sendOCPRequest");
    }
    void MputMData(Tdata* d, unsigned int w = 1, bool last_of_a_burst = true)
    {
        noSupport("MputMData","sendOCPRequest");
    }
    void MputMThreadID(unsigned int a) 
    {
        noSupport("MputMThreadID","sendOCPRequest");
    }
    void MputMBurstLength(unsigned int a) 
    {
        noSignal("MBurstLength","DataLength");
        noSupport("MputMBurstLength");
    }
    void MputMBurstPrecise(bool a)
    {
        noSignal("MBurstPrecise");
        noSupport("MputMBurstPrecise");
    }
    void MputMBurstSingleReq(bool a)
    {
        noSignal("MBurstSingleReq");
        noSupport("MputMBurstSingleReq");
    }
    void MputMReqInfo(unsigned int a)
    {
        noSupport("MputMReqInfo","sendOCPRequest");
    }
    void MputMReqLast(bool a)
    {
        noSignal("MReqLast");
        noSupport("MputMReqLast");
    }
    void SetRequestFields(OCPRequestGrp<Tdata,Taddr>& req, 
            unsigned int ReqChunkLen = 1, 
            bool last_chunk_of_a_burst = true)
    {
        noSupport("SetRequestFields","sendOCPRequest");
    }
    void GetRequestFields(OCPRequestGrp<Tdata,Taddr>& req, 
            bool mdata_copy,
            unsigned int& ReqChunkLen, bool& last_chunk_of_a_burst)
    {
        noSupport("GetRequestFields","getOCPRequest");
    }

    // Old Master Data Class Response Methods
    Tdata* MgetSData(unsigned int& w, bool& last_of_a_burst)
    {
        noSupport("MgetSData","getOCPResponse");
        return NULL;
    }
    Tdata* MgetSData(unsigned int& w)
    {
        noSupport("MgetSData","getOCPResponse");
        return NULL;
    }
    Tdata* MgetSData()
    {
        noSupport("MgetSData","getOCPResponse");
        return NULL;
    }
    unsigned int MgetSDataInfo()
    {
        noSupport("MgetSDataInfo","getOCPResponse");
        return 0;
    }
    OCPSRespType MgetSResp()
    {
        noSupport("MgetSResp","getOCPResponse");
        return OCP_SRESP_NULL;
    }
    unsigned int MgetSRespInfo()
    {
        noSupport("MgetSRespInfo","getOCPResponse");
        return 0;
    }
    bool MgetSRespLast()
    {
        noSignal("SRespLast","LastOfBurst");
        noSupport("MgetSRespLast");
        return 0;
    }
    unsigned int MgetSThreadID()
    {
        noSupport("MgetSThreadID","getOCPResponse");
        return 0;
    }
    void SetResponseFields(OCPResponseGrp<Tdata>& resp, unsigned int
        RespChunkLen = 1, bool last_chunk_of_a_burst = true)
    {
        noSupport("SetResponseFields","sendOCPResponse");
    }
    void GetResponseFields(OCPResponseGrp<Tdata>& resp, bool sdata_copy,
        unsigned int& RespChunkLen, bool& last_chunk_of_a_burst)
    {
        noSupport("GetResponseFields","getOCPResponse");
    }

    // Old Master DataHS Group Access methods
    void MputMDataInfo(unsigned int a)
    {
        noSupport("MputMDataInfo","sendOCPResponse");
    }

    // Old Master ThreadBusy Group Access methods
    unsigned int MgetSThreadBusy(void) const
    {
        return getSThreadBusy();
    }
    void MputMThreadBusy(unsigned int mthreadbusy)
    {
        putMThreadBusy(mthreadbusy);
    }

    // Old Master Data Class TL2 Chunk methods
    void MputMReqChunkLen(unsigned int w)
    {
        noSupport("MputMReqChunkLen","sendOCPResponse");
    }
    void MputMReqChunkLast(bool w)
    {
        noSupport("MputMReqChunkLast","sendOCPResponse");
    }
    unsigned int MgetSRespChunkLen() const
    {
        noSupport("MgetSRespChunkLen","getOCPResponse");
        return 0;
    }
    bool MgetSRespChunkLast() const
    {
        noSupport("MgetSRespChunkLast","getOCPResponse");
        return 0;
    }

    // OLD Slave Data Class Request Methods
    Taddr SgetMAddr()
    {
        noSupport("SgetMAddr","getOCPRequest");
        return 0;
    }
    unsigned int SgetMAddrSpace()
    {
        noSupport("SgetMAddrSpace","getOCPRequest");
        return 0;
    }
    unsigned int SgetMAtomicLength()
    {
        noSignal("MAtomicLenght");
        noSupport("SgetMAtomicLength");
        return 0;
    }
    OCPMBurstSeqType SgetMBurstSeq()
    {
        noSupport("SgetMBurstSeq","getOCPRequest");
        return OCP_MBURSTSEQ_UNKN;
    }
    unsigned int SgetMByteEn()
    {
        noSupport("SgetMByteEn","getOCPRequest");
        return 0;
    }
    OCPMCmdType SgetMCmd()
    {
        noSupport("SgetMCmd","getOCPRequest");
        return OCP_MCMD_IDLE;
    }
    unsigned int SgetMConnID()
    {
        noSupport("SgetMConnID","getOCPRequest");
        return 0;
    }
    Tdata* SgetMData(unsigned int& w, bool& last_of_a_burst)
    {
        noSupport("SgetMData","getOCPRequest");
        return NULL;
    }
    Tdata* SgetMData(unsigned int& w)
    {
        noSupport("SgetMData","getOCPRequest");
        return NULL;
    }
    Tdata* SgetMData()
    {
        noSupport("SgetMData","getOCPRequest");
        return NULL;
    }
    unsigned int SgetMThreadID()
    {
        noSupport("SgetMThreadID","getOCPRequest");
        return 0;
    }
    unsigned int SgetMBurstLength()
    {
        noSignal("MBurstLength","DataLength");
        noSupport("SgetMBurstLength","getOCPRequest");
        return 0;
    }
    unsigned int SgetMBurstPrecise()
    {
        noSignal("MBurstPrecise");
        noSupport("SgetMBurstPrecise");
        return 0;
    }
    unsigned int SgetMBurstSingleReq()
    {
        noSignal("MBurstSingleReq");
        noSupport("SgetMBurstSingleReq");
        return 0;
    }
    unsigned int SgetMReqInfo()
    {
        noSupport("SgetMReqInfo","getOCPRequest");
        return 0;
    }
    unsigned int SgetMReqLast()
    {
        noSignal("MReqLast","LastOfBurst");
        noSupport("SgetMReqLast","getOCPRequest");
        return 0;
    }

    // Old Slave Response Group Access Methods
    void SputSData(Tdata* d, 
            unsigned int w = 1, 
            bool last_of_a_burst = true)
    {
        noSupport("SputSData","sendOCPResponse");
    }
    void SputSDataInfo(unsigned int a)
    {
        noSupport("SputSDataInfo","sendOCPResponse");
    }
    void SputSResp(OCPSRespType a)
    {
        noSupport("SputSResp","sendOCPResponse");
    }
    void SputSRespInfo(unsigned int a)
    {
        noSupport("SputSRespInfo","sendOCPResponse");
    }
    void SputSRespLast(bool a)
    {
        noSupport("SputSRespLast","sendOCPResponse");
    }
    void SputSThreadID(unsigned int a)
    {
        noSupport("SputSThreadID","sendOCPResponse");
    }

    // Old Slave DataHS Data Class Access Methods
    unsigned int SgetMDataInfo()
    {
        noSupport("SgetMDataInfo","getOCPRequest");
        return 0;
    }

    // Old Slave ThreadBusy Data Class Access Methods
    unsigned int SgetMThreadBusy(void) const
    {
        return getMThreadBusy();
    }
    void SputSThreadBusy(unsigned int sthreadbusy)
    {
        putSThreadBusy(sthreadbusy);
    }

    // Old Slave TL2-specific Burst Chunks Data Class Methods
    void SputSRespChunkLen(unsigned int w)
    {
        noSupport("SputSRespChunkLen","putOCPResponse");
    }
    void SputSRespChunkLast(bool w)
    {
        noSupport("SputSRespChunkLast","putOCPResponse");
    }
    unsigned int SgetMReqChunkLen() const
    {
        noSupport("SgetMReqChunkLen","getOCPRequest");
        return 0;
    }
    bool SgetMReqChunkLast() const
    {
        noSupport("SgetMReqChunkLast","getOCPRequest");
        return false;
    }

    // Old Generic Methods for requests/responses synchronization
    void Mrelease(sc_time time)
    {
        acceptResponse(time);
    }
    void Srelease(sc_time time)
    {
        acceptRequest(time);
    }
    void Mrelease(void)
    {
        acceptResponse();
    }
    void Srelease(void)
    {
        acceptRequest();
    }
    void SreleaseData(sc_time time)
    {
        noSupport("SreleaseData","acceptRequest");
    }
    void SreleaseData(void)
    {
        noSupport("SreleaseData","acceptRequest");
    }
    void MreleasePE(void)
    {
        noSupport("MreleasePE");
    }
    void MunreleasePE(void)
    {
        noSupport("MunreleasePE");
    }
    void SreleasePE(void)
    {
        noSupport("SreleasePE");
    }
    void SunreleasePE(void)
    {
        noSupport("SunreleasePE");
    }
    void SreleaseDataPE(void)
    {
        noSupport("SreleaseDataPE");
    }
    void SunreleaseDataPE(void)
    {
        noSupport("SunreleaseDataPE");
    }
    bool MgetSbusy(void) const
    {
        return getSBusy();
    }
    bool MgetSbusyData(void) const
    {
        noSupport("MgetSbusyData");
        return false;
    }
    bool SgetMbusy(void) const
    {
        return getMBusy();
    }
    bool IsWrite(void)
    {
        noSupport("IsWrite");
        return false;
    }

    // Old Generic Methods to send/get requests/responses
    bool MputWriteRequest(void)
    {
        noSupport("MputWriteRequest","sendOCPRequest");
        return false;
    }
    bool MputReadRequest(void)
    {
        noSupport("MputReadRequest","sendOCPRequest");
        return false;
    }
    bool MputWriteRequestBlocking(void) 
    {
        noSupport("MputWriteRequestBlocking","sendOCPRequestBlocking");
        return false;
    }
    bool MputReadRequestBlocking(void) 
    {
        noSupport("MputReadRequestBlocking","sendOCPRequestBlocking");
        return false;
    }
    bool MputRequest(void)
    {
        noSupport("MputRequest","sendOCPRequest");
        return false;
    }
    bool MputRequestBlocking(void)
    {
        noSupport("MputRequestBlocking","sendOCPRequestBlocking");
        return false;
    }
    bool SgetRequest(bool Release)
    {
        noSupport("SgetRequest","getOCPRequest");
        return false;
    }
    bool SgetRequestBlocking(bool Release)
    {
        noSupport("SgetRequestBlocking","getOCPRequestBlocking");
        return false;
    }
    bool SgetRequestPE(void)
    {
        noSupport("SgetRequestPE","getOCPRequest");
        return false;
    }
    bool SputResponse(void)
    {
        noSupport("SputResponse","sendOCPResponse");
        return false;
    }
    bool SputResponseBlocking(void)
    {
        noSupport("SputResponseBlocking","sendOCPResponseBlocking");
        return false;
    }
    bool MgetResponse(bool Release) 
    {
        noSupport("MgetResponse","getOCPResponse");
        return false;
    }
    bool MgetResponseBlocking(bool Release)
    {
        noSupport("MgetResponseBlocking","getOCPResponseBlocking");
        return false;
    }
    bool MgetResponsePE(void) 
    {
        noSupport("MgetResponsePE","getOCPResponsePE");
        return false;
    }
    bool MputDataRequest(void)
    {
        noSupport("MputDataRequest","sendOCPRequest");
        return false;
    }
    bool MputDataRequestBlocking(void)
    {
        noSupport("MputDataRequestBlocking","sendOCPRequestBlocking");
        return false;
    }
    bool SgetDataRequest(bool Release)
    {
        noSupport("SgetDataRequest","getOCPRequest");
        return false;
    }
    bool SgetDataRequestBlocking(bool Release)
    {
        noSupport("SgetDataRequestBlocking","getOCPRequestBlocking");
        return false;
    }
    bool SgetDataRequestPE(void)
    {
        noSupport("SgetDataRequestPE","getOCPRequest");
        return false;
    }

    // Old Generic Methods for direct access
    virtual bool MputDirect(int MasterID, bool IsWrite, 
            Tdata *Data, Taddr Address, int NumWords)
    {
        noSupport("MputDirect");
        return false;
    }
    virtual bool SputDirect(int SlaveID, bool IsWrite, 
            Tdata *Data, Taddr Address, int NumWords)
    {
        noSupport("SputDirect");
        return false;
    }
    virtual void register_port(sc_port_base& port, const char* if_typename)
    {
        // Do nothing. New TL2 Channel does not track who is connected to it.
    }

    // TODO:(Alan Kamas) These error tab function cannot be compiled as the 
    //   data class template class is undefined (ironically - that's the error.)
    //   These commented-out member functions should be removed. 
    //   
    // void SregisterDirectIF(SdirectIF<OCP_TL2_DataCl<Tdata,Taddr> >* A)
    // {
        // noSupport("SregisterDirectIF");
    // }
    // void MregisterDirectIF(MdirectIF<OCP_TL2_DataCl<Tdata,Taddr> >* A)
    // {
        // noSupport("MregisterDirectIF");
    // }

    // OLD 'Param', 'Data' and 'Comm' class accesses 
    // TODO:(Alan Kamas) The CommCl tab function cannot be compiled as the 
    //   CommCl class is undefined (ironically - that's the error.)
    //   These commented-out member functions should be removed. 
    //   
    //  CommCl* GetCommCl(void)
    //  {
        //  noSupport("GetCommCl");
        //  return NULL;
    //  }
    // TODO:(Alan Kamas) These error tab function cannot be compiled as the 
    //   data class template class is undefined (ironically - that's the error.)
    //  OCP_TL2_DataCl<Tdata,Taddr> * GetDataCl(void)
    //  {
        //  noSupport("GetDataCl");
        //  return NULL;
    //  }
    //  ParamCl<OCP_TL2_DataCl<Tdata,Taddr> >* GetParamCl(void)
    //  {
        //  noSupport("ParamCl<OCP_TL2_DataCl<Tdata,Taddr> >* GetParamCl",
                //  "OCPParameters* GetParamCl");
        //  return NULL;
    //  }
    // New alternative function
    OCPParameters* GetParamCl(void)
    {
        return &(ocpParams);
    }

    //----------------------------------------------------------------------
    // member data
    //----------------------------------------------------------------------
  public:
    // Channel Parameters
    OCPParameters ocpParams;

  protected:
    // Channel State
    bool m_requestInProgress;
    bool m_requestUnread;
    bool m_responseInProgress;
    bool m_responseUnread;

    // Channel Data
    OCPTL2RequestGrp<Tdata,Taddr> m_currentRequest; 
    OCPTL2ResponseGrp<Tdata> m_currentResponse; 
    
    // ThreadBusy Data
    // TODO (Alan Kamas): the "MAX_THREADS" should be based on the number of threads.
    bitset<MAX_THREADS> m_mThreadBusy;
    bitset<MAX_THREADS> m_sThreadBusy;

    // Timing Data
    MTimingGrp m_mTiming;
    STimingGrp m_sTiming;
    sc_time m_clkPeriod;

    // Sideband Data
    OCPSidebandGrp m_sideband;

    // Channel Events
    sc_event m_RequestStartEvent;
    sc_event m_RequestEndEvent;
    sc_event m_ResponseStartEvent;
    sc_event m_ResponseEndEvent;
    sc_event m_MThreadBusyEvent;
    sc_event m_SThreadBusyEvent;
    sc_event m_MasterTimingEvent;
    sc_event m_SlaveTimingEvent;
    sc_event m_ResetStartEvent;
    sc_event m_ResetEndEvent;

    // Sideband Events
    sc_event m_SidebandMasterEvent;
    sc_event m_SidebandSlaveEvent;
    sc_event m_SidebandSystemEvent;
    sc_event m_SidebandCoreEvent;
    
    // Delayed Acceptance Events
    sc_event m_ReleaseRequestEvent;
    sc_event m_ReleaseResponseEvent;
    
    // Transaction Logging
    OCP_TL2_TMon<Tdata,Taddr> *m_tMonPtr;

    // Temp Variables
    // members for efficiency only (to avoid new/deletes)
    OCPTL2RequestGrp<Tdata,Taddr> temp_Request;
    OCPTL2RequestGrp<Tdata,Taddr> temp_sendBlkRequest;
    OCPTL2RequestGrp<Tdata,Taddr> temp_getBlkRequest;
    OCPTL2ResponseGrp<Tdata> temp_Response;
    OCPTL2ResponseGrp<Tdata> temp_sendBlkResponse;
    OCPTL2ResponseGrp<Tdata> temp_getBlkResponse;
    WaitReq<Tdata,Taddr> temp_waitReq;
};

#endif  // _OCP_TL2_CHANNEL_H
