// 
//  (c) Copyright OCP-IP 2003
//  OCP-IP Confidential and Proprietary
//
// ============================================================================
//      Project : OCP SLD WG, OCP Transaction Level Layer-1
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com 
//                Alan Kamas, aok@sonicsinc.com
//         Date: 06/13/2003
//
//  Description : OCP Transaction Level 1 Channel Checker
//                - to prevent overrun between OCP request, response,
//                  and dataHS phases
//                - to ensure that the different phases of a transaction
//                  (request, data handshake, and response) complete in
//                  the OCP correct order.
//                 
// =========================================================================

#ifndef _OCP_TL1_CHECKER_CL_h
#define _OCP_TL1_CHECKER_CL_h

#include <string>
#include "ocp_globals.h"
#include "systemc.h"

template <class Td, class Ta>
class OCP_TL1_CheckerCl {
  
  public:

    // Constructor
    OCP_TL1_CheckerCl(unsigned short num_threads,
                      bool writeresp_enable = false,
                      string my_name="",
                      bool *fatalErrorFlagPtr = NULL)
      : m_threadCount(num_threads),
        m_WriteRespEnable(writeresp_enable),
        m_name(my_name),
        m_fatalErrorFlagPtr(fatalErrorFlagPtr),
        m_pReadWriteRespSendCredit(NULL),
        m_pReadWriteRespAcceptCredit(NULL),
        m_pWriteDataSendCredit(NULL),
        m_pWriteDataAcceptCredit(NULL),
        m_Reset(false)
    {
        initCounters();
    }

    // Destructor
    virtual ~OCP_TL1_CheckerCl()
    {
        eraseCounters();
    }
    
    // Master Data Flow
    void
    newRequestArrived(const OCPRequestGrp<Td,Ta>& request)
    {
        assert(request.MThreadID <= m_threadCount);
        switch (request.MCmd) {
        case OCP_MCMD_IDLE:
            // do nothing
            break;
        case OCP_MCMD_RD:
        case OCP_MCMD_RDEX:
        case OCP_MCMD_RDL:
            if (request.MBurstSingleReq) {
                // Single Request / Multiple Responses
                m_pReadWriteRespSendCredit[request.MThreadID]+=request.MBurstLength;
            } else {
                // One Request / One Response
                m_pReadWriteRespSendCredit[request.MThreadID]++;
            }
            break;
        case OCP_MCMD_WR:
        case OCP_MCMD_WRC:
        case OCP_MCMD_BCST:
        case OCP_MCMD_WRNP:
            if (m_WriteDataSentAhead) {
                if ( (m_WriteDataSentThread != request.MThreadID) || (m_WriteDataSentTime != sc_simulation_time()) ) {
                    // We have an error. Ignore it if we are in reset.
                    if (!m_Reset) {
                        // Write Sent too late / Data sent too early
                        cout << "ERROR in OCP Channel " << m_name << " found by ocp_checker at simulation time = " << sc_simulation_time() << endl;
                        cout << "requestArrived: write request arrived at least one cycle after data handshake"
                            << endl;
                        if (m_fatalErrorFlagPtr != NULL) {
                            *m_fatalErrorFlagPtr = true;
                        } else {
                            exit(-1000);
                        }
                    }
                } else {
                    m_WriteDataSentAhead = false;
                }
            }
            if (request.MBurstSingleReq) {
                // Single Request / Multiple Data
                m_pWriteDataSendCredit[ request.MThreadID ]+=request.MBurstLength;
            } else {
                // One Request / One Data
                m_pWriteDataSendCredit[ request.MThreadID ]++;
            }
            if (m_WriteRespEnable || (request.MCmd==OCP_MCMD_WRNP)) {
                m_pReadWriteRespSendCredit[request.MThreadID]++;
            }
            break;
        }
    }

    void
    newDataArrived(const OCPDataHSGrp<Td>& data)
    {
        assert(data.MDataThreadID <= m_threadCount);
        if (m_pWriteDataSendCredit[data.MDataThreadID] > 0) {
            m_pWriteDataSendCredit[data.MDataThreadID]--;
        } else if (m_pWriteDataSendCredit[data.MDataThreadID] == 0) {
            // Data came ahead of the request. This is still OK as long
            // as the request associated with this data arrives this cycle.
            m_pWriteDataSendCredit[data.MDataThreadID]--;
            // These flags used to check that the write Request does indeed
            // arrive this cycle
            m_WriteDataSentAhead = true;
            m_WriteDataSentTime = sc_simulation_time();
            m_WriteDataSentThread = data.MDataThreadID;
        } else {
            // We have an error. Ignore it if we are in reset.
            if (!m_Reset) {
                cout << "ERROR in OCP Channel " << m_name << " found by ocp_checker at simulation time = " << sc_simulation_time() << endl;
                cout << "newDataArrived: data sent ahead of its request"
                    << endl;
                if (m_fatalErrorFlagPtr != NULL) {
                    *m_fatalErrorFlagPtr = true;
                } else {
                    exit(-1000);
                }
            }
        }
    }

    void
    responseAccepted(const OCPResponseGrp<Td>& response )
    {
        assert(response.SThreadID <= m_threadCount);
        if (m_pReadWriteRespAcceptCredit[response.SThreadID] > 0) {
            m_pReadWriteRespAcceptCredit[response.SThreadID]--;
        } else {
            // We have an error. Ignore it if we are in reset.
            if (!m_Reset) {
                cout << "ERROR in OCP Channel " << m_name << " found by ocp_checker at simulation time = " << sc_simulation_time() << endl;
                cout << "responseAccepted: response accepted ahead of its request"
                    << endl;
                if (m_fatalErrorFlagPtr != NULL) {
                    *m_fatalErrorFlagPtr = true;
                } else {
                    exit(-1001);
                }
            }
        }
    }

    /* Slave Data Flow */
    void
    requestAccepted(const OCPRequestGrp<Td,Ta>& request)
    {
        assert(request.MThreadID <= m_threadCount);
        switch (request.MCmd) {
        case OCP_MCMD_IDLE:
            // do nothing
            break;
        case OCP_MCMD_RD:
        case OCP_MCMD_RDEX:
        case OCP_MCMD_RDL:
            if (request.MBurstSingleReq) {
                // Single Request / Multiple Responses
                m_pReadWriteRespAcceptCredit[request.MThreadID] += request.MBurstLength;
            } else {
                // One Request / One Response
                m_pReadWriteRespAcceptCredit[request.MThreadID]++;
            }
            break;
        case OCP_MCMD_WR:
        case OCP_MCMD_WRC:
        case OCP_MCMD_BCST:
        case OCP_MCMD_WRNP:
            if (m_WriteDataAcceptAhead) {
                if ( (m_WriteDataAcceptThread != request.MThreadID) || (m_WriteDataAcceptTime != sc_simulation_time()) ) {
                    // We have an error. Ignore it if we are in reset.
                    if (!m_Reset) {
                        // Write Accepted too late / Data accepted too early
                        cout << "ERROR in OCP Channel " << m_name << " found by ocp_checker at simulation time = " << sc_simulation_time() << endl;
                        cout << "requestAccepted: write request accepted at least one cycle after data handshake"
                            << endl;
                        if (m_fatalErrorFlagPtr != NULL) {
                            *m_fatalErrorFlagPtr = true;
                        } else {
                            exit(-1002);
                        }
                    }
                } else {
                    m_WriteDataAcceptAhead = false;
                }
            }

            if (request.MBurstSingleReq) {
                // Okay to accept multiple data
                m_pWriteDataAcceptCredit[ request.MThreadID ]+=request.MBurstLength;
            } else {
                // Okay to accept one data
                m_pWriteDataAcceptCredit[request.MThreadID]++;
            }

            if (m_WriteRespEnable || (request.MCmd == OCP_MCMD_WRNP) ) {
                m_pReadWriteRespAcceptCredit[request.MThreadID]++;
            }
            break;
        }
    }

    void
    dataAccepted(const OCPDataHSGrp<Td>& data)
    {
        assert(data.MDataThreadID <= m_threadCount);
        if (m_pWriteDataAcceptCredit[data.MDataThreadID] > 0) {
            m_pWriteDataAcceptCredit[data.MDataThreadID]--;
        } else if (m_pWriteDataAcceptCredit[data.MDataThreadID] == 0) {
            // Data was accepted ahead of the write Request, but this is okay
            // if the write request is accepted sometime during this cycle
            m_pWriteDataAcceptCredit[data.MDataThreadID]--;
            // These flags used to check that the write request accept 
            // does indeed arrive this cycle
            m_WriteDataAcceptAhead = true;
            m_WriteDataAcceptTime = sc_simulation_time();
            m_WriteDataAcceptThread = data.MDataThreadID;
        } else {
            // Data accepted way too early
            // We have an error. Ignore it if we are in reset.
            if (!m_Reset) {
                cout << "ERROR in OCP Channel " << m_name << " found by ocp_checker at simulation time = " << sc_simulation_time() << endl;
                cout << "dataAccepted: data accepted ahead of its request"
                    << endl;
                if (m_fatalErrorFlagPtr != NULL) {
                    *m_fatalErrorFlagPtr = true;
                } else {
                    exit(-1002);
                }
            }
        }
    }

    void
    newResponseArrived(const OCPResponseGrp<Td>& response)
    {
        assert(response.SThreadID <= m_threadCount);
        if (m_pReadWriteRespSendCredit[response.SThreadID] > 0) {
            m_pReadWriteRespSendCredit[response.SThreadID]--;
        } else {
            // We have an error. Ignore it if we are in reset.
            if (!m_Reset) {
                cout << "ERROR in OCP Channel " << m_name << " found by ocp_checker at simulation time = " << sc_simulation_time() << endl;
                cout << "newResponseArrived: response sent ahead of its request"
                    << endl;
                if (m_fatalErrorFlagPtr != NULL) {
                    *m_fatalErrorFlagPtr = true;
                } else {
                    exit(-1003);
                }
            }
        }
    }

    void
    resetState(void)
    {
        // zero out all counters;
        for (int i = 0; i < m_threadCount; i++) {
            m_pReadWriteRespSendCredit[i] = 0;
            m_pReadWriteRespAcceptCredit[i] = 0;
            m_pWriteDataSendCredit [i] = 0;
            m_pWriteDataAcceptCredit[i] = 0;
        }

        m_WriteDataAcceptAhead = false;
        m_WriteDataAcceptTime = 0;
        m_WriteDataAcceptThread = 0;

        m_WriteDataSentAhead = false;
        m_WriteDataSentTime = 0;
        m_WriteDataSentThread = 0;

    }

    void
    startReset(void)
    {
        // Set the reset Flag
        m_Reset = true;
        // Reset our state
        resetState();
    }

    void
    endReset(void)
    {
        // Reset our State
        resetState();
        // Unset the reset Flag
        m_Reset = false;
    }

  protected:

    unsigned short m_threadCount;
    bool  m_WriteRespEnable;
    string m_name;
    bool* m_fatalErrorFlagPtr;
    int*  m_pReadWriteRespSendCredit;
    int*  m_pReadWriteRespAcceptCredit;
    int*  m_pWriteDataSendCredit;
    int*  m_pWriteDataAcceptCredit;
    bool  m_WriteDataAcceptAhead;
    double  m_WriteDataAcceptTime;
    unsigned int m_WriteDataAcceptThread;
    bool  m_WriteDataSentAhead;
    double  m_WriteDataSentTime;
    unsigned int m_WriteDataSentThread;
    bool m_Reset;
    
    // make default constructor, copying, and assignment not accessible
    OCP_TL1_CheckerCl();
    OCP_TL1_CheckerCl(const OCP_TL1_CheckerCl& other);
    void operator=(const OCP_TL1_CheckerCl& other);
    
    void
    initCounters()
    {
        m_pReadWriteRespSendCredit = new int[m_threadCount];
        assert(m_pReadWriteRespSendCredit);

        m_pReadWriteRespAcceptCredit = new int[m_threadCount];
        assert(m_pReadWriteRespAcceptCredit);

        m_pWriteDataSendCredit = new int[m_threadCount];
        assert(m_pWriteDataSendCredit);

        m_pWriteDataAcceptCredit = new int[m_threadCount];
        assert(m_pWriteDataAcceptCredit);

        resetState();
    }

    void
    eraseCounters(void)
    {
        // need "delete[]" as these are all arrays
        delete[] m_pReadWriteRespSendCredit;
        delete[] m_pReadWriteRespAcceptCredit;
        delete[] m_pWriteDataSendCredit;
        delete[] m_pWriteDataAcceptCredit;
        m_pReadWriteRespSendCredit = NULL;
        m_pReadWriteRespAcceptCredit = NULL;
        m_pWriteDataSendCredit = NULL;
        m_pWriteDataAcceptCredit = NULL;
    }
};

#endif // _OCP_TL1_CHECKER_CL_h
