// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlReader.cc,v 1.1 2007/01/25 22:09:07 halexan Exp $

#include "StlReader.h"
#include <iostream>
#include <string>
#include <glob.h>

////////////////////////////////////////////////////////////////////////////
using namespace std;
using namespace Sonics;
using namespace OcpIp;

#ifndef TEMPLATE_DEF_INCLUDE

StlReader::StlReader( const string& fileName,
                      const OcpParams& params,
                      bool  coreSystem ) :
    LineBuffer( fileName ),
    StlParser( params, coreSystem )
{}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlReader::next
// DESCRIPTION: retrieves the STL commands produced by the next
// line of STL in the input stream.
// ARGUMENTS: bool reference to determine eof. Caller is responsible
// for checking it and not call the function after eof==true
// RETURNS: Nothing
// SIDE EFFECTS: If next line in stream is valid, current STL
// command will be updated
//////////////////////////////////////////////////////////////////
void
StlReader::next( bool& eof )
{
    int len;
    const string stlLine = getNextLine( len );
    eof = ( len == -1 );
    if ( eof )
        return;
    readLine( stlLine );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlReader::setErrStream
// getCommandDelay, getCommandType, getLineNum, getFileName
// DESCRIPTION: interface functions: simply redirect to the base
// class that implements it
//////////////////////////////////////////////////////////////////
void
StlReader::setErrStream( ostream& err )
{
    StlParser::m_pErr = &err;
}
void
StlReader::setDefaultThread( uint32_t thread )
{
    StlParser::setDefaultThread( thread );
}
uint32_t
StlReader::getCommandDelay() const
{
    return StlParser::m_commandDelay;
}
StlCommandType
StlReader::getCommandType() const
{
    return StlParser::m_commandType;
}
uint32_t
StlReader::getLineNum() const
{
    return LineBuffer::getLineNum();
}
string
StlReader::getFileName() const
{
    return LineBuffer::m_fileName;
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlReader::getXCommand
// DESCRIPTION: Retrieves the current STL command
//////////////////////////////////////////////////////////////////
StlWaitCommand
StlReader::getWaitCommand() const {
    return StlParser::getWaitCommand();
}

StlSignalCommand
StlReader::getSignalCommand() const {
    return StlParser::getSignalCommand();
}

StlResetCommand
StlReader::getResetCommand() const {
    return StlParser::getResetCommand();
}

StlControlStatusCommand
StlReader::getControlStatusCommand() const {
    return StlParser::getControlStatusCommand();
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlReader::reportError
// DESCRIPTION: We are overriding this because we now have a file
// name and a line number to report. Causes some code duplication
// could be cleaned up with further function partitioning.
// ARGUMENTS: line where error occurred, and parse_info result
// RETURNS: Nothing
//////////////////////////////////////////////////////////////////
void
StlReader::reportError( const string& stlLine, parse_info<> info )
{
    assert( !info.full );
    assert( m_pErr != NULL );
    ostream& err = *m_pErr;
    err << m_fileName << " line " << getLineNum() << ": ";
    StlParser::reportError( stlLine, info );
}
void
StlReader::reportError( const string& stlLine, const string& errors )
{
    assert( m_pErr != NULL );
    ostream& err = *m_pErr;
    err << m_fileName << " line " << getLineNum() << ": ";
    StlParser::reportError( stlLine, errors );
}

//////////////////////////////////////////////////////////////////
// FUNCTION: StlReader::findStlPrograms (static)
// DESCRIPTION: Based on the <instance>.S or <instance>_<module>.S
// conventions, globs all the matching files in a directory and
// creates an STL reader for each of them
// ARGUMENTS: IChipDMCore_1 pointer for a qmaster2 instance
// RETURNS: vector of StlReader pointers. The caller owns the pointers
//////////////////////////////////////////////////////////////////
deque<IStlReader*>
IStlReader::findStlPrograms( const char* instanceName      , bool coreSystem,
                             const OCPParameters& intParams, const string& directory )
{
    deque<IStlReader*> ret;

    const string  stlExtension  = ".stl";

    string dirslash = directory;
    if ( !dirslash.empty() && ( *dirslash.rbegin() != '/' ) )
        dirslash += '/';

    const char *dot1   = instanceName;

    glob_t theglob;
    for (;;dot1++) {
        const string pattern1 = dirslash + dot1 + ".*" + stlExtension;
        if ( ! glob( pattern1.c_str(), GLOB_NOSORT, NULL, &theglob ) ) {
	    int matches = theglob.gl_pathc;
            for ( int i=0; i < matches; ++i ) {
                ret.push_back( create( instanceName, coreSystem,
                                       intParams, theglob.gl_pathv[i] ) );
            }
            if (matches) break;
        }
        const string pattern2 = dirslash + dot1 + stlExtension;
        if ( ! glob( pattern2.c_str(), GLOB_NOSORT, NULL, &theglob ) ) {
            int matches = theglob.gl_pathc;
            for ( int i=0; i < matches; ++i ) {
                ret.push_back( create( instanceName, coreSystem,
                                       intParams   , theglob.gl_pathv[i] ) );
            }
            if (matches) break;
        }

        if (!(dot1 = strstr(dot1,"."))) break;
    } 
    globfree( &theglob );
    return ret;
}

IStlReader*
IStlReader::create( const char* instanceName      , bool coreSystem,
                    const OCPParameters& intParams, const string& fileName )
{
    // look for a name qualifier in the middle ( <instanceName><qualifier>.stl )
    const string  stlExtension = ".stl";
    string qualifier = fileName;
    size_t instancePos = fileName.find( instanceName );
    if ( instancePos != string::npos )
        qualifier = fileName.substr( instancePos + strlen( instanceName ),
                                     string::npos );
    size_t extensionPos = qualifier.rfind( stlExtension );
    if ( extensionPos != string::npos )
        qualifier = qualifier.erase( extensionPos );
    // maybe .<thread>
    uint32_t defaultThread( 0 );
    size_t dotpos = qualifier.rfind( "." );
    if ( dotpos != string::npos ) {
        string postDot = qualifier.substr( dotpos+1 );
        defaultThread = strtoul( postDot.c_str(), 0, 10 );
    }

    OcpParams params = convertInterfaceToParamSet( intParams );
    StlReader* pReader = new StlReader( fileName, params, coreSystem );
    pReader->setDefaultThread( defaultThread );
    return pReader;
}    

#else // TEMPLATE_DEF_INCLUDE
//////////////////////////////////////////////////////////////////
// FUNCTION: template IStlReader::getTransferCommand()
// DESCRIPTION: conversion function to retrieve the transfer command
// with the desired width
// ARGUMENTS: None
// NOTE: This function is odd because it is a peer of the
// getXCommands() in IStlReader that are pure virtual.
// Only this one is templated and a virtual declaration is a can of worms
//////////////////////////////////////////////////////////////////
template<typename Td, typename Ta>
void
IStlReader::setConversionTypes() {
    StlParser* pParser = dynamic_cast<StlParser*>( this );
    assert(NULL != pParser);
    delete pParser->m_pCommandConverter;
    pParser->m_pCommandConverter = new StlParser::CommandConverterImpl<Td, Ta>
      ( pParser->m_ocpParams, pParser->m_transferCmd );
									       
}

template<typename Td, typename Ta>
StlTransferCommand<Td,Ta>
IStlReader::getTransferCommand() const {
    assert( getCommandType() == STL_TRANSFER );
    const StlParser* pParser = dynamic_cast<const StlParser*>( this );
    StlParser::CommandConverterImpl<Td, Ta>* pConv =
        dynamic_cast<StlParser::CommandConverterImpl<Td, Ta>*>(
            pParser->m_pCommandConverter );
    assert( pConv != NULL );
    return pConv->m_command;
}

#endif // TEMPLATE_DEF_INCLUDE
