// ============================================================================
//      Project :  OCP SLD WG, OCP Transaction Level
//       Author : Norman Weyrich, Synopsys Inc., weyrich@synopsys.com
//                Anssi Haverinen, Nokia Inc., anssi.haverinen@nokia.com
//                Joe Chou, Sonics Inc., joechou@sonicsinc.com
//         Date : 07/11/2003
//
//  Description : Transaction Level - Layer-2 data class to model
//                the Open Core Protocol (OCP) 1.0
//                 The data class can be used in conjunction with
//                OCP transaction channel (tl_channel.h) to transport
//                data between Initiator and Target modules.
//                Contains member variables and access methods
//                to set or querry member variables.
//                The member variables are split in to two groups:
//                - Request-Channel members (set by Master read by Slave)
//                - Response-Channel members (set by Slave, read by Master)
//
//                See definition of these groups in OCP 1.0 specification
// ============================================================================
//
//  Copyright @ 2003 OCP-IP
//
// ============================================================================

// This file contains the definition of the classes:
// template <class Td, class Ta> class OCP_TL2_DataCl

#ifndef _OCP_TL2_DATA_CL
#define _OCP_TL2_DATA_CL

#include "systemc.h"

#ifndef DATAFLOW_ENUM
#define DATAFLOW_ENUM

// -------------------------------------------------
// MCmd encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMCmdType {
  OCP_MCMD_IDLE = 0,
  OCP_MCMD_WR,
  OCP_MCMD_RD,
  OCP_MCMD_RDEX,
  OCP_MCMD_RESERVED4,
  OCP_MCMD_WRNP,
  OCP_MCMD_RESERVED6,
  OCP_MCMD_BCST
};

// -------------------------------------------------
// SResp encoding (OCP 1.0)
// -------------------------------------------------
enum OCPSRespType {
  OCP_SRESP_NULL = 0,
  OCP_SRESP_DVA,
  OCP_SRESP_RESERVED2,
  OCP_SRESP_ERR
};

// -------------------------------------------------
// MBurst encoding (OCP 1.0)
// -------------------------------------------------
enum OCPMBurstType {
  OCP_MBURST_LAST = 0,
  OCP_MBURST_DFLT1,
  OCP_MBURST_TWO,
  OCP_MBURST_DFLT2,
  OCP_MBURST_FOUR,
  OCP_MBURST_STRM,
  OCP_MBURST_EIGHT,
  OCP_MBURST_CONT
};

#endif

// -------------------------------------------------
//
// -------------------------------------------------
template <class Td, class Ta> class OCP_TL2_DataCl : public sc_prim_channel
{
public:

  typedef Td DataType;
  typedef Ta AddrType;

  // -------------------------------------------------
  // sideband signals
  // -------------------------------------------------
  struct OCPSidebandSignals {
    // reset signal
    bool Reset_n;

    // system/core signals
    int  Control;
    bool ControlWr;
    bool ControlBusy;
    int  Status;
    bool StatusRd;
    bool StatusBusy;

    // sideband signals
    int  MFlag;
    bool MError;
    int  SFlag;
    bool SError;
    bool SInterrupt;

    void reset()
    {
      SInterrupt = false;
      SError = false;
      MError = false;
      MFlag = 0;
      SFlag = 0;
      Control = 0;;
      ControlWr = false;
      ControlBusy = false;
      Status = 0;
      StatusRd = false;
      StatusBusy = false;
    }

    OCPSidebandSignals()
    {
      Reset_n = true;
      reset();
    }
  };

  // Constructor
  OCP_TL2_DataCl()
  {
    Reset();
  }

  // Destructor
  ~OCP_TL2_DataCl()
  {
  }

  void Reset()
  {
    // Request Signal Group
    m_MAddr = 0;
    m_MAddrSpace = 0;
    m_MBurst = OCP_MBURST_LAST;
    m_MByteEn  = 0;
    m_MCmd = OCP_MCMD_IDLE;
    m_MConnID = 0;
    m_MData = NULL;
    m_MThreadID = 0;
    
    //
    m_MDataLast = true;
    m_MDataWordLength = 0;

    //
    m_MThreadBusy= false;

    // Response Signal Group
    m_SData = NULL;
    m_SResp = OCP_SRESP_NULL;
    m_SThreadID = 0;

    //
    m_SAddr = 0;
    m_SCmd = OCP_MCMD_IDLE;
    m_SConnID = 0;

    //
    m_SThreadBusy = false;

    //
    m_SDataLast = true;
    m_SDataWordLength = 0;

    //
    m_Sidebands.reset();
  }

//---------------------------------------------------------------------
// -------------- Start of access methods -------------------------------

// These three methods are used in the Channel,
// and hence they must always be present!

  bool IsWriteRequest()
  {
    return((m_MCmd == OCP_MCMD_WR) ? true : false);
  }
  void SetWriteRequest()
  {
    m_MCmd = OCP_MCMD_WR;
  }
  void SetReadRequest() 
  {
    m_MCmd = OCP_MCMD_RD;
  }


  // Mandatory channel calls, not used in TL2
  void ToggleRequest(){}

  void ToggleDataRequest()  {}

  void ToggleResponse() {}

  // Time stamp methods: These methods could e.g. be used by
  // Masters and Slaves to indicate a duration of a packet.
  void MputEndTime(sc_time tt)
  {
    ReqEndTime = tt;
  }
  sc_time SgetEndTime()
  {
    return ReqEndTime;
  }
  void SputEndTime(sc_time tt)
  {
    ResEndTime = tt;
  }
  sc_time MgetEndTime()
  {
    return ResEndTime;
  }

//-----------------------------------------------------------
// The following methods are needed to run the examples
// We recommend to use them in user applications as well
//
// These are access methods for data fields to
//   - make code more readable and
//   - make Master and Slave code independent of the
//     data structure. If a different data structure is used
//     only the data class must be changed but not Master and Slave code.


  // ---------- Request Signal Group ----------


  // Access methods for MAddr
  void MputMAddr(Ta a)
  {
    m_MAddr = a;
  }
  Ta SgetMAddr()
  {
    return m_MAddr;
  }

  // Access methods for MAddrSpace
  void MputMAddrSpace(int a)
  {
    m_MAddrSpace = a;
  }
  int SgetMAddrSpace()
  {
    return m_MAddrSpace;
  }

  // Access methods for MBurst
  void MputMBurst(OCPMBurstType a)
  {
    m_MBurst = a;
  }
  OCPMBurstType SgetMBurst()
  {
    return m_MBurst;
  }

  // Access methods for MByteEn
  void m_MputMByteEn(int a)
  {
    m_MByteEn = a;
  }
  int SgetMByteEn()
  {
    return m_MByteEn;
  }

  // Access methods for MCmd
  void MputMCmd(OCPMCmdType a)
  {
    m_MCmd = a;
  }
  OCPMCmdType SgetMCmd()
  {
    OCPMCmdType a = m_MCmd;
    m_MCmd = OCP_MCMD_IDLE; // reset
    return a;
  }

  // Access methods for MConnID
  void MputMConnID(int a)
  {
    m_MConnID = a;
  }
  int SgetMConnID()
  {
    return m_MConnID;
  }

  // Automic length. Not part of OCP 1.0, but necessary for TL2 to work
  void MputAtomicLen(int w) {
    m_MDataWordLength = w;
  }
  int SgetAtomicLen() {
    return m_MDataWordLength;
  }
  void SputAtomicLen(int w) {
    m_SDataWordLength = w;
  }
  int MgetAtomicLen() {
    return m_SDataWordLength;
  }


  // Access methods for MData
  void MputMData(Td* d, int w = 1, bool last_of_a_burst = true)
  {
    m_MData = d;
    m_MDataWordLength = w;
    m_MDataLast = last_of_a_burst;
  }
  Td* SgetMData(int& w, bool& last_of_a_burst)
  {
    w = m_MDataWordLength;
    last_of_a_burst = m_MDataLast;

    return (m_MData);
  }
  Td* SgetMData(int& w)
  {
    w = m_MDataWordLength;
    return (m_MData);
  }
  Td* SgetMData()
  {
    return (m_MData);
  }

  // Access methods for MThreadID
  void MputMThreadID(int a)
  {
    m_MThreadID = a;
  }
  int SgetMThreadID()
  {
    return m_MThreadID;
  }


  // ---------- Response Signal Group ----------

  // Access methods for SData
  void SputSData(Td* d, int w = 1, bool last_of_a_burst = true)
  {
    m_SData = d;
    m_SDataWordLength = w;
    m_SDataLast = last_of_a_burst;
  }

  Td* MgetSData(int& w, bool& last_of_a_burst)
  {
    Td* data;
    w = m_SDataWordLength;
    last_of_a_burst = m_SDataLast;

    return (data);
  }
  Td* MgetSData(int& w)
  {
    w = m_SDataWordLength;

    return (m_SData);
  }
  Td* MgetSData()
  {
    return (m_SData);
  }


  // Access methods for SResp
  void SputSResp(OCPSRespType a)
  {
    m_SResp = a;
  }
  OCPSRespType MgetSResp()
  {
    OCPSRespType a = m_SResp;
    m_SResp = OCP_SRESP_NULL; // reset
    return a;
  }

  // Access methods for SThreadID
  void SputSThreadID(int a)
  {
    m_SThreadID = a;
  }
  int MgetSThreadID()
  {
    return m_SThreadID;
  }

  // Auxiliary methods for SAddr, not part of OCP 1.0
  void SputSAddr(Ta a)
  {
    m_SAddr = a;
  }
  Ta MgetSAddr()
  {
    return m_SAddr;
  }

  // Auxiliary methods to SCmd, not part of OCP 1.0
  bool IsWriteResponse()
  {
    return((m_SCmd == OCP_MCMD_WR) ? true : false);
  }
  void SputWriteResponse()
  {
    m_SCmd = OCP_MCMD_WR;
  }
  void SputReadResponse()
  {
    m_SCmd = OCP_MCMD_RD;
  }
  void SputSCmd(OCPMCmdType a)
  {
    m_SCmd = a;
  }
  OCPMCmdType MgetSCmd()
  {
    return m_SCmd;
  }

  // Auxiliary methods to SConnID, not part of OCP 1.0
  void SputSConnID(int a)
  {
    m_SConnID = a;
  }
  int MgetSConnID()
  {
    return m_SConnID;
  }

  
//---------------------------------------------------------------------
// The following members have not been throughly tested
// for this release, and may change in later releases.
// They are included for completeness.
//---------------------------------------------------------------------

  //---------------------------------------------------------------------
  // reset (Alpha version)
  //---------------------------------------------------------------------

  bool getReset_n()
  { return (m_Sidebands.Reset_n); }

  void putReset_n(bool reset_n)
  { 
    m_Sidebands.Reset_n = reset_n; 
    SBResetEvent.notify(SC_ZERO_TIME);
  }

  //---------------------------------------------------------------------
  // system/core signaling (Alpha version)
  //---------------------------------------------------------------------

  int  CgetControl()
  { return (m_Sidebands.Control); }

  void SysputControl(int control)
  { 
    m_Sidebands.Control = control; 
    SBControlEvent.notify(SC_ZERO_TIME);
  }

  int  CgetControlWr()
  { return (m_Sidebands.ControlWr); }

  void SysputControlWr(bool controlwr)
  { 
    m_Sidebands.ControlWr = controlwr; 
    SBControlEvent.notify(SC_ZERO_TIME);
  }


  int  SysgetControlBusy()
  { return (m_Sidebands.ControlBusy); }

  void CputControlBusy(bool controlbusy)
  { 
    m_Sidebands.ControlBusy = controlbusy; 
    SBControlEvent.notify(SC_ZERO_TIME);
  }

  int  SysgetStatus()
  { return (m_Sidebands.Status); }

  void CputStatus(int status)
  { 
    m_Sidebands.Status = status; 
    SBStatusEvent.notify(SC_ZERO_TIME);
  }

  int  SysgetStatusBusy()
  { return (m_Sidebands.StatusBusy); }

  void CputStatusBusy(int statusbusy)
  { 
    m_Sidebands.StatusBusy = statusbusy; 
    SBStatusEvent.notify(SC_ZERO_TIME);
  }

  int  CgetStatusRd()
  { return (m_Sidebands.StatusRd); }

  void SysputStatusRd(int statusrd)
  { m_Sidebands.StatusRd = statusrd; }

 
  //---------------------------------------------------------------------
  // threadbusy signaling, not usable for this release
  //---------------------------------------------------------------------

  bool MgetSThreadBusy()
  {
    return m_SThreadBusy;
  }

  void SputSThreadBusy(int sthreadbusy)
  {
    m_SThreadBusy = sthreadbusy;
  }

  bool SgetMThreadBusy()
  {
    return m_MThreadBusy;
  }

  void MputMThreadBusy(int mthreadbusy)
  {
    m_MThreadBusy = mthreadbusy;
  }

  //---------------------------------------------------------------------
  // sideband signaling (Alpha version)
  //---------------------------------------------------------------------

  bool MgetSError()
  { return (m_Sidebands.SError); }

  void SputSError(bool serror)
  { 
    m_Sidebands.SError = serror; 
    SBSlaveEvent.notify(SC_ZERO_TIME);
  }

  bool MgetSInterrupt()
  { return (m_Sidebands.SInterrupt); }

  void SputSInterrupt(bool sinterrupt)
  { 
    m_Sidebands.SInterrupt = sinterrupt; 
    SBSlaveEvent.notify(SC_ZERO_TIME);
  }

  int  MgetSFlag()
  { return (m_Sidebands.SFlag); }

  void SputSFlag(int sflag)
  { 
    m_Sidebands.SFlag = sflag; 
    SBSlaveEvent.notify(SC_ZERO_TIME);
  }

  bool SgetMError()
  { return (Sidebands.MError); }

  void MputMError(bool merror)
  { 
    m_Sidebands.MError = merror; 
    SBMasterEvent.notify(SC_ZERO_TIME);
  }

  int  SgetMFlag()
  { return (m_Sidebands.MFlag); }

  void MputMFlag(int mflag)
  { 
    m_Sidebands.MFlag = mflag; 
    SBMasterEvent.notify(SC_ZERO_TIME);
  }


  //---------------------------------------------------------------------
  // Not used in TL2
  //---------------------------------------------------------------------
  void update()
  {
  }

  sc_event SBControlEvent;
  sc_event SBStatusEvent;
  sc_event SBMasterEvent;
  sc_event SBSlaveEvent;

// -------------- End of access methods -------------------------------

private:

// --------------- Data members --------------- 

  // Request Signal Group
  Ta            m_MAddr;
  int           m_MAddrSpace;
  OCPMBurstType m_MBurst;
  int           m_MByteEn;
  OCPMCmdType   m_MCmd;
  int           m_MConnID;
  Td*           m_MData;
  int           m_MThreadID;

  // Response Signal Group
  Td*           m_SData;
  OCPSRespType  m_SResp;
  int           m_SThreadID;

  // helping signal value
  Ta            m_SAddr;
  OCPMCmdType   m_SCmd;
  int           m_SConnID;

  // ungrouped threadbusy signals
  bool          m_MThreadBusy;
  bool          m_SThreadBusy;

  // for sideband and threadbusy signals
  OCPSidebandSignals m_Sidebands;

  // -------------------------------------------------------------
  // for burst transactions (atomic and/or non-atomic)
  // -------------------------------------------------------------

  // for non-atomic burst transactions
  bool  m_MDataLast;
  bool  m_SDataLast;

  //
  int   m_MDataWordLength;
  int   m_SDataWordLength;

  // -------------------------------------------------------------
  // -------------------------------------------------------------

  // The ReqEndTime time stamps can be used to pass on information
  // about the durations of request packets to other modules.
  sc_time ReqEndTime;

  // The ResEndTime time stamps can be used to pass on information
  // about the durations of response packets to other modules.
  sc_time ResEndTime;
};

#endif // _OCP_TL2_DATA_CL
