// 
//  Copyright 2003 OCP-IP
//
// ============================================================================
//      Project : OCP SLD WG, Layer adapter examples 
//      Authors : Yann Bajot, PROSILOG, bajot@prosilog.com 
//                Stphane Guntz, PROSILOG, guntz@prosilog.com
//         Date : 06/2003
//
//  Description : Transaction Level - Layer-1 to Layer-2 Slave Adapter

//    
//  Parameter    :
//    Template arguments.
//     - TdataCl_tl1: TL1 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//	   - TdataCl_tl2: TL2 Data class containing data members and
//                access methods for the data exchanged between
//                Masters and Slaves
//     - Td: Data type 
//     - Ta: Address type
//                
//	Constructor arguments:
//      - sc_module_name: Name of the module instance
//	    - max_burst_length: Maximum size of burst length
//		- adapter_depth: Number of missed events and responses to be cached.
//						 The adapter stores new request/response infos which have arrived during the processing of the current request/response.
//						 It sends them when the current request/response is finished.
//		- writeResponse: boolean indicating if the slave connected to the adapter sends repsonses to write requests. In that case, the adapter has to send these responses to the master
//	 
//	 
//  The TL1 slave adapter has a TL2 slave interface on one side, and a TL1 master interface on the other side.
//  It must be connected to:
//      - a TL2 master through a TL2 channel
//      - a TL1 slave through a TL1 channel
//
//	The adapter retrieves TL2 requests and stores them in its internal buffer. It sends corresponding TL1 requests to the TL1 slave.
//	  
//  On the response side, the TL1 slave adapter retrieves the TL1 responses from the slave, packs them in burst and sends corresponding
//  TL2 responses to the master. 
// 
// ============================================================================

#include "ocp_tl1_tl2_slave_adapter.h"


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter constructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::OCP_TL1_TL2_Slave_Adapter(sc_module_name name_, int max_burst_length_, int adapter_depth_, bool writeResponse_): 
	sc_module(name_),
    max_burst_length(max_burst_length_),
	adapter_depth(adapter_depth_),
	writeResponse(writeResponse_)
{

	buffer_size=max_burst_length*adapter_depth;

	m_request_buffer= new request_parameters[buffer_size];
	m_response_buffer=new response_parameters[buffer_size];
	m_data=new Td_tl2[buffer_size];
	m_response_data= new Td_tl1[buffer_size];
	
	//arrays for storing characteristics of TL2 responses
	burst_response_index=new int[adapter_depth];
	burst_response_length=new int[adapter_depth];

	//arrays for storing characteristics of TL1 requests
	request_index=new int[adapter_depth];
	request_length=new int[adapter_depth];

	//initialize internal request, data and response buffer
	for(int i=0; i<buffer_size; i++)  {
		reset_response_parameters(i);
		m_data[i]=0;
		m_response_data[i]=0;
	}

	beginning_burst_index= current_index=0;

	for(int i=0; i<adapter_depth; i++)   {
		burst_response_index[i]=0;
		burst_response_length[i]=0;
		request_index[i]=0;
		request_length[i]=0;
	}
	last_datum_sent=false;
	release_request_thread=false;
	release_response_thread=false;
	m_pending_request=0;
	m_request_number=0;
	m_pending_response=0;
	m_response_number=0;
	current_processed_request=current_processed_response=0;
	count_response=0;
	nb_response=0;

	m_NumBytesPerWord = sizeof(Td_tl1);

	//SystemC processes 
	SC_THREAD(SlaveResponse);
	sensitive<<MasterP;

	SC_THREAD(MasterResponse);
	sensitive<<send_TL2_response_event;

	SC_THREAD(SlaveRequest);
	sensitive<<SlaveP;

	SC_THREAD(MasterRequest);
	sensitive<<send_TL1_request_event;

}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Master_Adapter destructor
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::~OCP_TL1_TL2_Slave_Adapter()    {
	
	delete [] m_response_data;
	delete [] m_request_buffer;
	delete [] m_response_buffer;
	delete [] m_data;
	delete [] burst_response_index;
	delete [] burst_response_length;
	delete [] request_index;
	delete [] request_length;
}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::SlaveResponse 
//get the TL1 responses, store them in the internal buffer, prepare TL2 response to be sent to the TL2 master
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::SlaveResponse()   {

	while(true)   {

		wait();			//wait for default event of the channel
		
		if(MasterP->MgetResponseBlocking(false))   {  //get a pending response

			//store the request fields in the internal buffer
			m_response_buffer[current_index].m_SResp=		m_DataCl_tl1->MgetSResp();
			m_response_buffer[current_index].m_SThreadID=	m_DataCl_tl1->MgetSThreadID();
		
			//store the data in the internal buffer
			m_response_data[current_index]=m_DataCl_tl1->MgetSData();
			
			#ifdef DEBUG_SLAVE_ADAPTER_TL1   
				cout<<"---\nSlaveResponse(): slave adapter gets a TL1 response at time "<<sc_time_stamp()<<endl;
				cout<<"SThreadID: "<<m_response_buffer[current_index].m_SThreadID<<endl;
				cout<<"SResp: "<<m_response_buffer[current_index].m_SResp<<endl;
				cout<<"SData: "<<m_response_data[current_index]<<"\n---"<<endl;
			#endif

			switch(define_adapter_task())   {

			case SEND_BURST_WITH_DATA:
				//specify burst length and beginning
				burst_response_index[m_response_number]=beginning_burst_index;
				burst_response_length[m_response_number]=abs(current_index-beginning_burst_index+1); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index+1;

				//the following transaction is the beginning of a new burst
				new_burst_transaction=true;

				//increment number of pending responses that the adapter has to send
				m_pending_response++;
				//increment number of the next response that is going to be stored
				//check if the response buffer is full: next response is currently being processed
				if((m_response_number+1)%adapter_depth==current_processed_response)
					release_response_thread=true;
				else {
					release_response_thread=false;
					m_response_number=(m_response_number+1)%adapter_depth;
					MasterP->Mrelease();
				}
				
				//notify response event
				send_TL2_response_event.notify();	
				break;

			case SEND_BURST_WITHOUT_DATA:
				//specify burst length and beginning
				burst_response_index[m_response_number]=beginning_burst_index;
				burst_response_length[m_response_number]=abs(current_index-beginning_burst_index); //use the abs function because of circular buffer

				//change the internal index
				beginning_burst_index=current_index;

				//the following transaction is the beginning of a new burst
				new_burst_transaction=true;

				//last datum of the burst is not part of the burst
				//last_datum_sent=false;

				//increment number of pending responses that the adapter has to send
				m_pending_response++;
				//increment number of the next response that is going to be stored
				//check if the response buffer is full: next response is not currently being processed
				if((m_response_number+1)%adapter_depth==current_processed_response)
					release_response_thread=true;
				else {
					release_response_thread=false;
					m_response_number=(m_response_number+1)%adapter_depth;
					MasterP->Mrelease();
				}
				
				//notify response event
				send_TL2_response_event.notify();
				break;

			case STORE_NEW_DATA:
				new_burst_transaction=false;
				MasterP->Mrelease();
				release_response_thread=false;
				break;

			default:
				break;
			}		

			//change the internal index
			if(current_index!=buffer_size-1)
				current_index++;
			else current_index=0;

		}  //get a pending response
	} //end of infinite loop
}  //end of SlaveResponse


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::MasterResponse
//send TL2 response made of atomic TL1 response
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::MasterResponse()   {
	
	while(true)   {
		
		if(m_pending_response==0)
			wait();  //wait for notification of send_TL2_response_event

		//put data in the TL2 data class
		m_DataCl_tl2->SputSData( &m_response_data[burst_response_index[current_processed_response]], burst_response_length[current_processed_response], true);  //last datum sent
		//put request fields into the TL2 data class
		put_TL2_response_fields(m_response_buffer[burst_response_index[current_processed_response]]);

		#ifdef DEBUG_SLAVE_ADAPTER_TL1
			cout<<"\n**MasterResponse: send a TL2 response at time t="<<sc_time_stamp()<<endl;
			cout<<"SResp= "<<m_response_buffer[burst_response_index[current_processed_response]].m_SResp<<endl;
			cout<<"SThreadID= "<<m_response_buffer[burst_response_index[current_processed_response]].m_SThreadID<<endl;
			cout<<"Length= "<<burst_response_length[current_processed_response]<<endl;
		#endif
		SlaveP->SputResponseBlocking();
		current_processed_response=(current_processed_response+1)%adapter_depth;

		//release the response thread, if it was not released before
		if(release_response_thread)  {
			m_response_number=(m_response_number+1)%adapter_depth;
			SlaveP->Srelease();
			release_response_thread=false;
		}
				
		if(m_pending_response!=0)
			m_pending_response--;

	} //end of infinite loop

} //end of MasterResponse method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::SlaveRequest
//get the TL2 request from the master
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::SlaveRequest()   {

	Td_tl2* temp;

	while(true)    {
		wait();
		//get TL2 request from the master 
		if(SlaveP->SgetRequestBlocking(false))   {
			//transfer request fields from TL2 data class to the adapter
			m_request_buffer[m_request_number].m_MAddr=m_DataCl_tl2->SgetMAddr();
			m_request_buffer[m_request_number].m_MAddrSpace=m_DataCl_tl2->SgetMAddrSpace();
			m_request_buffer[m_request_number].m_MBurst=m_DataCl_tl2->SgetMBurst();
			m_request_buffer[m_request_number].m_MByteEn=m_DataCl_tl2->SgetMByteEn();
			m_request_buffer[m_request_number].m_MCmd=m_DataCl_tl2->SgetMCmd();
			m_request_buffer[m_request_number].m_MConnID=m_DataCl_tl2->SgetMConnID();
			m_request_buffer[m_request_number].m_MThreadID=m_DataCl_tl2->SgetMThreadID();		

			//request
			#ifdef DEBUG_SLAVE_ADAPTER_TL1
					cout<<"\n***Master has sent a TL2 request at time t="<<sc_time_stamp()<<endl;
					cout<<"MAddr= "<<m_request_buffer[m_request_number].m_MAddr<<endl;
					cout<<"MCmd= "<<m_request_buffer[m_request_number].m_MCmd<<endl;
					cout<<"MThreadID= "<<m_request_buffer[m_request_number].m_MThreadID<<"\n***"<<endl;
			#endif
			
			//copy the request data and address from the TL2 data class to the adapter
			if(m_request_number==0)  
				request_index[m_request_number]=0;
			else request_index[m_request_number]=request_index[m_request_number-1]+request_length[m_request_number-1];
				
			temp = m_DataCl_tl2->SgetMData(request_length[m_request_number]);

			//checks if the size of the request has not greater than the maximum burst length
			if(request_length[m_request_number]>max_burst_length)   {
				cout<<"\n-------------\nError: size of one burst ("<<request_length[m_request_number]<<") exceeds the maximum burst length\n-------------"<<endl;
				for(int i=0; i<buffer_size-request_index[m_request_number]; i++)   
					m_data[i+request_index[m_request_number]] =temp[i];
			}
			else {
				for(int i=0; i<request_length[m_request_number]; i++)   
					m_data[i+request_index[m_request_number]] =temp[i];
			}

			send_TL1_request_event.notify();

			//increment number of pending requests that the adapter has to send
			m_pending_request++;

			//increment next request number
			if((m_request_number+1)%adapter_depth==current_processed_request)
				release_request_thread=true;
			else {
				release_request_thread=false;
				m_request_number=(m_request_number+1)%adapter_depth;
				//release request channel
				SlaveP->Srelease();
			}
			
		}  //end of get Request
	} //end of infinite loop
} //end of SlaveRequest method


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::MasterRequest
//send the group of TL1 requests to the slave, made from the TL2 request from the master 
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::MasterRequest()   {
	while(true)    {
		if(m_pending_request==0)
			wait();  //wait for notification of send_TL1_request_event

		wait(clk->posedge_event());

		//send atomic TL1 requests
		for(int i=request_index[current_processed_request]; i<request_index[current_processed_request]+request_length[current_processed_request]; i++)  {
			while(MasterP->MgetSbusy())  
				wait(clk->posedge_event());
			
			m_DataCl_tl1->MputMData(m_data[i]);
			m_DataCl_tl1->MputMAddr(m_request_buffer[current_processed_request].m_MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord);

			//check if the request is a write or a read request
			switch(m_request_buffer[current_processed_request].m_MCmd)   {

				case OCP_MCMD_WR:
					#ifdef DEBUG_SLAVE_ADAPTER_TL1
						cout<<"\n**MasterRequest: send a TL1 write request at time t="<<sc_time_stamp()<<endl;
						cout<<"Data= "<<m_data[i]<<endl;
						cout<<"Addr= "<<m_request_buffer[current_processed_request].m_MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord<<endl;
					#endif

					//checks if the slave sends response to write requests, to determine the number of responses that the slave will send
					if(writeResponse)
						nb_response=request_length[current_processed_request];
					else nb_response=0;

					MasterP->MputWriteRequest();
					break;

				case OCP_MCMD_RD:
					#ifdef DEBUG_SLAVE_ADAPTER_TL1
						cout<<"\n**MasterRequest: send a TL1 read request at time t="<<sc_time_stamp()<<endl;
						cout<<"Addr= "<<m_request_buffer[current_processed_request].m_MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord<<endl;
					#endif

					//the adapter is expecting the same number of response as the number of sent requests
					nb_response=request_length[current_processed_request];
					
					MasterP->MputReadRequest();
					break;

				case OCP_MCMD_RDEX:
					#ifdef DEBUG_SLAVE_ADAPTER_TL1
						cout<<"\n**MasterRequest: send an exclusive TL1 read request at time t="<<sc_time_stamp()<<endl;
						cout<<"Data= "<<m_data[i]<<endl;
						cout<<"Addr= "<<m_request_buffer[current_processed_request].m_MAddr+(i-request_index[current_processed_request])*m_NumBytesPerWord<<endl;
					#endif
					
					//the adapter is expecting the same number of response as the number of sent requests
					nb_response=request_length[current_processed_request];

					MasterP->MputReadRequest();
					break;

				default:
					cout<<"++++\ndefault\n++++"<<endl;
					break;
			} //end of switch

			wait(clk->posedge_event());  //wait for next clock edge to send next TL1 request;
		}

		current_processed_request=(current_processed_request+1)%adapter_depth;

		//release the request thread, if it was not released before
		if(release_request_thread)  {
			m_request_number=(m_request_number+1)%adapter_depth;
			SlaveP->Srelease();
			release_request_thread=false;
		}

		if(m_pending_request!=0)
			m_pending_request--;

	} //end of infinite loop
} //end of MasterRequest method



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::define_adapter_task
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
AdapterTask OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::define_adapter_task()   {
	
	count_response++;  //increments number of received TL1 responses

	//compare with response parameters in the internal buffer
	if( !new_burst_transaction)   {

		bool same_fields=same_response_fields(m_response_buffer[current_index], m_response_buffer[beginning_burst_index]);

		//the response fields are the same		
		if( same_fields)  {
			if(count_response==nb_response)
				return SEND_BURST_WITH_DATA;
			else {
				if(abs(current_index-beginning_burst_index+1)==max_burst_length)
					return SEND_BURST_WITH_DATA;
				else return STORE_NEW_DATA;
			}
		}
		//the fields have changed: send the current burst, keep the new data
		else return SEND_BURST_WITHOUT_DATA;
	}
	//new burst transaction
	else  return STORE_NEW_DATA;
	
}



//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::put_TL2_response_fields
//
//---------------------------------------------------------------------------
template< class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::put_TL2_response_fields(response_parameters& res_param)   {
	m_DataCl_tl2->SputSResp(res_param.m_SResp);
	m_DataCl_tl2->SputSThreadID(res_param.m_SThreadID);

} 
	


//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::same_response_fields
//
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::same_response_fields(response_parameters& new_res_param, response_parameters& current_res_param )  {

	//check if the fields have changed
	if(new_res_param.m_SThreadID != current_res_param.m_SThreadID)
		return false;
	else if(new_res_param.m_SResp != current_res_param.m_SResp)
		return false;

	//fields have not changed:
	return true;

}

//---------------------------------------------------------------------------
//Method: OCP_TL1_TL2_Slave_Adapter::reset_response_parameters
//reset request parameters
//---------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::reset_response_parameters(int i)   {
	m_response_buffer[i].m_SResp=		OCP_SRESP_NULL;
	m_response_buffer[i].m_SThreadID=	0;
}



// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Slave_Adapter::end_of_elaboration()
//
//  This method is activated at the end of the elaboration phase
//  (when all binding operations are finished).
// ----------------------------------------------------------------------------

template<class TdataCl_tl1, class TdataCl_tl2>
void OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::end_of_elaboration()
{

  sc_module::end_of_elaboration();

  // Initialize debug interface
  MasterP->MregisterDirectIF((MdirectIF<TdataCl_tl1>*)(this));
  SlaveP->SregisterDirectIF((SdirectIF<TdataCl_tl2>*)(this));


  // Get data structure
  m_DataCl_tl2 = SlaveP->GetDataCl();
  m_DataCl_tl1 = MasterP->GetDataCl();


  // Get communication structure
  m_CommCl_tl2 = SlaveP->GetCommCl();
  m_CommCl_tl1 = MasterP->GetCommCl();


  // Get system parameter structure
  m_ParamCl_tl2 = SlaveP->GetParamCl();
  m_ParamCl_tl1 = MasterP->GetParamCl();


  // Put parameter into Channel
  // No parameters needed

}

// ----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Slave_Adapter::MPutDirect()
//  Debug interface method.
//  Read/Write data, Master to Slave direction
//  Returns true for success and false for failure
//
//	calls Slave's MputDirect function
//
// ----------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::MputDirect(
    int MasterID, bool IsWrite, Td_tl1 *Data, Ta_tl1 Address, int NumWords)

{
  return(MasterP->MputDirect(MasterID, IsWrite, Data, Address, NumWords));

}


//----------------------------------------------------------------------------
//  Method : OCP_TL1_TL2_Slave_Adapter::SPutDirect()
//  Debug interface method.
//  Read/Write data, Slave to Master direction
//
//	calls Master's SputDirect function
//
// ----------------------------------------------------------------------------
template<class TdataCl_tl1, class TdataCl_tl2>
bool OCP_TL1_TL2_Slave_Adapter<TdataCl_tl1, TdataCl_tl2>::SputDirect(
         int SlaveID, bool IsWrite, Td_tl2 *Data, Ta_tl2 Address, int NumWords)
{
	return(SlaveP->SputDirect(SlaveID, IsWrite, Data, Address, NumWords));	
}


// ----------------------------------------------------------------------------
//
//  Instantiation of the adapter
//
// ----------------------------------------------------------------------------

template class OCP_TL1_TL2_Slave_Adapter<TL1_TEMPL_DATA_CL, OCP_TL2_TEMPL_DATA_CL >; // see ocp_tl1_globals.h and ocp_tl2_globals.h

