//
// Copyright (c) 2004 Sonics, Inc.
//
// Confidential and Proprietary Information of Sonics, Inc.
// Use, disclosure, or reproduction is prohibited without
// written permission from Sonics, Inc.
//
// $Id: StlParser.h,v 1.1 2007/01/25 22:09:07 halexan Exp $

#ifndef _OcpIp_StlParser_h_
#define _OcpIp_StlParser_h_

#include <iostream>
#include <string>
#include "IntTypes.h"
#include "StlCommand.h"
#include "boost/spirit/core.hpp"
#include "boost/spirit/utility/confix.hpp"
#include "boost/random/linear_congruential.hpp"
#include "OcpBundleDefs.h"
#include "Params.h"

using namespace boost::spirit;

class OCPParameters;
namespace OcpIp {
class StlParser : public boost::spirit::grammar<StlParser>
{
  public:
    friend class IStlReader;
    typedef StlParserTransferCommand<Sonics::BigInt<256>, snx_uint64_t>
        MyTransferCommand;
    typedef ParamSet<int, ParamName> OcpParams;
        
    StlParser( const OcpParams& params,
               bool coreSystem,
               ostream& err=cerr );
    virtual ~StlParser() {}
    void setDefaultThread( uint32_t thread ) {
        m_transferCmd.m_defaultThreadId = thread;
    }

    // must be public as it is used by the parse method
    template <typename ScannerT>
    struct definition
    {
        definition( StlParser const& self );
        rule<ScannerT> statement_p;
        rule<ScannerT> command_p;

        rule<ScannerT> waitCommand_p;
        rule<ScannerT> signalCommand_p;
        rule<ScannerT> resetCommand_p;
        rule<ScannerT> controlStatusCommand_p;

        rule<ScannerT> transferCommand_p;
        rule<ScannerT> readCommand_p;
        rule<ScannerT> writeCommand_p;
        rule<ScannerT> breadCommand_p;
        rule<ScannerT> bwriteCommand_p;
        rule<ScannerT> idleCommand_p;

        rule<ScannerT> rdcmd_p;
        rule<ScannerT> wrcmd_p;
        rule<ScannerT> brdcmd_p;
        rule<ScannerT> bwrcmd_p;

        rule<ScannerT> dataSpec_p;
        rule<ScannerT> expectData_p;
        rule<ScannerT> dataInfoSpec_p;
        rule<ScannerT> expectDataInfo_p;
        rule<ScannerT> as_addr_p;
        rule<ScannerT> tid_p;
        rule<ScannerT> cid_p;
        rule<ScannerT> tagid_p;
        rule<ScannerT> burstseq_p;
        rule<ScannerT> burst_p;
        rule<ScannerT> hdnum_p;

        rule<ScannerT> const& start() const {
            return statement_p;
        }
    };

    // user interface
    void readLine( const string& stlInput );
    template<typename Td, typename Ta>
    StlTransferCommand<Td,Ta> getTransferCommand()      const;
    StlWaitCommand            getWaitCommand()          const;
    StlSignalCommand          getSignalCommand()        const;
    StlResetCommand           getResetCommand()         const;
    StlControlStatusCommand   getControlStatusCommand() const;

    int  ocpParam( ParamName index ) const {
        return m_ocpParams.getParam( index );
    }
    static void setRandomSeed( uint32_t );
    static uint32_t getRandom();
    static const MyTransferCommand::DataType s_fullOneData;

  protected:
    const OcpParams      m_ocpParams;
    ostream* m_pErr;

    bool         setVersion( const char* first, const char* last );
    virtual void reportError( const string& stlLine, parse_info<> info );
    virtual void reportError( const string& stlLine, const string& errors );
      
    // the commands we are parsing to
    mutable StlCommandType                   m_commandType;
    mutable uint32_t                         m_commandDelay;
    mutable MyTransferCommand                m_transferCmd;
    mutable StlParserWaitCommand             m_waitCmd;
    mutable StlParserSignalCommand           m_signalCmd;
    mutable StlParserResetCommand            m_resetCmd;
    mutable StlParserControlStatusCommand    m_controlStatusCmd;

    // Version checking
    mutable bool                             m_versionChecked;
    bool                                     m_versionWarned;

    static boost::rand48                     s_randGenerator;

    // The width converter thing is a little weird. But we would like to
    // keep this function as the only template instead of the StlParser, and we
    // need to own the Transfer Command we return because data storage is thorny
    // Fortunately the Td,Ta pair is known at construction based on the OCP
    // data width.
    struct CommandConverter {
        static CommandConverter*
        create( const StlParser::OcpParams& params,
                const StlParser::MyTransferCommand& ref );
        virtual ~CommandConverter() {}
        virtual void convert() = 0;
      protected:
        CommandConverter( const StlParser::MyTransferCommand& ref ) :
            m_reference( ref ) {}
        const StlParser::MyTransferCommand& m_reference;
    };
    template<typename Td, typename Ta>
    struct CommandConverterImpl : public CommandConverter {
        CommandConverterImpl( const StlParser::OcpParams& params,
                              const StlParser::MyTransferCommand& ref );
        virtual ~CommandConverterImpl() {};
        virtual void convert();
        StlParserTransferCommand<Td,Ta> m_command;
    };

    CommandConverter* m_pCommandConverter;
};

// utility function to create a ParamSet from a OCP channel config structure
OcpParams convertInterfaceToParamSet( const OCPParameters& );
}

#endif /* _OcpIp_StlParser_h_ */
