// Reads digital inputs and converts them to EEnet.

`define Z `wrealZState
`define X `wrealXState

module i2c_bridge_core import EE_pkg::*; (
  inout  sda_drive,
  input	 scl_drive,
  inout  EEnet SDA,
  output EEnet SCL
  );

  import uvm_pkg::*;
  import uvm_ms_pkg::*;
  `include "uvm_ms.svh"
  timeunit 1ns;
  timeprecision 1ps;

  parameter real Vsup_p = 1.8; 
  parameter real Rout = 100;
  parameter real r_SUPPLY = 0;           // (str=7) supply resistance
  parameter real r_STRONG = Rout;         // (str=6) strong resistance
  parameter real r_PULL   = 4700;        // (str=5) pullup resistance
  parameter real r_LARGE  = 9e3;         // (str=4) large cap resistance
  parameter real r_WEAK   = 55e3;        // (str=3) weak resistance
  parameter real r_MEDIUM = 320e3;       // (str=2) medium cap resistance
  parameter real r_SMALL  = 1.9e6;       // (str=1) small cap resistance
  real Vsup = Vsup_p; // configurable by proxy method
  real Vlow = 0.35*Vsup;
  real Vhigh = 0.65*Vsup;
  real Rstr[7:0];       // resistance lookup based on logic strength
  bit[2:0] Slo[7:0] = '{6,5,3,3,0,0,0,0};  // next-lower strength of 5 levels
  bit[2:0] Shi[7:0] = '{7,7,6,5,5,3,3,3};  // next-higher strengths of 5 levels


  always @(Vsup) begin
    `uvm_ms_info("VSUP",$sformatf("Vsup set to %2.2f",Vsup),UVM_MEDIUM)
    Vlow = 0.35*Vsup;
    Vhigh = 0.65*Vsup;
  end
  
  real out_v_sda;
  real out_r_sda;
  
  real out_v_scl;
  real out_r_scl;
  
  // Target is driving
  reg sda_read;

  real r_ext, v_ext;
  assign (pull1, strong0) sda_drive = (v_ext > Vhigh) ? 1'b1 : (v_ext < Vlow ? 1'b0 : 1'bz);
  
  always @(SDA.R, out_r_sda, out_v_sda, SDA.V) begin 
    if(out_r_sda === `wrealZState) begin
      r_ext = SDA.R;
      v_ext = SDA.V;
    end
    else begin
      r_ext = (SDA.R*out_r_sda)/(out_r_sda - SDA.R);
      v_ext = (1/out_r_sda) * (SDA.V * (out_r_sda + r_ext) - out_v_sda*r_ext);
    end
  end

  reg sie;
  wire [7:0] sda_l_ext;
  bit [2:0] str0, str1, str_ext;
  bit [1:0] Dlev;

  initial begin
    sie = $cged(sda_drive, sda_l_ext);
    Rstr[7] = r_SUPPLY;           // save resistor strength values into an array
    Rstr[6] = r_STRONG;
    Rstr[5] = r_PULL;
    Rstr[4] = r_LARGE;
    Rstr[3] = r_WEAK;
    Rstr[2] = r_MEDIUM;
    Rstr[1] = r_SMALL;
    Rstr[0] = `wrealZState;
  end

  // Convert incoming logic value and strength to voltage and resistance:
  always begin
    {str0,str1,Dlev} = sda_l_ext;    // break up into 3 fields
    case (Dlev)
      2'b00:  begin              // "0"
                out_v_sda = 0.0;
                str_ext = str0;
                out_r_sda = Rstr[str_ext];
              end
      2'b01:  begin              // "1"
                out_v_sda = Vsup;
                str_ext = str1;
                out_r_sda = Rstr[str_ext]; 
              end
      2'b10:  begin              // "Z"
                str_ext = 0;        // strength must be 0 for Z signal
                out_r_sda = Rstr[0];
                if (out_v_sda===`wrealXState) out_v_sda=0.0; // only update V if previously X
                else out_v_sda = `wrealZState;
              end
      default: begin            // "X"
                out_v_sda = `wrealXState;              // logic X maps to EEnet X
                str_ext = (str0<str1) ? str0:str1;   // use lower strength
                out_r_sda = Rstr[str_ext];
              end
    endcase
    @(sda_l_ext);
  end

  // Drive
  assign SDA = '{out_v_sda, 0, out_r_sda};
  assign out_v_scl = scl_drive ? `Z : 0.0;
  assign out_r_scl = scl_drive ? `Z : Rout;
  assign SCL = '{out_v_scl, 0, out_r_scl};

endmodule
