// SystemVerilog analog 4 to 1 mux with parameterizable rising time and falling time

`timescale 1ns/1fs
`ifdef XRUN
import cds_rnm_pkg::*;
`endif
`ifdef VCS
import snps_msv_nettype_pkg::*;
`endif
module mux4to1(
	out,
	in,
	sel,
	en
);
	output wreal4state out;
	input wreal4state in [3:0];
	input logic [1:0] sel;      //Select bits - 00 = in[0], 01 = in[1], 10 = in[2], 11 = in[3]
	input logic en;

    
  real outval;        //Real variable for mux output voltage
  real out_int;

	parameter real tr = 1ps; // rising time = 1ps
	parameter real num_step = 10;  // samping time = 0.1ps
	parameter real tf = tr; 
	real vinc; 
	int nstep;
	real vout = 1; //[unit: V]
	parameter real vout_max = 1;
	real ts;
	
	always begin
		case(sel)
		    2'b00:  outval <= (en == 1) ? (in[0]) : 0; // div4
		    2'b01:  outval <= (en == 1) ? (in[1]) : 0; // div2
		    2'b10:  outval <= (en == 1) ? (in[2]) : 0; // fx1
		    2'b11:  outval <= (en == 1) ? (in[3]) : 0; // fx2
		endcase
		@(in, sel);
	end
		
  always  @(posedge(outval > 0)) begin // repeat when input changes
  	//Evaluate ts here. Based on tr and num_steps
  	ts = tr / num_step;
		if(out_int < vout_max) begin
			if ($realtime>0) begin  // if not DC op point,
				if(tr > 0) begin
					if(ts > 1fs) begin 	// Continue ramping if ts is greater than its minimum condition 1fs (timeprecision)
						if (num_step>=1) begin             // if interim steps needed,
							vinc = (1/num_step)*(vout_max);  // compute V change per ts 
							out_int += 1e-9; // wiggle voltage
							repeat (num_step-1) #(ts) out_int += vinc;
							#(ts) out_int = vout_max;
						end
					end
					else out_int = vout_max;  // if ts is below the minimum condition, directly step to final voltage
				end
				
				else begin
					out_int = vout_max; // output square wave when rise time is zero;
				end
			end
		end
	end	
	
	// generate falling ramping edge, similarly to rising ramping edge
  always  @(posedge(outval < vout_max)) begin // repeat when input changes
  	ts = tf / num_step;
  	if(out_int > 2e-9) begin // skip generating the falling edge if the output is already at the minimun value -- 0V
			if ($realtime>0) begin    // if not DC op point,
				if(ts > 1fs) begin
					if (num_step>=1) begin        
						vinc = (1/num_step)*(-1*(vout_max - outval));  // compute V change per ts 
						out_int += 1e-9;
						repeat (num_step-1) #(ts) out_int += vinc; // step only n-1 times, final step put out_int to 0
						#(ts) out_int = 0;
					end
				end
				else out_int = 0;
				
			end
		end
	end
		
		
	assign out = out_int;
	

endmodule
